﻿# backend/app/api/generic_opportunities.py
"""
Endpoints para análisis de oportunidades de medicamentos genéricos por farmacia
"""

import logging
from typing import Any, Dict, List, Optional

from fastapi import APIRouter, Depends, HTTPException, Path, Query, status
from sqlalchemy.orm import Session

from app.database import get_db
from app.models.homogeneous_group_master import HomogeneousGroupMaster
from app.models.pharmacy import Pharmacy
from app.models.pharmacy_homogeneous_metrics import PharmacyHomogeneousMetrics
from app.services.generic_opportunities_service import generic_opportunities_service
from app.utils.datetime_utils import utc_now

logger = logging.getLogger(__name__)

router = APIRouter()


@router.get(
    "/pharmacies/{pharmacy_id}/generic-opportunities",
    response_model=Dict[str, Any],
    summary="Análisis de oportunidades de genéricos por farmacia",
    description="Calcula y devuelve oportunidades de descuento en medicamentos genéricos",
)
async def get_pharmacy_generic_opportunities(
    pharmacy_id: str = Path(..., description="ID de la farmacia"),
    db: Session = Depends(get_db),
    recalculate: bool = Query(False, description="Recalcular métricas antes de devolver resultados"),
    analysis_months: Optional[int] = Query(
        None,
        ge=1,
        le=24,
        description="Meses de análisis (por defecto usa configuración de farmacia)",
    ),
    min_opportunity_units: int = Query(0, ge=0, description="Unidades mínimas para considerar oportunidad"),
    min_opportunity_amount: float = Query(0.0, ge=0, description="Importe mínimo para considerar oportunidad"),
    page: int = Query(1, ge=1, description="Página (empezando desde 1)"),
    per_page: int = Query(20, ge=1, le=100, description="Oportunidades por página"),
):
    """
    Obtiene análisis completo de oportunidades de medicamentos genéricos para una farmacia

    Este endpoint:
    - Calcula oportunidades basadas en datos de ventas reales
    - Considera configuración de partners específica de la farmacia
    - Identifica conjuntos homogéneos con potencial de ahorro
    - Proporciona métricas para optimización de márgenes
    """
    try:
        # Verificar que la farmacia existe
        pharmacy = db.query(Pharmacy).filter(Pharmacy.id == pharmacy_id).first()
        if not pharmacy:
            raise HTTPException(
                status_code=status.HTTP_404_NOT_FOUND,
                detail=f"Farmacia {pharmacy_id} no encontrada",
            )

        # Recalcular métricas si se solicita
        if recalculate:
            logger.info(f"[API] Recalculando oportunidades para farmacia {pharmacy_id}")
            opportunities_stats = generic_opportunities_service.calculate_opportunities_for_pharmacy(
                db, pharmacy_id, analysis_months
            )
        else:
            # Usar métricas existentes
            logger.info(f"[API] Obteniendo oportunidades existentes para farmacia {pharmacy_id}")
            opportunities_stats = {"pharmacy_id": pharmacy_id, "recalculated": False}

        # Obtener métricas de la base de datos
        query = db.query(PharmacyHomogeneousMetrics).filter(PharmacyHomogeneousMetrics.pharmacy_id == pharmacy_id)

        # Aplicar filtros
        if min_opportunity_units > 0:
            query = query.filter(PharmacyHomogeneousMetrics.non_partner_units_sold >= min_opportunity_units)

        if min_opportunity_amount > 0:
            query = query.filter(PharmacyHomogeneousMetrics.potential_savings_base >= min_opportunity_amount)

        # Ordenar por potencial de ahorro (unidades × base)
        query = query.order_by(
            (
                PharmacyHomogeneousMetrics.non_partner_units_sold * PharmacyHomogeneousMetrics.potential_savings_base
            ).desc()
        )

        # Contar total
        total_items = query.count()

        # Aplicar paginación
        offset = (page - 1) * per_page
        metrics = query.offset(offset).limit(per_page).all()

        # Convertir a formato de respuesta enriquecido
        opportunities = []
        for metric in metrics:
            # Obtener datos del catálogo maestro
            master_group = (
                db.query(HomogeneousGroupMaster)
                .filter(HomogeneousGroupMaster.homogeneous_code == metric.homogeneous_code)
                .first()
            )

            opportunity = {
                # Identificación
                "homogeneous_code": metric.homogeneous_code,
                "homogeneous_name": (master_group.homogeneous_name if master_group else None),
                # Métricas de oportunidad
                "opportunity_units": metric.non_partner_units_sold,
                "opportunity_base_amount": float(metric.potential_savings_base or 0),
                "partner_penetration_percentage": metric.partner_penetration_percentage,
                # Métricas generales
                "total_units_sold": metric.total_units_sold,
                "total_revenue": float(metric.total_revenue or 0),
                # Datos del catálogo maestro
                "calculated_pvl": (float(master_group.calculated_pvl or 0) if master_group else None),
                "available_laboratories": (master_group.available_laboratories if master_group else []),
                "total_labs_in_group": (master_group.total_labs_in_group if master_group else 0),
                "main_active_ingredient": (master_group.main_active_ingredient if master_group else None),
                # Información específica de farmacia
                "partner_laboratories": metric.partner_laboratories or [],
                "has_partner_sales": metric.has_partner_sales,
                # Cálculos de ahorro potencial
                "savings_with_10_percent": _calculate_savings(metric, 10.0),
                "savings_with_15_percent": _calculate_savings(metric, 15.0),
                "savings_with_20_percent": _calculate_savings(metric, 20.0),
                # Metadatos
                "last_updated": (metric.updated_at.isoformat() if metric.updated_at else None),
                "data_period": {
                    "start": (metric.data_period_start.isoformat() if metric.data_period_start else None),
                    "end": (metric.data_period_end.isoformat() if metric.data_period_end else None),
                },
            }
            opportunities.append(opportunity)

        # Calcular estadísticas de resumen
        summary_stats = _calculate_summary_stats(db, pharmacy_id)

        # Metadatos de paginación
        total_pages = (total_items + per_page - 1) // per_page
        has_next = page < total_pages
        has_previous = page > 1

        return {
            "pharmacy": {
                "id": pharmacy_id,
                "name": pharmacy.name,
                "partner_laboratories": pharmacy.partner_laboratories or [],
            },
            "opportunities": opportunities,
            "summary": summary_stats,
            "pagination": {
                "page": page,
                "per_page": per_page,
                "total_items": total_items,
                "total_pages": total_pages,
                "has_next": has_next,
                "has_previous": has_previous,
            },
            "filters_applied": {
                "min_opportunity_units": min_opportunity_units,
                "min_opportunity_amount": min_opportunity_amount,
            },
            "calculation_info": {
                "recalculated": recalculate,
                "analysis_months": analysis_months or int(pharmacy.analysis_period_months or 13),
                "timestamp": utc_now().isoformat(),
            },
        }

    except HTTPException:
        raise
    except Exception as e:
        logger.error(f"Error obteniendo oportunidades para farmacia {pharmacy_id}: {str(e)}")
        raise HTTPException(
            status_code=status.HTTP_500_INTERNAL_SERVER_ERROR,
            detail=f"Error interno del servidor: {str(e)}",
        )


@router.post(
    "/pharmacies/{pharmacy_id}/generic-opportunities/recalculate",
    response_model=Dict[str, Any],
    summary="Recalcular oportunidades de genéricos",
    description="Fuerza el recálculo de todas las métricas de oportunidades para la farmacia",
)
async def recalculate_pharmacy_opportunities(
    pharmacy_id: str = Path(..., description="ID de la farmacia"),
    db: Session = Depends(get_db),
    analysis_months: Optional[int] = Query(None, ge=1, le=24, description="Meses de análisis a usar"),
):
    """
    Recalcula todas las métricas de oportunidades para una farmacia específica

    Este proceso:
    - Analiza todos los datos de ventas en el período especificado
    - Actualiza métricas en PharmacyHomogeneousMetrics
    - Recalcula potenciales de ahorro basados en partners configurados
    """
    try:
        # Verificar que la farmacia existe
        pharmacy = db.query(Pharmacy).filter(Pharmacy.id == pharmacy_id).first()
        if not pharmacy:
            raise HTTPException(
                status_code=status.HTTP_404_NOT_FOUND,
                detail=f"Farmacia {pharmacy_id} no encontrada",
            )

        logger.info(f"[API] Iniciando recálculo de oportunidades para farmacia {pharmacy_id}")

        # Ejecutar recálculo
        result = generic_opportunities_service.calculate_opportunities_for_pharmacy(db, pharmacy_id, analysis_months)

        return {
            "message": "Oportunidades recalculadas exitosamente",
            "pharmacy_id": pharmacy_id,
            "statistics": result,
            "timestamp": utc_now().isoformat(),
        }

    except HTTPException:
        raise
    except Exception as e:
        logger.error(f"Error recalculando oportunidades para farmacia {pharmacy_id}: {str(e)}")
        raise HTTPException(
            status_code=status.HTTP_500_INTERNAL_SERVER_ERROR,
            detail=f"Error durante el recálculo: {str(e)}",
        )


@router.get(
    "/pharmacies/{pharmacy_id}/partners",
    response_model=Dict[str, Any],
    summary="Obtener laboratorios partners de la farmacia",
    description="Devuelve la configuración actual de laboratorios partners",
)
async def get_pharmacy_partners(
    pharmacy_id: str = Path(..., description="ID de la farmacia"),
    db: Session = Depends(get_db),
):
    """
    Obtiene la configuración actual de laboratorios partners de la farmacia
    """
    try:
        pharmacy = db.query(Pharmacy).filter(Pharmacy.id == pharmacy_id).first()
        if not pharmacy:
            raise HTTPException(
                status_code=status.HTTP_404_NOT_FOUND,
                detail=f"Farmacia {pharmacy_id} no encontrada",
            )

        return {
            "pharmacy_id": pharmacy_id,
            "pharmacy_name": pharmacy.name,
            "partner_laboratories": pharmacy.partner_laboratories or [],
            "updated_at": (pharmacy.updated_at.isoformat() if pharmacy.updated_at else None),
        }

    except HTTPException:
        raise
    except Exception as e:
        logger.error(f"Error obteniendo partners para farmacia {pharmacy_id}: {str(e)}")
        raise HTTPException(
            status_code=status.HTTP_500_INTERNAL_SERVER_ERROR,
            detail=f"Error interno del servidor: {str(e)}",
        )


@router.put(
    "/pharmacies/{pharmacy_id}/partners",
    response_model=Dict[str, Any],
    summary="Actualizar laboratorios partners de la farmacia",
    description="Actualiza la configuración de laboratorios partners",
)
async def update_pharmacy_partners(
    pharmacy_id: str,
    partners_data: Dict[str, List[str]],
    db: Session = Depends(get_db),
    recalculate_opportunities: bool = Query(True, description="Recalcular oportunidades tras actualizar partners"),
):
    """
    Actualiza la lista de laboratorios partners de la farmacia

    Body esperado:
    {
        "partner_laboratories": ["NORMON S.A.", "CINFA", "KERN PHARMA", ...]
    }
    """
    try:
        pharmacy = db.query(Pharmacy).filter(Pharmacy.id == pharmacy_id).first()
        if not pharmacy:
            raise HTTPException(
                status_code=status.HTTP_404_NOT_FOUND,
                detail=f"Farmacia {pharmacy_id} no encontrada",
            )

        # Validar datos de entrada
        if "partner_laboratories" not in partners_data:
            raise HTTPException(
                status_code=status.HTTP_400_BAD_REQUEST,
                detail="Campo 'partner_laboratories' requerido",
            )

        partner_labs = partners_data["partner_laboratories"]
        if not isinstance(partner_labs, list):
            raise HTTPException(
                status_code=status.HTTP_400_BAD_REQUEST,
                detail="'partner_laboratories' debe ser una lista",
            )

        # Limpiar y validar laboratorios
        clean_partners = []
        for lab in partner_labs:
            if isinstance(lab, str) and lab.strip():
                clean_partners.append(lab.strip())

        # Actualizar farmacia
        pharmacy.partner_laboratories = clean_partners
        pharmacy.updated_at = utc_now()

        db.commit()

        result = {
            "message": "Partners actualizados exitosamente",
            "pharmacy_id": pharmacy_id,
            "partner_laboratories": clean_partners,
            "updated_at": pharmacy.updated_at.isoformat(),
        }

        # Recalcular oportunidades si se solicita
        if recalculate_opportunities:
            logger.info(f"[API] Recalculando oportunidades tras actualizar partners para farmacia {pharmacy_id}")
            opportunities_stats = generic_opportunities_service.calculate_opportunities_for_pharmacy(db, pharmacy_id)
            result["opportunities_recalculated"] = True
            result["recalculation_stats"] = opportunities_stats

        return result

    except HTTPException:
        raise
    except Exception as e:
        logger.error(f"Error actualizando partners para farmacia {pharmacy_id}: {str(e)}")
        db.rollback()
        raise HTTPException(
            status_code=status.HTTP_500_INTERNAL_SERVER_ERROR,
            detail=f"Error interno del servidor: {str(e)}",
        )


# === MÉTODOS AUXILIARES ===


def _calculate_savings(metric: PharmacyHomogeneousMetrics, discount_percentage: float) -> Dict[str, Any]:
    """Calcula ahorro potencial con un porcentaje de descuento específico"""
    if not metric.potential_savings_base or discount_percentage <= 0:
        return {
            "discount_percentage": discount_percentage,
            "savings_amount": 0.0,
            "units_affected": metric.non_partner_units_sold or 0,
        }

    base_amount = float(metric.potential_savings_base)
    savings = base_amount * (discount_percentage / 100)

    return {
        "discount_percentage": discount_percentage,
        "savings_amount": round(savings, 2),
        "units_affected": metric.non_partner_units_sold or 0,
        "savings_per_unit": round(savings / max(metric.non_partner_units_sold or 1, 1), 2),
    }


def _calculate_summary_stats(db: Session, pharmacy_id: str) -> Dict[str, Any]:
    """Calcula estadísticas de resumen para la farmacia"""
    from sqlalchemy import func

    # Estadísticas generales
    total_metrics = (
        db.query(func.count(PharmacyHomogeneousMetrics.id))
        .filter(PharmacyHomogeneousMetrics.pharmacy_id == pharmacy_id)
        .scalar()
        or 0
    )

    opportunities_count = (
        db.query(func.count(PharmacyHomogeneousMetrics.id))
        .filter(
            PharmacyHomogeneousMetrics.pharmacy_id == pharmacy_id,
            PharmacyHomogeneousMetrics.non_partner_units_sold > 0,
        )
        .scalar()
        or 0
    )

    # Sumas agregadas
    aggregates = (
        db.query(
            func.sum(PharmacyHomogeneousMetrics.total_units_sold),
            func.sum(PharmacyHomogeneousMetrics.non_partner_units_sold),
            func.sum(PharmacyHomogeneousMetrics.potential_savings_base),
            func.avg(
                PharmacyHomogeneousMetrics.non_partner_units_sold / PharmacyHomogeneousMetrics.total_units_sold * 100
            ),
        )
        .filter(
            PharmacyHomogeneousMetrics.pharmacy_id == pharmacy_id,
            PharmacyHomogeneousMetrics.total_units_sold > 0,
        )
        .first()
    )

    total_units = aggregates[0] or 0
    total_opportunity_units = aggregates[1] or 0
    total_savings_base = float(aggregates[2] or 0)
    avg_opportunity_percentage = aggregates[3] or 0

    return {
        "total_homogeneous_groups": total_metrics,
        "groups_with_opportunities": opportunities_count,
        "total_units_analyzed": total_units,
        "total_opportunity_units": total_opportunity_units,
        "total_savings_base_amount": total_savings_base,
        "avg_opportunity_percentage": round(avg_opportunity_percentage, 2),
        "potential_savings_scenarios": {
            "with_10_percent": round(total_savings_base * 0.10, 2),
            "with_15_percent": round(total_savings_base * 0.15, 2),
            "with_20_percent": round(total_savings_base * 0.20, 2),
        },
    }
