﻿# backend/app/api/generic_analysis.py
"""
API endpoints modernos para análisis de genéricos usando sistema de medidas Power BI-style.
Reemplaza funcionalidad de generic_study.py con arquitectura de medidas Power BI-style.
"""

import logging
from datetime import date
from typing import Any, Dict, List, Optional
from uuid import UUID

from fastapi import APIRouter, Depends, HTTPException, Query
from sqlalchemy.orm import Session

from app.utils.datetime_utils import utc_now

from ..api.deps import get_current_user, verify_pharmacy_access
from ..database import get_db
from ..measures import measure_registry
from ..measures.base import FilterContext, QueryContext
from ..models.user import User

logger = logging.getLogger(__name__)

router = APIRouter(prefix="/generic-analysis", tags=["generic-analysis"])


@router.get("/dashboard/{pharmacy_id}")
async def get_generic_dashboard(
    pharmacy_id: UUID,
    db: Session = Depends(get_db),
    current_user: User = Depends(get_current_user),
    start_date: Optional[date] = Query(None),
    end_date: Optional[date] = Query(None),
) -> Dict[str, Any]:
    """
    Dashboard completo de análisis de genéricos usando medidas avanzadas.
    Combina múltiples medidas para análisis integral.
    """
    try:
        # SECURITY: Validar acceso a farmacia (REGLA #7 - Issue #479)
        verify_pharmacy_access(pharmacy_id, current_user)

        # Construir filtros base
        filters = FilterContext(pharmacy_id=str(pharmacy_id), start_date=start_date, end_date=end_date)

        context = QueryContext(db, filters)

        # Medidas principales para dashboard
        dashboard_measures = [
            "total_ventas",
            "total_unidades",
            "generic_vs_branded_ratio",
            "generic_savings_opportunity",
            "partner_lab_sales",
            "therapeutic_category_sales",
        ]

        results = {}
        metadata = {}

        for measure_name in dashboard_measures:
            try:
                results[measure_name] = measure_registry.calculate_measure(measure_name, context, use_cache=True)

                measure = measure_registry.get_measure(measure_name)
                if measure:
                    metadata[measure_name] = measure.get_metadata()

            except Exception as e:
                logger.warning(f"Error calculating {measure_name}: {e}")
                results[measure_name] = None
                metadata[measure_name] = {"error": str(e)}

        # Organizar resultados por secciones del dashboard
        return {
            "dashboard_type": "analisis_genericos_completo",
            "kpis_generales": {
                "total_ventas": results.get("total_ventas"),
                "total_unidades": results.get("total_unidades"),
            },
            "analisis_genericos": {
                "ratio_genericos_marca": results.get("generic_vs_branded_ratio"),
                "oportunidades_ahorro": results.get("generic_savings_opportunity"),
            },
            "partners": {"ventas_partners": results.get("partner_lab_sales")},
            "categorias_terapeuticas": {"distribucion_ventas": results.get("therapeutic_category_sales")},
            "periodo_analizado": {
                "inicio": start_date.isoformat() if start_date else None,
                "fin": end_date.isoformat() if end_date else None,
            },
            "metadata": metadata,
            "generado_en": date.today().isoformat(),
        }

    except Exception as e:
        logger.error(f"Error generating generic dashboard for {pharmacy_id}: {e}")
        raise HTTPException(status_code=500, detail=f"Error generating dashboard: {str(e)}")


@router.get("/savings-opportunities/{pharmacy_id}")
async def get_savings_opportunities_analysis(
    pharmacy_id: UUID,
    db: Session = Depends(get_db),
    current_user: User = Depends(get_current_user),
    start_date: Optional[date] = Query(None),
    end_date: Optional[date] = Query(None),
    min_savings: float = Query(0.0, description="Mínimo ahorro potencial en €"),
) -> Dict[str, Any]:
    """
    Análisis detallado de oportunidades de ahorro usando medida especializada.
    Con cache automático y reutilizable.
    """
    try:
        # SECURITY: Validar acceso a farmacia (REGLA #7 - Issue #479)
        verify_pharmacy_access(pharmacy_id, current_user)

        filters = FilterContext(
            pharmacy_id=str(pharmacy_id),
            start_date=start_date,
            end_date=end_date,
            min_amount=min_savings,
        )

        context = QueryContext(db, filters)

        # Usar medida especializada
        opportunities = measure_registry.calculate_measure("generic_savings_opportunity", context, use_cache=True)

        # Complementar con ratio genéricos/marca para contexto
        ratio_analysis = measure_registry.calculate_measure("generic_vs_branded_ratio", context, use_cache=True)

        return {
            "oportunidades_ahorro": opportunities,
            "contexto_genericos": ratio_analysis,
            "parametros": {
                "min_savings": min_savings,
                "start_date": start_date.isoformat() if start_date else None,
                "end_date": end_date.isoformat() if end_date else None,
            },
            "generado_en": utc_now().isoformat(),
        }

    except Exception as e:
        logger.error(f"Error in savings opportunities analysis: {e}")
        raise HTTPException(status_code=500, detail=f"Error analyzing savings: {str(e)}")


@router.get("/homogeneous-groups/{pharmacy_id}")
async def get_homogeneous_groups_analysis(
    pharmacy_id: UUID,
    db: Session = Depends(get_db),
    current_user: User = Depends(get_current_user),
    start_date: Optional[date] = Query(None),
    end_date: Optional[date] = Query(None),
    discount_percentage: float = Query(25.0, ge=10.0, le=50.0, description="% descuento a simular"),
    top_groups: int = Query(20, description="Top N grupos a analizar"),
) -> Dict[str, Any]:
    """
    Análisis de conjuntos homogéneos usando medida avanzada.
    Reemplaza funcionalidad de matriz de oportunidades homogéneas.
    """
    try:
        # SECURITY: Validar acceso a farmacia (REGLA #7 - Issue #479)
        verify_pharmacy_access(pharmacy_id, current_user)

        filters = FilterContext(
            pharmacy_id=str(pharmacy_id),
            start_date=start_date,
            end_date=end_date,
            discount_percentage=discount_percentage,
            top_groups=top_groups,
        )

        context = QueryContext(db, filters)

        # Usar medida especializada para conjuntos homogéneos
        homogeneous_analysis = measure_registry.calculate_measure("homogeneous_group_sales", context, use_cache=True)

        # Complementar con análisis de partners si están disponibles
        partner_analysis = measure_registry.calculate_measure("partner_lab_sales", context, use_cache=True)

        return {
            "conjuntos_homogeneos": homogeneous_analysis,
            "partners_context": partner_analysis,
            "simulacion": {
                "descuento_aplicado": discount_percentage,
                "grupos_analizados": top_groups,
            },
            "periodo": {
                "inicio": start_date.isoformat() if start_date else None,
                "fin": end_date.isoformat() if end_date else None,
            },
            "generado_en": utc_now().isoformat(),
        }

    except Exception as e:
        logger.error(f"Error in homogeneous groups analysis: {e}")
        raise HTTPException(status_code=500, detail=f"Error analyzing groups: {str(e)}")


@router.get("/partners-trend/{pharmacy_id}")
async def get_partners_trend_analysis(
    pharmacy_id: UUID,
    db: Session = Depends(get_db),
    current_user: User = Depends(get_current_user),
    months_back: int = Query(12, description="Meses hacia atrás para análisis"),
) -> Dict[str, Any]:
    """
    Análisis de tendencia de partners usando medida temporal especializada.
    Para análisis de evolución mensual.
    """
    try:
        # SECURITY: Validar acceso a farmacia (REGLA #7 - Issue #479)
        verify_pharmacy_access(pharmacy_id, current_user)

        filters = FilterContext(pharmacy_id=str(pharmacy_id), months_back=months_back)

        context = QueryContext(db, filters)

        # Usar medida temporal especializada
        trend_analysis = measure_registry.calculate_measure("monthly_partner_trend", context, use_cache=True)

        # Complementar con snapshot actual de partners
        current_partners = measure_registry.calculate_measure("partner_lab_sales", context, use_cache=True)

        return {
            "tendencia_temporal": trend_analysis,
            "situacion_actual": current_partners,
            "parametros": {"meses_analizados": months_back},
            "generado_en": utc_now().isoformat(),
        }

    except Exception as e:
        logger.error(f"Error in partners trend analysis: {e}")
        raise HTTPException(status_code=500, detail=f"Error analyzing trend: {str(e)}")


@router.get("/therapeutic-distribution/{pharmacy_id}")
async def get_therapeutic_distribution_analysis(
    pharmacy_id: UUID,
    db: Session = Depends(get_db),
    current_user: User = Depends(get_current_user),
    start_date: Optional[date] = Query(None),
    end_date: Optional[date] = Query(None),
) -> Dict[str, Any]:
    """
    Distribución por categorías terapéuticas usando medida especializada.
    Análisis farmacológico avanzado.
    """
    try:
        # SECURITY: Validar acceso a farmacia (REGLA #7 - Issue #479)
        verify_pharmacy_access(pharmacy_id, current_user)

        filters = FilterContext(pharmacy_id=str(pharmacy_id), start_date=start_date, end_date=end_date)

        context = QueryContext(db, filters)

        # Medida especializada para categorías terapéuticas
        therapeutic_analysis = measure_registry.calculate_measure("therapeutic_category_sales", context, use_cache=True)

        # Complementar con ratio genéricos para contexto completo
        generic_ratio = measure_registry.calculate_measure("generic_vs_branded_ratio", context, use_cache=True)

        return {
            "distribucion_terapeutica": therapeutic_analysis,
            "contexto_genericos": generic_ratio,
            "periodo": {
                "inicio": start_date.isoformat() if start_date else None,
                "fin": end_date.isoformat() if end_date else None,
            },
            "generado_en": utc_now().isoformat(),
        }

    except Exception as e:
        logger.error(f"Error in therapeutic distribution analysis: {e}")
        raise HTTPException(status_code=500, detail=f"Error analyzing distribution: {str(e)}")


@router.post("/calculate-multiple-generic-measures")
async def calculate_multiple_generic_measures(
    measures_request: Dict[str, Any],
    db: Session = Depends(get_db),
    current_user: User = Depends(get_current_user),
) -> Dict[str, Any]:
    """
    Calcular múltiples medidas de genéricos en una sola petición.
    Para dashboards complejos.

    Body esperado:
    {
        "pharmacy_id": "uuid",
        "measure_names": ["generic_vs_branded_ratio", "partner_lab_sales", ...],
        "filters": {
            "start_date": "2024-01-01",
            "end_date": "2024-12-31",
            "discount_percentage": 25.0,
            "partner_laboratories": ["LAB1", "LAB2"]
        }
    }
    """
    try:
        pharmacy_id = measures_request.get("pharmacy_id")
        measure_names = measures_request.get("measure_names", [])
        filters_dict = measures_request.get("filters", {})

        if not pharmacy_id or not measure_names:
            raise HTTPException(status_code=400, detail="pharmacy_id and measure_names are required")

        # SECURITY: Validar acceso a farmacia (REGLA #7 - Issue #479)
        from uuid import UUID as UUIDType
        verify_pharmacy_access(UUIDType(pharmacy_id), current_user)

        # Construir contexto de filtros extendido
        filters = FilterContext(
            pharmacy_id=pharmacy_id,
            start_date=(date.fromisoformat(filters_dict["start_date"]) if filters_dict.get("start_date") else None),
            end_date=(date.fromisoformat(filters_dict["end_date"]) if filters_dict.get("end_date") else None),
            # Filtros específicos de genéricos
            partner_laboratories=filters_dict.get("partner_laboratories"),
            discount_percentage=filters_dict.get("discount_percentage"),
            top_groups=filters_dict.get("top_groups"),
            months_back=filters_dict.get("months_back"),
            homogeneous_group=filters_dict.get("homogeneous_group"),
            # Filtros generales
            product_codes=filters_dict.get("product_codes"),
            therapeutic_categories=filters_dict.get("therapeutic_categories"),
            laboratories=filters_dict.get("laboratories"),
            requires_prescription=filters_dict.get("requires_prescription"),
            is_generic=filters_dict.get("is_generic"),
            min_amount=filters_dict.get("min_amount"),
            max_amount=filters_dict.get("max_amount"),
        )

        context = QueryContext(db, filters)

        # Calcular todas las medidas solicitadas
        results = {}
        metadata = {}

        for measure_name in measure_names:
            try:
                results[measure_name] = measure_registry.calculate_measure(measure_name, context, use_cache=True)

                measure = measure_registry.get_measure(measure_name)
                if measure:
                    metadata[measure_name] = measure.get_metadata()

            except Exception as e:
                logger.warning(f"Error calculating {measure_name}: {e}")
                results[measure_name] = None
                metadata[measure_name] = {"error": str(e)}

        return {
            "results": results,
            "metadata": metadata,
            "filters_applied": filters.to_dict(),
            "calculation_timestamp": utc_now().isoformat(),
        }

    except HTTPException:
        raise
    except Exception as e:
        logger.error(f"Error in multiple generic measures calculation: {e}")
        raise HTTPException(status_code=500, detail=f"Error calculating measures: {str(e)}")


@router.get("/universe-summary/{pharmacy_id}")
async def get_universe_summary(
    pharmacy_id: UUID,
    db: Session = Depends(get_db),
    current_user: User = Depends(get_current_user),
    start_date: Optional[date] = Query(None),
    end_date: Optional[date] = Query(None),
) -> Dict[str, Any]:
    """
    Obtiene resumen del universo sustituible usando medida Power BI-style.

    Issue #478 Fase 2: Migración de Context Panel a patrón Measures.

    Retorna estructura compatible con context-store del frontend:
    - universe_summary: Métricas del universo sustituible
      - total_substitutable_groups: Conjuntos homogéneos con genéricos
      - total_substitutable_units: Unidades vendidas en esos conjuntos
      - total_substitutable_revenue: Ingresos de esos conjuntos
      - total_pharmacy_revenue: Total farmacia (para calcular %)
      - substitutable_percentage: % que representa el sustituible
    - homogeneous_groups: Lista vacía (backward compatibility)
    - laboratories_in_universe: Laboratorios genéricos para dropdown partners

    Error 503: Si nomenclator no está sincronizado (< 1000 productos con código homogéneo)
    """
    from sqlalchemy import func

    from ..core.constants import NOMENCLATOR_MIN_THRESHOLD
    from ..models.product_catalog import ProductCatalog
    from ..models.sales_data import SalesData

    try:
        # SECURITY: Validar acceso a farmacia (REGLA #7 - Issue #479)
        verify_pharmacy_access(pharmacy_id, current_user)

        # Validación nomenclator sincronizado (paridad con endpoint antiguo)
        sales_count = db.query(func.count(SalesData.id)).filter(
            SalesData.pharmacy_id == pharmacy_id
        ).scalar() or 0

        nomenclator_ready = (
            db.query(ProductCatalog.id)
            .filter(ProductCatalog.nomen_codigo_homogeneo.isnot(None))
            .limit(NOMENCLATOR_MIN_THRESHOLD)
            .count()
        )

        if sales_count > 0 and nomenclator_ready < NOMENCLATOR_MIN_THRESHOLD:
            logger.warning(
                f"[UNIVERSE_SUMMARY] Farmacia {pharmacy_id}: nomenclator sync required. "
                f"Sales: {sales_count}, products with homogeneous code: {nomenclator_ready}/{NOMENCLATOR_MIN_THRESHOLD}"
            )
            raise HTTPException(
                status_code=503,
                detail="El catálogo de productos necesita sincronización. Contacta al administrador del sistema.",
            )

        filters = FilterContext(
            pharmacy_id=str(pharmacy_id),
            start_date=start_date,
            end_date=end_date,
        )
        context = QueryContext(db, filters)

        # Issue #478: Usar medida SubstitutableUniverseSummary
        result = measure_registry.calculate_measure(
            "substitutable_universe_summary", context, use_cache=True
        )

        # Añadir pharmacy_id para validación de cache stale en frontend (Issue #438)
        result["_pharmacy_id"] = str(pharmacy_id)

        return result

    except HTTPException:
        raise
    except Exception as e:
        logger.error(f"Error getting universe summary for {pharmacy_id}: {e}")
        raise HTTPException(status_code=500, detail=f"Error getting universe summary: {str(e)}")


@router.get("/context-treemap/{pharmacy_id}")
async def get_context_treemap_measures(
    pharmacy_id: UUID,
    db: Session = Depends(get_db),
    current_user: User = Depends(get_current_user),
    start_date: Optional[date] = Query(None),
    end_date: Optional[date] = Query(None),
    employee_names: Optional[List[str]] = Query(None, description="Filtrar por empleados (PRO)"),
) -> Dict[str, Any]:
    """
    Treemap de distribución de ventas usando medidas Power BI-style.

    Issue #472: Migración a patrón Measures con GenericContextTreemap.

    Usa la medida GenericContextTreemap que internamente:
    - Calcula TotalVentas con diferentes combinaciones de filtros
    - Segmenta por: Prescripción/OTC → Sustituible/No → Analizable/No → Partners/Oportunidad
    - Retorna estructura jerárquica para Plotly Treemap

    Retorna estructura compatible con Plotly Treemap:
    - labels: Nombres de segmentos
    - parents: Jerarquía
    - values: Ventas totales
    - text: Texto descriptivo con porcentajes
    - colors: Colores semánticos
    """
    try:
        # SECURITY: Validar acceso a farmacia (REGLA #7)
        verify_pharmacy_access(pharmacy_id, current_user)

        filters = FilterContext(
            pharmacy_id=str(pharmacy_id),
            start_date=start_date,
            end_date=end_date,
            employee_names=employee_names,
        )
        context = QueryContext(db, filters)

        # Issue #472: Usar GenericContextTreemap - medida compuesta que usa TotalVentas + filtros
        treemap_result = measure_registry.calculate_measure(
            "generic_context_treemap", context, use_cache=True
        )

        # La medida ya retorna estructura compatible con Plotly
        return {
            "treemap_data": treemap_result.get("treemap_data", {}),
            "hierarchy": treemap_result.get("hierarchy", {}),
            "summary": treemap_result.get("summary", {}),
            "measure_used": "generic_context_treemap",
            "generado_en": utc_now().isoformat(),
        }

    except Exception as e:
        logger.error(f"Error generating treemap from measures: {e}")
        raise HTTPException(status_code=500, detail=f"Error generating treemap: {str(e)}")
