# backend/app/api/endpoints/feedback.py
"""
API endpoints para Human-in-the-Loop feedback del clasificador.

Endpoints:
- GET /review-queue: Cola priorizada de productos a revisar
- POST /approve: Aprobar clasificación actual
- POST /correct: Corregir clasificación
- POST /outlier: Marcar como outlier
- GET /stats: Estadísticas de correcciones
- GET /categories: Categorías disponibles para corrección

Issue #457: M4 Feedback Loop
"""

from typing import Optional
from uuid import UUID

from fastapi import APIRouter, Depends, HTTPException, Query
from sqlalchemy.orm import Session

from app.api.deps import get_current_user, get_db
from app.schemas.symptom_taxonomy import NecesidadEspecifica

# Build category dict from enum for validation
NECESIDAD_CATEGORIES = {e.value: e.value.replace("_", " ").title() for e in NecesidadEspecifica}
from app.models.user import User
from app.models.product_correction import OutlierReason, OutlierStatus
from app.schemas.feedback import (
    ApproveRequest,
    AvailableCategory,
    CorrectRequest,
    CorrectionResponse,
    CorrectionStats,
    OutlierRequest,
    OutlierResolveRequest,
    OutlierStatusUpdateRequest,
    OutliersResponse,
    ReviewQueueBreakdown,
    ReviewQueueResponse,
    SkipRequest,
)
from app.services.feedback_service import FeedbackService

router = APIRouter()


@router.get("/review-queue", response_model=ReviewQueueResponse)
def get_review_queue(
    pharmacy_id: Optional[UUID] = None,
    priority: Optional[str] = Query(
        None, regex="^(P1|P2|P3)$", description="Filtrar por prioridad: P1, P2, o P3"
    ),
    limit: int = Query(50, ge=1, le=100),
    offset: int = Query(0, ge=0),
    db: Session = Depends(get_db),
    current_user: User = Depends(get_current_user),
):
    """
    Obtiene cola priorizada de productos pendientes de revisión.

    Prioridades:
    - P1: Productos clasificados como "otros" (fallback)
    - P2: Productos con confianza < 0.6
    - P3: Resto de productos sin revisión

    Solo incluye productos de venta libre que no han sido revisados.
    """
    service = FeedbackService(db)

    # Si el usuario no es admin, filtrar por su pharmacy
    effective_pharmacy_id = pharmacy_id
    if current_user.role != "admin" and current_user.pharmacy_id:
        effective_pharmacy_id = current_user.pharmacy_id

    result = service.get_review_queue(
        pharmacy_id=effective_pharmacy_id,
        limit=limit,
        offset=offset,
        priority=priority,
    )

    return ReviewQueueResponse(
        items=result["items"],
        total=result["total"],
        breakdown=ReviewQueueBreakdown(**result["breakdown"]) if result["breakdown"] else ReviewQueueBreakdown(),
        offset=result["offset"],
        limit=result["limit"],
    )


@router.post("/approve", response_model=CorrectionResponse)
def approve_classification(
    request: ApproveRequest,
    db: Session = Depends(get_db),
    current_user: User = Depends(get_current_user),
):
    """
    Aprueba la clasificación actual del sistema.

    El farmacéutico confirma que el clasificador acertó.
    Esto ayuda a medir la accuracy del clasificador.
    """
    service = FeedbackService(db)

    try:
        correction = service.approve_classification(
            sales_enrichment_id=request.sales_enrichment_id,
            pharmacy_id=current_user.pharmacy_id,
            reviewer_notes=request.reviewer_notes,
        )
    except ValueError as e:
        raise HTTPException(status_code=404, detail=str(e))

    return CorrectionResponse(
        id=correction.id,
        sales_enrichment_id=correction.sales_enrichment_id,
        product_name=correction.product_name,
        predicted_category=correction.predicted_category,
        prediction_source=correction.prediction_source.value,
        confidence_score=correction.confidence_score,
        correction_type=correction.correction_type.value,
        corrected_category=correction.corrected_category,
        reviewer_notes=correction.reviewer_notes,
        created_at=correction.created_at,
    )


@router.post("/correct", response_model=CorrectionResponse)
def correct_classification(
    request: CorrectRequest,
    db: Session = Depends(get_db),
    current_user: User = Depends(get_current_user),
):
    """
    Corrige la clasificación del sistema.

    El farmacéutico indica la categoría correcta.
    Esta corrección:
    1. Actualiza el enrichment con la categoría corregida
    2. Registra la corrección para análisis y generación de reglas
    """
    # Validar que la categoría existe
    if request.corrected_category not in NECESIDAD_CATEGORIES:
        valid_cats = ", ".join(sorted(NECESIDAD_CATEGORIES.keys())[:10]) + "..."
        raise HTTPException(
            status_code=400,
            detail=f"Categoría '{request.corrected_category}' no válida. Usar: {valid_cats}",
        )

    service = FeedbackService(db)

    try:
        correction = service.correct_classification(
            sales_enrichment_id=request.sales_enrichment_id,
            corrected_category=request.corrected_category,
            pharmacy_id=current_user.pharmacy_id,
            reviewer_notes=request.reviewer_notes,
        )
    except ValueError as e:
        raise HTTPException(status_code=404, detail=str(e))

    return CorrectionResponse(
        id=correction.id,
        sales_enrichment_id=correction.sales_enrichment_id,
        product_name=correction.product_name,
        predicted_category=correction.predicted_category,
        prediction_source=correction.prediction_source.value,
        confidence_score=correction.confidence_score,
        correction_type=correction.correction_type.value,
        corrected_category=correction.corrected_category,
        reviewer_notes=correction.reviewer_notes,
        created_at=correction.created_at,
    )


@router.post("/outlier", response_model=CorrectionResponse)
def mark_as_outlier(
    request: OutlierRequest,
    db: Session = Depends(get_db),
    current_user: User = Depends(get_current_user),
):
    """
    Marca un producto como outlier.

    Issue #457: Ahora incluye outlier_reason y crea con status PENDING_REVIEW
    para posterior investigación.

    Para productos que no encajan en ninguna categoría estándar.
    Razones válidas:
    - pack_promocional: "Pack 3x2", "Neceser regalo"
    - descatalogado: Producto ya no existe
    - ambiguo: No se puede determinar categoría
    - nueva_categoria: Necesita categoría nueva
    - otro: Casos no cubiertos
    """
    service = FeedbackService(db)

    try:
        correction = service.mark_as_outlier(
            sales_enrichment_id=request.sales_enrichment_id,
            pharmacy_id=current_user.pharmacy_id,
            reviewer_notes=request.reviewer_notes,
            outlier_reason=request.outlier_reason,
        )
    except ValueError as e:
        raise HTTPException(status_code=404, detail=str(e))

    return CorrectionResponse(
        id=correction.id,
        sales_enrichment_id=correction.sales_enrichment_id,
        product_name=correction.product_name,
        predicted_category=correction.predicted_category,
        prediction_source=correction.prediction_source.value,
        confidence_score=correction.confidence_score,
        correction_type=correction.correction_type.value,
        corrected_category=correction.corrected_category,
        reviewer_notes=correction.reviewer_notes,
        created_at=correction.created_at,
    )


@router.post("/skip", response_model=CorrectionResponse)
def skip_product(
    request: SkipRequest,
    db: Session = Depends(get_db),
    current_user: User = Depends(get_current_user),
):
    """
    Salta un producto temporalmente, enviándolo al final de la cola.

    Issue #457: El producto no se pierde, simplemente se marca como
    "skip" y aparecerá al final de la cola de revisión.

    Si se salta un producto que ya estaba saltado, se actualiza el
    timestamp para enviarlo de nuevo al final de la cola.
    """
    service = FeedbackService(db)

    try:
        correction = service.skip_product(
            sales_enrichment_id=request.sales_enrichment_id,
            pharmacy_id=current_user.pharmacy_id,
            reviewer_notes=request.reviewer_notes,
        )
    except ValueError as e:
        raise HTTPException(status_code=404, detail=str(e))

    # Handle prediction_source whether it's enum or string
    prediction_source = correction.prediction_source
    if hasattr(prediction_source, 'value'):
        prediction_source = prediction_source.value

    # Handle correction_type whether it's enum or string
    correction_type = correction.correction_type
    if hasattr(correction_type, 'value'):
        correction_type = correction_type.value

    return CorrectionResponse(
        id=correction.id,
        sales_enrichment_id=correction.sales_enrichment_id,
        product_name=correction.product_name,
        predicted_category=correction.predicted_category,
        prediction_source=prediction_source,
        confidence_score=correction.confidence_score,
        correction_type=correction_type,
        corrected_category=correction.corrected_category,
        reviewer_notes=correction.reviewer_notes,
        created_at=correction.created_at,
    )


# ==========================================================================
# Issue #457: OUTLIER INVESTIGATION ENDPOINTS
# ==========================================================================

@router.get("/outliers", response_model=OutliersResponse)
def get_outliers(
    status: Optional[str] = Query(None, description="Filtrar por status: pending_review, investigating, etc."),
    reason: Optional[str] = Query(None, description="Filtrar por razón: pack_promocional, ambiguo, etc."),
    limit: int = Query(50, ge=1, le=100),
    offset: int = Query(0, ge=0),
    db: Session = Depends(get_db),
    current_user: User = Depends(get_current_user),
):
    """
    Obtiene lista de outliers para investigación.

    Issue #457: Los outliers ya no son "basura en la BD". Ahora tienen
    un flujo de investigación que permite detectar categorías faltantes.

    Por defecto muestra solo outliers pendientes (pending_review + investigating).
    """
    service = FeedbackService(db)

    # Convert string params to enums
    status_enum = None
    if status:
        try:
            status_enum = OutlierStatus(status)
        except ValueError:
            raise HTTPException(
                status_code=400,
                detail=f"Status '{status}' no válido. Usar: {[s.value for s in OutlierStatus]}"
            )

    reason_enum = None
    if reason:
        try:
            reason_enum = OutlierReason(reason)
        except ValueError:
            raise HTTPException(
                status_code=400,
                detail=f"Razón '{reason}' no válida. Usar: {[r.value for r in OutlierReason]}"
            )

    result = service.get_outliers_pending(
        status=status_enum,
        reason=reason_enum,
        limit=limit,
        offset=offset,
    )

    return OutliersResponse(**result)


@router.put("/outliers/{correction_id}/status")
def update_outlier_status(
    correction_id: int,
    request: OutlierStatusUpdateRequest,
    db: Session = Depends(get_db),
    current_user: User = Depends(get_current_user),
):
    """
    Actualiza el estado de un outlier.

    Útil para marcar como "investigating" sin cerrar.
    Solo admin puede cambiar estado.
    """
    if current_user.role != "admin":
        raise HTTPException(status_code=403, detail="Solo administradores pueden cambiar estado de outliers")

    service = FeedbackService(db)

    try:
        correction = service.update_outlier_status(
            correction_id=correction_id,
            new_status=request.new_status,
        )
    except ValueError as e:
        raise HTTPException(status_code=404, detail=str(e))

    return {"message": "Status actualizado", "new_status": correction.outlier_status.value}


@router.put("/outliers/{correction_id}/resolve")
def resolve_outlier(
    correction_id: int,
    request: OutlierResolveRequest,
    db: Session = Depends(get_db),
    current_user: User = Depends(get_current_user),
):
    """
    Resuelve un outlier cambiando su estado final.

    Issue #457: Flujo de resolución:
    - resolved_category: Se creó nueva categoría para este tipo de producto
    - resolved_corrected: Era clasificable, se corrigió el clasificador
    - true_outlier: Confirmado como no clasificable (pack, descatalogado)

    Solo admin puede resolver outliers.
    """
    if current_user.role != "admin":
        raise HTTPException(status_code=403, detail="Solo administradores pueden resolver outliers")

    service = FeedbackService(db)

    try:
        correction = service.resolve_outlier(
            correction_id=correction_id,
            new_status=request.new_status,
            resolution_notes=request.resolution_notes,
            new_category=request.new_category,
        )
    except ValueError as e:
        raise HTTPException(status_code=404, detail=str(e))

    return {
        "message": "Outlier resuelto",
        "new_status": correction.outlier_status.value,
        "resolved_at": correction.outlier_resolved_at.isoformat() if correction.outlier_resolved_at else None,
    }


@router.get("/outliers/reasons")
def get_outlier_reasons(
    current_user: User = Depends(get_current_user),
):
    """
    Lista las razones de outlier disponibles.

    Útil para poblar dropdowns en la UI.
    """
    return [
        {"value": r.value, "label": r.value.replace("_", " ").title()}
        for r in OutlierReason
    ]


@router.get("/outliers/statuses")
def get_outlier_statuses(
    current_user: User = Depends(get_current_user),
):
    """
    Lista los estados de outlier disponibles.

    Útil para poblar dropdowns en la UI.
    """
    return [
        {"value": s.value, "label": s.value.replace("_", " ").title()}
        for s in OutlierStatus
    ]


@router.get("/stats", response_model=CorrectionStats)
def get_correction_stats(
    pharmacy_id: Optional[UUID] = None,
    db: Session = Depends(get_db),
    current_user: User = Depends(get_current_user),
):
    """
    Obtiene estadísticas de correcciones.

    Incluye:
    - Total de correcciones
    - Desglose por tipo (approve, correct, outlier)
    - Accuracy del clasificador basada en correcciones
    """
    service = FeedbackService(db)

    # Si el usuario no es admin, filtrar por su pharmacy
    effective_pharmacy_id = pharmacy_id
    if current_user.role != "admin" and current_user.pharmacy_id:
        effective_pharmacy_id = current_user.pharmacy_id

    stats = service.get_corrections_stats(pharmacy_id=effective_pharmacy_id)

    return CorrectionStats(**stats)


@router.get("/categories", response_model=list[AvailableCategory])
def get_available_categories(
    current_user: User = Depends(get_current_user),
):
    """
    Obtiene lista de categorías disponibles para corrección.

    Útil para poblar dropdowns en la UI de validación.
    """
    categories = []
    for slug, name in sorted(NECESIDAD_CATEGORIES.items()):
        categories.append(
            AvailableCategory(
                slug=slug,
                name=name,
                description=None,  # Podríamos añadir descripciones en el futuro
            )
        )
    return categories
