"""
Manejadores de error estandarizados para las APIs.
Proporciona un patrón consistente de manejo de excepciones.
"""

from typing import Any, Dict

import structlog
from fastapi import HTTPException
from sqlalchemy.exc import IntegrityError, OperationalError

from app.exceptions import (
    AuthenticationError,
    ConfigurationError,
    DatabaseError,
    EnrichmentError,
    InvalidERPFormatError,
    PharmacyException,
    PharmacyNotFoundError,
    ProcessingError,
    ProductNotFoundError,
    RateLimitError,
)

logger = structlog.get_logger(__name__)


def handle_exception(e: Exception, context: Dict[str, Any] = None) -> HTTPException:
    """
    Convierte excepciones del dominio a HTTPException con logging estructurado.

    Args:
        e: La excepción a manejar
        context: Contexto adicional para el logging (endpoint, user_id, etc.)

    Returns:
        HTTPException con status code y mensaje apropiados

    Usage:
        try:
            # Lógica del endpoint
            pass
        except HTTPException:
            raise  # Re-raise HTTPExceptions preserving their status codes
        except Exception as e:
            raise handle_exception(e, {"endpoint": "upload", "pharmacy_id": pharmacy_id})
    """
    context = context or {}

    # Manejar excepciones custom del dominio
    if isinstance(e, ProductNotFoundError):
        logger.warning(
            "Producto no encontrado", extra={**context, "error": str(e), "product_code": e.details.get("product_code")}
        )
        raise HTTPException(status_code=404, detail=str(e))

    elif isinstance(e, PharmacyNotFoundError):
        logger.warning(
            "Farmacia no encontrada", extra={**context, "error": str(e), "pharmacy_id": e.details.get("pharmacy_id")}
        )
        raise HTTPException(status_code=404, detail=str(e))

    elif isinstance(e, InvalidERPFormatError):
        logger.error(
            "Formato ERP inválido",
            extra={
                **context,
                "error": str(e),
                "parser_type": e.details.get("parser_type"),
                "file_name": e.details.get("file_name"),
            },
        )
        raise HTTPException(status_code=400, detail=str(e))

    elif isinstance(e, EnrichmentError):
        logger.error(
            "Error de enriquecimiento",
            extra={
                **context,
                "error": str(e),
                "stage": e.details.get("stage"),
                "product_count": e.details.get("product_count"),
            },
        )
        raise HTTPException(status_code=422, detail=str(e))

    elif isinstance(e, RateLimitError):
        logger.warning(
            "Límite de tasa excedido",
            extra={
                **context,
                "error": str(e),
                "endpoint": e.details.get("endpoint"),
                "retry_after": e.details.get("retry_after"),
            },
        )
        raise HTTPException(
            status_code=429,
            detail=str(e),
            headers={"Retry-After": str(e.details.get("retry_after", 60))},
        )

    elif isinstance(e, AuthenticationError):
        logger.warning(
            "Error de autenticación", extra={**context, "error": str(e), "auth_type": e.details.get("auth_type")}
        )
        raise HTTPException(status_code=401, detail=str(e))

    elif isinstance(e, ConfigurationError):
        logger.error(
            "Error de configuración",
            extra={
                **context,
                "error": str(e),
                "component": e.details.get("component"),
                "missing_config": e.details.get("missing_config"),
            },
        )
        raise HTTPException(status_code=503, detail="Servicio temporalmente no disponible")

    elif isinstance(e, DatabaseError):
        logger.error(
            "Error de base de datos",
            extra={
                **context,
                "error": str(e),
                "operation": e.details.get("operation"),
                "table": e.details.get("table"),
            },
        )
        raise HTTPException(status_code=503, detail="Error de base de datos")

    elif isinstance(e, ProcessingError):
        logger.error(
            "Error de procesamiento",
            extra={
                **context,
                "error": str(e),
                "process": e.details.get("process"),
                "step": e.details.get("step"),
            },
        )
        raise HTTPException(status_code=422, detail=str(e))

    elif isinstance(e, PharmacyException):
        # Excepción genérica del dominio
        logger.error(
            "Error del dominio farmacéutico", extra={**context, "error": str(e), "code": e.code, "details": e.details}
        )
        raise HTTPException(status_code=400, detail=str(e))

    # Manejar excepciones de SQLAlchemy
    elif isinstance(e, IntegrityError):
        logger.error("Error de integridad de datos", extra={**context, "error": str(e), "error_type": "IntegrityError"})
        raise HTTPException(status_code=409, detail="Conflicto de datos: registro duplicado o restricción violada")

    elif isinstance(e, OperationalError):
        logger.error(
            "Error operacional de base de datos", extra={**context, "error": str(e), "error_type": "OperationalError"}
        )
        raise HTTPException(status_code=503, detail="Base de datos no disponible temporalmente")

    # HTTPException - re-raise tal cual
    elif isinstance(e, HTTPException):
        raise e

    # Excepción no manejada - error genérico 500
    else:
        logger.exception("Error inesperado en API", extra={**context, "error": str(e), "error_type": type(e).__name__})
        raise HTTPException(status_code=500, detail="Error interno del servidor")


def api_error_handler(func):
    """
    Decorador para manejo consistente de errores en endpoints.

    Usage:
        @router.get("/endpoint")
        @api_error_handler
        async def my_endpoint(db: Session = Depends(get_db)):
            # Código del endpoint
            pass
    """

    async def wrapper(*args, **kwargs):
        try:
            return await func(*args, **kwargs)
        except HTTPException:
            raise  # Re-raise HTTPExceptions preserving their status codes
        except Exception as e:
            # Extraer contexto del endpoint si es posible
            context = {
                "endpoint": func.__name__,
                "module": func.__module__,
            }
            raise handle_exception(e, context)

    return wrapper
