# backend/app/api/erp_sync.py
"""
API endpoints for ERP Sync management (Pivot 2026).

Provides endpoints to:
- Get sync status
- Trigger manual sync
- Reset error state

Local single-tenant: No pharmacy_id needed, always syncs the ONE pharmacy.
"""
import os
from typing import Any, Dict

import structlog
from fastapi import APIRouter, BackgroundTasks, Depends, HTTPException
from pydantic import BaseModel
from sqlalchemy.orm import Session

from app.database import get_db
from app.models.erp_sync_state import ERPSyncState
from app.services.erp_sync_service import get_erp_sync_service

# NOTE: Auth relaxed for local single-tenant context (Pivot 2026)
# Access is controlled by being on the local machine/network

logger = structlog.get_logger()
router = APIRouter(prefix="/erp-sync", tags=["erp-sync"])


class SyncStatusResponse(BaseModel):
    """Response model for sync status."""

    status: str
    sync_enabled: bool
    last_sync_completed_at: str | None
    last_sale_timestamp: str | None
    last_sync_records: int
    last_sync_duration_seconds: int
    total_syncs: int
    total_records_synced: int
    consecutive_errors: int
    last_error: str | None

    # Config info (from environment)
    erp_type: str
    erp_host: str
    erp_database: str
    sync_interval_minutes: int


class SyncResult(BaseModel):
    """Response model for sync operation."""

    success: bool
    records_synced: int
    records_skipped: int = 0
    duration_seconds: int = 0
    error: str | None = None
    skipped: bool = False
    reason: str | None = None


class ResetResult(BaseModel):
    """Response model for reset operation."""

    success: bool
    message: str


@router.get("/status", response_model=SyncStatusResponse)
def get_sync_status(db: Session = Depends(get_db)) -> Dict[str, Any]:
    """
    Get current ERP sync status.

    Returns sync state and configuration info.
    """
    # Get state from service
    service = get_erp_sync_service()
    state_dict = service.get_state()

    # Add config info from environment
    state_dict["erp_type"] = os.getenv("ERP_TYPE", "farmanager")
    state_dict["erp_host"] = os.getenv("ERP_HOST", "localhost")
    state_dict["erp_database"] = os.getenv("ERP_DATABASE", "gesql")
    state_dict["sync_interval_minutes"] = int(os.getenv("ERP_SYNC_INTERVAL", "15"))

    # Ensure all required fields exist
    defaults = {
        "last_sync_completed_at": None,
        "last_sale_timestamp": None,
        "last_sync_records": 0,
        "last_sync_duration_seconds": 0,
        "total_syncs": 0,
        "total_records_synced": 0,
        "consecutive_errors": 0,
        "last_error": None,
        "sync_enabled": True,
        "status": "not_initialized",
    }
    for key, default in defaults.items():
        if key not in state_dict:
            state_dict[key] = default

    return state_dict


@router.post("/run", response_model=SyncResult)
def run_sync(
    force_full: bool = False,
    background_tasks: BackgroundTasks = None,
    db: Session = Depends(get_db),
) -> Dict[str, Any]:
    """
    Trigger ERP sync manually.

    Args:
        force_full: If True, do full sync ignoring last_sale_timestamp

    Returns:
        Sync results including records synced/skipped and duration.

    Note: This runs synchronously. For large datasets, consider
    using the background scheduler instead.
    """
    logger.info("erp_sync.api.run_triggered", force_full=force_full)

    service = get_erp_sync_service()
    result = service.sync(force_full=force_full)

    logger.info("erp_sync.api.run_completed", **result)

    return result


@router.post("/run-async")
def run_sync_async(
    force_full: bool = False,
    background_tasks: BackgroundTasks = None,
) -> Dict[str, str]:
    """
    Trigger ERP sync in background.

    Returns immediately with acknowledgment.
    Check /status endpoint for progress.
    """
    if background_tasks is None:
        raise HTTPException(
            status_code=500,
            detail="Background tasks not available",
        )

    def _run_in_background():
        service = get_erp_sync_service()
        service.sync(force_full=force_full)

    background_tasks.add_task(_run_in_background)

    logger.info("erp_sync.api.async_triggered", force_full=force_full)

    return {
        "status": "started",
        "message": "Sync started in background. Check /status for progress.",
    }


@router.post("/reset-errors", response_model=ResetResult)
def reset_errors(db: Session = Depends(get_db)) -> Dict[str, Any]:
    """
    Reset error state after manual intervention.

    Use this when:
    - ERP connection has been fixed
    - Consecutive errors need to be cleared
    - Want to re-enable automatic sync after 5+ errors

    Returns:
        Success status and message.
    """
    logger.info("erp_sync.api.reset_triggered")

    service = get_erp_sync_service()
    result = service.reset_errors()

    logger.info("erp_sync.api.reset_completed", **result)

    return result


@router.post("/enable")
def enable_sync(db: Session = Depends(get_db)) -> Dict[str, str]:
    """Enable automatic sync."""
    state = db.query(ERPSyncState).first()
    if not state:
        state = ERPSyncState(id=1)
        db.add(state)

    state.sync_enabled = True
    db.commit()

    logger.info("erp_sync.api.enabled")
    return {"status": "ok", "message": "Sync enabled"}


@router.post("/disable")
def disable_sync(db: Session = Depends(get_db)) -> Dict[str, str]:
    """Disable automatic sync."""
    state = db.query(ERPSyncState).first()
    if not state:
        state = ERPSyncState(id=1)
        db.add(state)

    state.sync_enabled = False
    db.commit()

    logger.info("erp_sync.api.disabled")
    return {"status": "ok", "message": "Sync disabled"}


@router.get("/config")
def get_config() -> Dict[str, Any]:
    """
    Get ERP configuration (from environment).

    Note: Password is not included for security.
    """
    return {
        "erp_type": os.getenv("ERP_TYPE", "farmanager"),
        "host": os.getenv("ERP_HOST", "localhost"),
        "port": int(os.getenv("ERP_PORT", "3306")),
        "database": os.getenv("ERP_DATABASE", "gesql"),
        "charset": os.getenv("ERP_CHARSET", "utf8"),
        "sync_interval_minutes": int(os.getenv("ERP_SYNC_INTERVAL", "15")),
        "initial_sync_days": int(os.getenv("ERP_INITIAL_SYNC_DAYS", "90")),
        "batch_size": int(os.getenv("ERP_SYNC_BATCH_SIZE", "500")),
        # Password hidden for security
        "user": os.getenv("ERP_USER", ""),
        "password_configured": bool(os.getenv("ERP_PASSWORD")),
    }
