# backend/app/api/employees.py
"""
API endpoints para Employee filtering (Issue #402).

Feature: Employee filtering en página /generics (PRO plan)

Endpoints:
- GET /api/v1/employees/{pharmacy_id} - Lista empleados de la farmacia (PRO+ only)

Security:
- JWT authentication requerida (REGLA #7)
- Subscription plan validation (PRO+ only)
- User must own the pharmacy (REGLA #10: 1:1 relationship)
"""

import logging
from uuid import UUID

from fastapi import APIRouter, Depends, HTTPException, status
from sqlalchemy.orm import Session

from app.api.deps import get_current_user, require_subscription_plan, get_current_with_employee_access
from app.database import get_db
from app.models.user import User
from app.schemas.employee import EmployeeListResponse
from app.services.employee_service import EmployeeService

logger = logging.getLogger(__name__)

router = APIRouter(prefix="/employees", tags=["employees"])


@router.get("/{pharmacy_id}", response_model=EmployeeListResponse)
async def get_pharmacy_employees(
    pharmacy_id: UUID,
    db: Session = Depends(get_db),
    # Issue #402: Plan PRO+ required
    current_user: User = Depends(require_subscription_plan("pro")),
    # Issue #541: Titular access required (operativo no puede ver datos de empleados)
    _: User = Depends(get_current_with_employee_access),
):
    """
    Get unique employees for a pharmacy from sales data.

    **Feature**: Employee filtering (PRO+ plan only)

    **DECISIONES APROBADAS**:
    - DECISIÓN #1: Plan enforcement usando decorator @require_subscription_plan("pro")
    - DECISIÓN #2: Filtrado solo por employee_name (sin employee_code)
    - DECISIÓN UX #3: Incluye ventas sin empleado como "__sin_empleado__"

    **Authorization**:
    - JWT required (REGLA #7)
    - Plan PRO+ required (returns 403 for FREE plan) via decorator
    - User must own the pharmacy (returns 404 otherwise)

    **Returns**:
    - List of employees ordered by sales_count (descending)
    - Each employee includes: name, sales_count, last_sale_date
    - May include special value "__sin_empleado__" for sales without employee

    **Example Response**:
    ```json
    {
        "employees": [
            {
                "name": "María García",
                "sales_count": 1250,
                "last_sale_date": "2025-11-05"
            },
            {
                "name": "Juan Pérez",
                "sales_count": 890,
                "last_sale_date": "2025-11-04"
            },
            {
                "name": "__sin_empleado__",
                "sales_count": 320,
                "last_sale_date": "2025-11-03"
            }
        ],
        "total_count": 3
    }
    ```

    **Errors**:
    - 401: No JWT token provided
    - 403: User plan doesn't have access (detail includes upgrade_url)
    - 404: Pharmacy not found or user doesn't own pharmacy
    - 500: Database error

    **Issue**: #402
    """
    logger.info(
        f"[GET /employees/{pharmacy_id}] Request from user {current_user.email} "
        f"(plan: {current_user.subscription_plan})"
    )

    try:
        # Use EmployeeService to get employees
        # Service handles:
        # - Subscription plan validation (PRO+ only)
        # - Pharmacy ownership verification
        # - Query optimization with indexes
        service = EmployeeService(db)
        result = service.get_pharmacy_employees(
            pharmacy_id=pharmacy_id,
            user=current_user,
        )

        logger.info(
            f"[GET /employees/{pharmacy_id}] Success: {result.total_count} employees found"
        )

        return result

    except HTTPException:
        # Re-raise HTTP exceptions from service (403, 404, 500)
        raise

    except Exception as e:
        logger.error(
            f"[GET /employees/{pharmacy_id}] Unexpected error: {str(e)}",
            exc_info=True,
        )
        raise HTTPException(
            status_code=status.HTTP_500_INTERNAL_SERVER_ERROR,
            detail="Error fetching employees data. Please try again later.",
        )
