# backend/app/api/dropzone.py
"""
API endpoints for Dropzone Watcher management (Pivot 2026).

Provides endpoints to:
- Get dropzone status
- Start/stop watcher
- Get processing statistics

Local single-tenant: No auth required for local network access.
"""
import os
from typing import Any, Dict

import structlog
from fastapi import APIRouter, HTTPException
from pydantic import BaseModel

from app.services.dropzone_watcher import dropzone_watcher, WATCHDOG_AVAILABLE

# NOTE: Auth relaxed for local single-tenant context (Pivot 2026)
# Access is controlled by being on the local machine/network

logger = structlog.get_logger()
router = APIRouter(prefix="/dropzone", tags=["dropzone"])


class DropzoneStatusResponse(BaseModel):
    """Response model for dropzone status."""

    enabled: bool
    running: bool
    watchdog_available: bool
    paths: list[str]
    poll_interval_seconds: int
    debounce_seconds: float
    stats: Dict[str, Any]


class DropzoneActionResponse(BaseModel):
    """Response model for dropzone actions."""

    success: bool
    message: str
    running: bool


@router.get("/status", response_model=DropzoneStatusResponse)
def get_dropzone_status() -> Dict[str, Any]:
    """
    Get current dropzone watcher status.

    Returns status, configuration, and statistics.
    """
    return dropzone_watcher.get_status()


@router.post("/start", response_model=DropzoneActionResponse)
def start_dropzone_watcher() -> Dict[str, Any]:
    """
    Start the dropzone watcher.

    Requires DROPZONE_ENABLED=true and valid DROPZONE_PATHS.
    """
    if not WATCHDOG_AVAILABLE:
        raise HTTPException(
            status_code=400,
            detail="watchdog library not installed. Install with: pip install watchdog",
        )

    if dropzone_watcher.is_running():
        return {
            "success": True,
            "message": "Dropzone watcher already running",
            "running": True,
        }

    success = dropzone_watcher.start()

    if success:
        logger.info("dropzone.api.started")
        return {
            "success": True,
            "message": "Dropzone watcher started successfully",
            "running": True,
        }
    else:
        return {
            "success": False,
            "message": "Failed to start dropzone watcher. Check DROPZONE_ENABLED and DROPZONE_PATHS.",
            "running": False,
        }


@router.post("/stop", response_model=DropzoneActionResponse)
def stop_dropzone_watcher() -> Dict[str, Any]:
    """
    Stop the dropzone watcher.
    """
    if not dropzone_watcher.is_running():
        return {
            "success": True,
            "message": "Dropzone watcher not running",
            "running": False,
        }

    dropzone_watcher.stop()
    logger.info("dropzone.api.stopped")

    return {
        "success": True,
        "message": "Dropzone watcher stopped",
        "running": False,
    }


@router.get("/config")
def get_dropzone_config() -> Dict[str, Any]:
    """
    Get dropzone configuration (from environment).
    """
    return {
        "dropzone_enabled": os.getenv("DROPZONE_ENABLED", "false"),
        "dropzone_paths": os.getenv("DROPZONE_PATHS", ""),
        "poll_interval_seconds": int(os.getenv("DROPZONE_POLL_INTERVAL", "5")),
        "debounce_seconds": float(os.getenv("DROPZONE_DEBOUNCE_SECONDS", "2")),
        "watchdog_available": WATCHDOG_AVAILABLE,
    }
