﻿# backend/app/api/deps.py
"""
Common dependencies for API endpoints.
This module provides reusable dependency injections for FastAPI routes.

Pivot 2026: Conditional JWT imports for local mode (PIN-based auth)
"""

import logging
import os
import uuid
from typing import Optional

from fastapi import Depends, HTTPException, Request, status
from fastapi.security import HTTPAuthorizationCredentials, HTTPBearer
from sqlalchemy.orm import Session

# Pivot 2026: Conditional JWT imports
IS_LOCAL_MODE = os.getenv("KAIFARMA_LOCAL", "").lower() == "true"

if IS_LOCAL_MODE:
    # Local mode: Stub JWT classes (not used - PIN-based auth)
    JWTError = Exception
    jwt = None  # type: ignore
else:
    # Cloud mode: Full JWT support
    from jose import JWTError, jwt

from app.core.permissions import VIEW_SYSTEM_STATUS
from app.core.security import ALGORITHM, SECRET_KEY
from app.core.subscription_limits import Permission  # Issue #541: Business permissions
from app.database import get_db as get_database_session
from app.models.user import User
from app.utils.datetime_utils import utc_now

logger = logging.getLogger(__name__)

# Security scheme for JWT Bearer token
security = HTTPBearer(auto_error=False)


# =============================================================================
# Pivot 2026: Local Session-Based Authentication
# =============================================================================


async def get_current_session():
    """
    Dependency for local mode - replaces get_current_user.

    Returns the current PharmacySession if:
    1. Running in local mode (KAIFARMA_LOCAL=true)
    2. Security manager is initialized
    3. Terminal is unlocked

    Raises:
        503: System not initialized
        401: Terminal locked (needs PIN)
        500: Not in local mode
    """
    from app.core.security_local.local_state import security_manager, PharmacySession

    if not security_manager.is_local_mode():
        raise HTTPException(
            status_code=status.HTTP_500_INTERNAL_SERVER_ERROR,
            detail="get_current_session called outside local mode",
        )

    session = security_manager.get_session()
    if not session:
        raise HTTPException(
            status_code=status.HTTP_503_SERVICE_UNAVAILABLE,
            detail="Sistema no inicializado",
        )

    if not session.is_unlocked:
        raise HTTPException(
            status_code=status.HTTP_401_UNAUTHORIZED,
            detail="Terminal bloqueado. Introduzca PIN.",
        )

    return session


def get_db():
    """
    Dependency to get database session.
    Yields a database session and ensures it's closed after use.
    """
    yield from get_database_session()


def get_or_create_dev_user(db: Session, email: str) -> User:
    """
    Get or create development user for testing.

    This function ensures the database has the correct schema and assigns a pharmacy.
    Thread-safe implementation handles race conditions when multiple workers try to
    create the same development user/pharmacy concurrently (important for Render multi-worker).

    Development Pharmacy Convention:
        NIF: "DEV00000000" - Reserved identifier for development/testing environments
        Name: "Farmacia de Desarrollo"
        This convention is documented in backend/CLAUDE.md

    ⚠️  CRITICAL (Issue #225 - 1:1 User-Pharmacy Relationship):
        Since each pharmacy has exactly ONE user (1:1 relationship), we MUST:
        1. Check if the development pharmacy (DEV00000000) already has a user
        2. If YES: Return that existing user (regardless of email)
        3. If NO: Create a new user for the development pharmacy

        This prevents IntegrityError violations of the UNIQUE constraint on pharmacy_id.

    Args:
        db: Database session
        email: Email address for the development user (may be ignored if user exists)

    Returns:
        User: Development user instance with associated pharmacy

    Raises:
        HTTPException: If database operations fail after retry
    """
    # Import here to avoid circular imports
    from sqlalchemy.exc import IntegrityError

    from app.core.security import get_password_hash
    from app.models.pharmacy import Pharmacy

    # Get or create development pharmacy (thread-safe)
    dev_pharmacy = db.query(Pharmacy).filter(Pharmacy.nif == "DEV00000000").first()

    if not dev_pharmacy:
        # Create development pharmacy
        dev_pharmacy = Pharmacy(
            id=uuid.uuid4(),
            name="Farmacia de Desarrollo",
            nif="DEV00000000",
            address="Dirección de Desarrollo",
            created_at=utc_now(),
            updated_at=utc_now(),
        )
        try:
            db.add(dev_pharmacy)
            db.flush()  # Flush to get the ID
            logger.info(
                "Created development pharmacy",
                extra={"pharmacy_id": str(dev_pharmacy.id), "nif": "DEV00000000", "operation": "create_dev_pharmacy"},
            )
        except IntegrityError as e:
            # Race condition: Another worker created it concurrently
            db.rollback()
            db.expire_all()  # Clear stale references after rollback
            logger.warning(
                "Development pharmacy already exists (race condition handled)",
                extra={"nif": "DEV00000000", "operation": "create_dev_pharmacy", "error_type": "IntegrityError"},
            )
            # Retrieve the pharmacy created by the other worker
            dev_pharmacy = db.query(Pharmacy).filter(Pharmacy.nif == "DEV00000000").first()

            if not dev_pharmacy:
                # This should never happen, but handle it defensively
                logger.error(
                    "Failed to retrieve development pharmacy after IntegrityError",
                    extra={"nif": "DEV00000000", "operation": "create_dev_pharmacy", "error": str(e)},
                )
                raise HTTPException(
                    status_code=status.HTTP_500_INTERNAL_SERVER_ERROR,
                    detail="Failed to create or retrieve development pharmacy",
                )
        except Exception as e:
            db.rollback()
            logger.error(
                "Unexpected error creating development pharmacy",
                extra={
                    "nif": "DEV00000000",
                    "operation": "create_dev_pharmacy",
                    "error_type": type(e).__name__,
                    "error": str(e),
                },
            )
            raise HTTPException(
                status_code=status.HTTP_500_INTERNAL_SERVER_ERROR,
                detail=f"Failed to create development pharmacy: {str(e)}",
            )

    # ✅ CRITICAL FIX (Issue #225): Check if pharmacy already has a user (1:1 relationship)
    existing_user = db.query(User).filter(User.pharmacy_id == dev_pharmacy.id).first()

    if existing_user:
        # Check for email mismatch and warn if different
        if existing_user.email != email:
            logger.warning(
                "Email mismatch: Requested email differs from existing user email due to 1:1 pharmacy constraint",
                extra={
                    "user_id": str(existing_user.id),
                    "existing_email": existing_user.email,
                    "requested_email": email,
                    "pharmacy_id": str(dev_pharmacy.id),
                    "pharmacy_nif": "DEV00000000",
                    "operation": "get_dev_user",
                    "constraint": "1:1 User-Pharmacy relationship enforced",
                },
            )

        logger.info(
            "Using existing development user (1:1 pharmacy-user relationship)",
            extra={
                "user_id": str(existing_user.id),
                "email": existing_user.email,
                "pharmacy_id": str(dev_pharmacy.id),
                "requested_email": email,
                "operation": "get_dev_user",
            },
        )
        return existing_user

    # No existing user for this pharmacy - safe to create new one
    # Get development user password from environment
    dev_password = os.getenv("DEV_USER_PASSWORD", "dev123")

    # Create new development user with all required fields INCLUDING pharmacy_id
    user = User(
        id=uuid.uuid4(),
        pharmacy_id=dev_pharmacy.id,  # CRITICAL: Assign pharmacy_id
        email=email,
        username=email.split("@")[0],
        full_name="Development User",
        hashed_password=get_password_hash(dev_password),
        is_active=True,
        is_superuser=True,
        is_verified=True,
        role="admin",
        created_at=utc_now(),
        updated_at=utc_now(),
        # OAuth fields (nullable)
        oauth_provider=None,
        oauth_provider_id=None,
        # Notification preferences
        notify_uploads=True,
        notify_errors=True,
        notify_analysis=True,
    )

    try:
        db.add(user)
        db.commit()
        db.refresh(user)
        logger.info(
            "Created development user",
            extra={
                "user_id": str(user.id),
                "email": email,
                "pharmacy_id": str(dev_pharmacy.id),
                "operation": "create_dev_user",
            },
        )
    except IntegrityError as e:
        # Race condition: Another worker created the user concurrently
        # OR pharmacy_id constraint violated (should not happen due to check above)
        db.rollback()

        # CRITICAL: After rollback, session may have stale object references.
        # We need to clear the cache and re-query to get fresh data.
        db.expire_all()

        logger.warning(
            "Development user creation failed (race condition or constraint violation)",
            extra={
                "email": email,
                "pharmacy_id": str(dev_pharmacy.id) if dev_pharmacy else None,
                "operation": "create_dev_user",
                "error_type": "IntegrityError",
                "error": str(e),
            },
        )

        # Re-query the pharmacy to get a valid reference after rollback
        # The pharmacy might have been created by another thread
        fresh_pharmacy = db.query(Pharmacy).filter(Pharmacy.nif == "DEV00000000").first()

        if not fresh_pharmacy:
            logger.error(
                "Failed to retrieve development pharmacy after user IntegrityError",
                extra={
                    "operation": "create_dev_user",
                    "error": str(e),
                },
            )
            raise HTTPException(
                status_code=status.HTTP_500_INTERNAL_SERVER_ERROR,
                detail="Failed to create or retrieve development user",
            )

        # Try to retrieve existing user by pharmacy_id (1:1 relationship)
        user = db.query(User).filter(User.pharmacy_id == fresh_pharmacy.id).first()

        if not user:
            # This should never happen, but handle it defensively
            logger.error(
                "Failed to retrieve development user after IntegrityError",
                extra={
                    "email": email,
                    "pharmacy_id": str(fresh_pharmacy.id),
                    "operation": "create_dev_user",
                    "error": str(e),
                },
            )
            raise HTTPException(
                status_code=status.HTTP_500_INTERNAL_SERVER_ERROR,
                detail="Failed to create or retrieve development user",
            )
    except Exception as e:
        # If creation fails, handle gracefully
        db.rollback()
        logger.error(
            "Unexpected error creating development user",
            extra={
                "email": email,
                "operation": "create_dev_user",
                "error_type": type(e).__name__,
                "error": str(e),
                "pharmacy_id": str(dev_pharmacy.id) if dev_pharmacy else None,
            },
        )
        raise HTTPException(
            status_code=status.HTTP_500_INTERNAL_SERVER_ERROR,
            detail=f"Database schema issue: Please run 'alembic upgrade head' to apply migrations. Error: {str(e)}",
        )

    return user


async def get_current_user(
    request: Request,
    db: Session = Depends(get_db),
    credentials: Optional[HTTPAuthorizationCredentials] = Depends(security),
) -> User:
    """
    Get the current authenticated user from JWT token.

    All requests require valid JWT authentication (development and production).
    Pivot 2026: In local mode, use get_current_session() instead.

    Args:
        request: FastAPI request object (unused, kept for FastAPI compatibility)
        db: Database session
        credentials: JWT Bearer token from request header

    Returns:
        Current authenticated user

    Raises:
        HTTPException: If token is invalid or user not found
    """
    # Pivot 2026: This function is not available in local mode
    if IS_LOCAL_MODE:
        raise HTTPException(
            status_code=status.HTTP_501_NOT_IMPLEMENTED,
            detail="JWT auth not available in local mode. Use PIN-based auth via /api/v1/auth/local/",
        )

    # JWT authentication required for all environments
    if not credentials:
        raise HTTPException(
            status_code=status.HTTP_401_UNAUTHORIZED,
            detail="Autenticación requerida",
            headers={"WWW-Authenticate": "Bearer"},
        )

    try:
        # Decode JWT token
        payload = jwt.decode(credentials.credentials, SECRET_KEY, algorithms=[ALGORITHM])
        user_id: str = payload.get("sub")

        if user_id is None:
            raise HTTPException(
                status_code=status.HTTP_401_UNAUTHORIZED,
                detail="Token inválido",
                headers={"WWW-Authenticate": "Bearer"},
            )

    except JWTError:
        raise HTTPException(
            status_code=status.HTTP_401_UNAUTHORIZED,
            detail="Token inválido o expirado",
            headers={"WWW-Authenticate": "Bearer"},
        )

    # Get user from database
    try:
        # Try to parse UUID
        user_uuid = uuid.UUID(user_id)
        user = db.query(User).filter(User.id == user_uuid).first()
    except ValueError:
        # Invalid UUID format
        user = None

    if user is None:
        raise HTTPException(status_code=status.HTTP_404_NOT_FOUND, detail="Usuario no encontrado")

    if not user.is_active:
        raise HTTPException(status_code=status.HTTP_403_FORBIDDEN, detail="Usuario inactivo")

    # Validar que el usuario tiene farmacia asignada (Issue #102)
    # Esto previene estados corruptos donde un usuario no tiene farmacia
    if not user.pharmacy_id:
        raise HTTPException(
            status_code=status.HTTP_403_FORBIDDEN,
            detail="User must have an associated pharmacy. Please contact support.",
        )

    return user


async def get_current_active_user(current_user: User = Depends(get_current_user)) -> User:
    """
    Get current active user.

    Args:
        current_user: Current authenticated user

    Returns:
        Current active user

    Raises:
        HTTPException: If user is inactive
    """
    if not current_user.is_active:
        raise HTTPException(status_code=status.HTTP_403_FORBIDDEN, detail="Usuario inactivo")
    return current_user


async def get_current_admin_user(current_user: User = Depends(get_current_active_user)) -> User:
    """
    Get current user with admin privileges.

    Requires VIEW_SYSTEM_STATUS permission to access admin endpoints.

    Args:
        current_user: Current active user

    Returns:
        Current admin user

    Raises:
        HTTPException: If user doesn't have admin permissions
    """
    if not current_user.has_permission(VIEW_SYSTEM_STATUS):
        # Audit log: Permission denied
        logger.warning(
            f"Permission denied: {current_user.email} attempted {VIEW_SYSTEM_STATUS}",
            extra={
                "user_id": str(current_user.id),
                "email": current_user.email,
                "role": current_user.role,
                "permission_required": VIEW_SYSTEM_STATUS,
            },
        )

        raise HTTPException(status_code=status.HTTP_403_FORBIDDEN, detail="Permisos de administrador requeridos")

    return current_user


# ============================================================================
# SUBSCRIPTION PLAN ENFORCEMENT (Issue #402)
# ============================================================================

# Plan hierarchy for subscription feature gating
PLAN_HIERARCHY = {
    "free": 0,
    "pro": 1,
    "max": 2,
}


def require_subscription_plan(min_plan: str):
    """
    Decorator factory para enforcement de plan de suscripción.

    Issue #402: Employee filtering requiere plan PRO+.
    DECISIÓN #1: Plan enforcement usando decorator pattern.

    Args:
        min_plan: Plan mínimo requerido ('free', 'pro', 'max')

    Returns:
        Dependency function que valida el plan del usuario

    Raises:
        HTTPException 403: Si plan usuario < min_plan

    Example:
        @router.get("/employees/{pharmacy_id}")
        async def get_employees(
            current_user: User = Depends(require_subscription_plan("pro"))
        ):
            # Solo usuarios PRO+ pueden acceder
            ...
    """
    async def plan_checker(current_user: User = Depends(get_current_user)) -> User:
        """Verificar que el usuario tiene el plan mínimo requerido"""
        user_plan = current_user.subscription_plan or "free"
        min_plan_level = PLAN_HIERARCHY.get(min_plan, 0)
        user_plan_level = PLAN_HIERARCHY.get(user_plan, 0)

        if user_plan_level < min_plan_level:
            logger.warning(
                f"[PLAN_ENFORCEMENT] Usuario {current_user.email} "
                f"(plan: {user_plan}) intentó acceder a feature {min_plan}+",
                extra={
                    "user_id": str(current_user.id),
                    "user_email": current_user.email,
                    "user_plan": user_plan,
                    "required_plan": min_plan,
                    "security_event": "subscription_plan_denied",
                }
            )
            raise HTTPException(
                status_code=status.HTTP_403_FORBIDDEN,
                detail=f"Esta funcionalidad requiere plan {min_plan.upper()} o superior. "
                       f"Plan actual: {user_plan.upper()}. "
                       f"Por favor actualice su suscripción para acceder a esta funcionalidad."
            )

        return current_user

    return plan_checker


def verify_pharmacy_access(pharmacy_id: uuid.UUID, current_user: User) -> None:
    """
    Verifica que el usuario tenga acceso a la farmacia solicitada.

    Esta función es CRÍTICA para la seguridad del sistema multi-tenant (Issue #254, PR #289).
    Previene acceso no autorizado a datos de otras farmacias manipulando el pharmacy_id en requests.

    Arquitectura: 1 user = 1 pharmacy (ver REGLA #10 en CLAUDE.md)
    - Cada usuario tiene exactamente UNA farmacia asignada (User.pharmacy_id)
    - Relación 1:1 enforced por constraint UNIQUE en pharmacy_id
    - NO es posible que un usuario acceda a datos de otra farmacia

    Args:
        pharmacy_id: ID de la farmacia que se intenta acceder
        current_user: Usuario autenticado desde JWT token

    Raises:
        HTTPException: 403 Forbidden si el usuario no tiene acceso a la farmacia

    Example:
        ```python
        @router.get("/analysis/{pharmacy_id}")
        async def get_analysis(
            pharmacy_id: UUID,
            current_user: User = Depends(get_current_user)
        ):
            # Validar acceso ANTES de cualquier operación
            verify_pharmacy_access(pharmacy_id, current_user)

            # Proceder con la lógica del endpoint
            return service.get_analysis(pharmacy_id)
        ```

    Security Notes:
        - SIEMPRE llamar ANTES de acceder a datos de farmacia
        - NO confiar solo en el pharmacy_id del JWT (podría estar manipulado)
        - Loguear intentos de acceso no autorizado para auditoría
    """
    if str(current_user.pharmacy_id) != str(pharmacy_id):
        # Audit log: Intento de acceso no autorizado
        logger.warning(
            "SECURITY: Intento de acceso no autorizado a farmacia",
            extra={
                "user_id": str(current_user.id),
                "user_email": current_user.email,
                "user_pharmacy_id": str(current_user.pharmacy_id),
                "attempted_pharmacy_id": str(pharmacy_id),
                "security_event": "unauthorized_pharmacy_access_attempt",
            },
        )

        raise HTTPException(
            status_code=status.HTTP_403_FORBIDDEN, detail="No tiene permisos para acceder a los datos de esta farmacia"
        )


# =============================================================================
# RBAC PERMISSION DECORATOR (Issue #348)
# =============================================================================

def require_permission(permission: str):
    """
    Decorator factory para proteger endpoints con permisos RBAC.

    Issue #348: Admin Panel Unificado - RBAC enforcement

    Este decorator verifica que el usuario autenticado tenga el permiso
    requerido basándose en su role_obj.has_permission().

    Args:
        permission: Permission constant from app.core.subscription_limits

    Returns:
        Dependency function que valida el permiso

    Example:
        ```python
        from app.core.subscription_limits import Permission
        from app.api.deps import require_permission

        @router.post("/admin/clean-catalog")
        async def clean_catalog(
            current_user: User = Depends(require_permission(Permission.MANAGE_USERS))
        ):
            # Solo usuarios con permiso manage_users pueden ejecutar
            return {"status": "cleaning"}
        ```

    Permission Flow:
        1. Usuario hace request con JWT token
        2. get_current_user() valida token y obtiene User
        3. require_permission() verifica User.has_permission(permission)
        4. Si tiene permiso → proceder con endpoint
        5. Si NO tiene permiso → 403 Forbidden

    Security:
        - Prioridad: is_superuser > deleted_at > is_active > role_obj > legacy role
        - Soft deleted users (deleted_at) → 403 Forbidden
        - Inactive users (is_active=False) → 403 Forbidden
        - Superusers (is_superuser=True) → Always allowed
    """
    async def permission_checker(
        current_user: User = Depends(get_current_user)
    ) -> User:
        """
        Verifica que el usuario tenga el permiso requerido.

        Args:
            current_user: Usuario autenticado obtenido de JWT token

        Returns:
            Usuario autenticado (si tiene el permiso)

        Raises:
            HTTPException: 403 Forbidden si el usuario no tiene el permiso
        """
        # Verificar permiso usando User.has_permission()
        if not current_user.has_permission(permission):
            # Audit log: Intento de acceso sin permisos
            logger.warning(
                "RBAC: Intento de acceso sin permiso requerido",
                extra={
                    "user_id": str(current_user.id),
                    "user_email": current_user.email,
                    "user_role": current_user.role,
                    "user_role_id": str(current_user.role_id) if current_user.role_id else None,
                    "required_permission": permission,
                    "is_superuser": current_user.is_superuser,
                    "is_active": current_user.is_active,
                    "deleted_at": current_user.deleted_at.isoformat() if current_user.deleted_at else None,
                    "security_event": "rbac_permission_denied",
                },
            )

            # Mensaje específico según estado del usuario
            if current_user.deleted_at:
                detail = "Cuenta eliminada. No tiene permisos para acceder a este recurso."
            elif not current_user.is_active:
                detail = "Cuenta inactiva. Contacte al administrador para reactivar su cuenta."
            else:
                detail = f"No tiene el permiso requerido: {permission}. Contacte al administrador."

            raise HTTPException(
                status_code=status.HTTP_403_FORBIDDEN,
                detail=detail
            )

        return current_user

    return permission_checker


# =============================================================================
# ROLE-BASED ACCESS CONTROL - TITULAR VS OPERATIVO (Issue #541)
# =============================================================================

async def get_current_titular(
    current_user: User = Depends(get_current_user)
) -> User:
    """
    Require user to have TITULAR role (or admin).

    Issue #541: Titular has access to ALL data including financial.
    Operativo users are denied access to sensitive endpoints.

    Args:
        current_user: Authenticated user from JWT

    Returns:
        User if has titular/admin role

    Raises:
        HTTPException 403: If user is operativo or doesn't have financial permissions
    """
    # Admins and superusers always have access
    if current_user.is_superuser or current_user.role == "admin":
        return current_user

    # Check if user is titular
    if current_user.role == "titular":
        return current_user

    # Check permission explicitly (handles role_obj if set)
    if current_user.has_permission(Permission.VIEW_FINANCIAL_DATA):
        return current_user

    # Deny access for operativo and user roles
    logger.warning(
        "RBAC: Acceso denegado a datos financieros para rol operativo",
        extra={
            "user_id": str(current_user.id),
            "user_email": current_user.email,
            "user_role": current_user.role,
            "required_role": "titular",
            "security_event": "titular_access_denied",
        },
    )

    raise HTTPException(
        status_code=status.HTTP_403_FORBIDDEN,
        detail="Acceso restringido. Solo el titular de la farmacia puede ver datos financieros."
    )


def require_financial_access():
    """
    Decorator to protect endpoints that expose financial data.

    Issue #541: Use this for endpoints showing margins, costs, profitability.

    Financial data includes:
    - Márgenes de beneficio (pvp - pmc)
    - Costes de compra (PMC)
    - Análisis de rentabilidad
    - Facturación por empleado

    Example:
        ```python
        @router.get("/analysis/{pharmacy_id}/margins")
        async def get_margins(
            pharmacy_id: UUID,
            current_user: User = Depends(require_financial_access())
        ):
            # Only titular/admin can access
            return margin_service.get_margins(pharmacy_id)
        ```
    """
    return Depends(get_current_titular)


async def get_current_with_employee_access(
    current_user: User = Depends(get_current_user)
) -> User:
    """
    Require user to have access to employee data.

    Issue #541: Only titular can see employee-level data (facturación por empleado).

    Returns:
        User if has employee data access

    Raises:
        HTTPException 403: If user doesn't have employee data permission
    """
    if current_user.is_superuser or current_user.role in ("admin", "titular"):
        return current_user

    if current_user.has_permission(Permission.VIEW_EMPLOYEE_DATA):
        return current_user

    logger.warning(
        "RBAC: Acceso denegado a datos de empleados",
        extra={
            "user_id": str(current_user.id),
            "user_email": current_user.email,
            "user_role": current_user.role,
            "security_event": "employee_data_access_denied",
        },
    )

    raise HTTPException(
        status_code=status.HTTP_403_FORBIDDEN,
        detail="Acceso restringido. Solo el titular puede ver datos de empleados."
    )


def require_employee_access():
    """
    Decorator to protect endpoints that expose employee data.

    Issue #541: Use for employee filtering, facturación por empleado, etc.

    Example:
        ```python
        @router.get("/employees/{pharmacy_id}/sales")
        async def get_employee_sales(
            pharmacy_id: UUID,
            current_user: User = Depends(require_employee_access())
        ):
            # Only titular can access
            return employee_service.get_sales_by_employee(pharmacy_id)
        ```
    """
    return Depends(get_current_with_employee_access)


async def get_current_with_partner_access(
    current_user: User = Depends(get_current_user)
) -> User:
    """
    Require user to have access to partner management.

    Issue #541: Only titular can configure partners/laboratorios.

    Returns:
        User if has partner management access

    Raises:
        HTTPException 403: If user doesn't have partner permission
    """
    if current_user.is_superuser or current_user.role in ("admin", "titular"):
        return current_user

    if current_user.has_permission(Permission.MANAGE_PARTNERS):
        return current_user

    logger.warning(
        "RBAC: Acceso denegado a gestión de partners",
        extra={
            "user_id": str(current_user.id),
            "user_email": current_user.email,
            "user_role": current_user.role,
            "security_event": "partner_access_denied",
        },
    )

    raise HTTPException(
        status_code=status.HTTP_403_FORBIDDEN,
        detail="Acceso restringido. Solo el titular puede gestionar partners."
    )


def require_partner_access():
    """
    Decorator to protect partner management endpoints.

    Issue #541: Use for partner selection, laboratory config, etc.

    Example:
        ```python
        @router.put("/pharmacy-partners/{pharmacy_id}")
        async def update_partners(
            pharmacy_id: UUID,
            partners: List[str],
            current_user: User = Depends(require_partner_access())
        ):
            # Only titular can modify
            return partner_service.update_partners(pharmacy_id, partners)
        ```
    """
    return Depends(get_current_with_partner_access)
