# backend/app/api/customer_orders.py
"""
API endpoints for Customer Order Tracking (Modulo 1: Cazador de Encargos).

Provides endpoints to:
- Get pending orders summary and list
- Sync from ERP
- Mark orders as contacted/for-return/resolved
- Add notes to orders

Local single-tenant: No pharmacy_id needed.
Auth: PIN-based via LocalSecurityManager (Pivot 2026).

See: docs/CONTROL_MODULES_ARCHITECTURE.md
"""
import os
from datetime import datetime
from typing import Any, Dict, List, Optional

import structlog
from fastapi import APIRouter, BackgroundTasks, Depends, HTTPException, Query, status
from pydantic import BaseModel, Field
from sqlalchemy.orm import Session

from app.database import get_db
from app.models.tracking import CustomerOrderTracking
from app.services.customer_order_service import CustomerOrderService

logger = structlog.get_logger()
router = APIRouter(prefix="/customer-orders", tags=["customer-orders"])


# =============================================================================
# Local Auth Dependency (Pivot 2026)
# =============================================================================


async def require_unlocked_session():
    """
    Dependency to require unlocked terminal for protected endpoints.

    In local mode (KAIFARMA_LOCAL=true), verifies the terminal is unlocked.
    Returns the session if unlocked, raises 401 if locked.
    """
    from app.core.security_local.local_state import security_manager

    session = security_manager.get_session()

    if session is None:
        # Manager not initialized - system starting up
        raise HTTPException(
            status_code=status.HTTP_503_SERVICE_UNAVAILABLE,
            detail="Sistema inicializando. Espere un momento.",
        )

    if not session.is_unlocked:
        raise HTTPException(
            status_code=status.HTTP_401_UNAUTHORIZED,
            detail="Terminal bloqueado. Introduzca PIN para continuar.",
        )

    # Refresh activity timestamp to prevent auto-lock
    security_manager.refresh_activity()

    return session


# =============================================================================
# Pydantic Models (Request/Response)
# =============================================================================


class OrderSummaryResponse(BaseModel):
    """Summary statistics for dashboard widget."""

    total_pending: int = Field(description="Total pending orders")
    total_value: float = Field(description="Total PVP value (EUR)")
    by_severity: Dict[str, Dict[str, Any]] = Field(
        description="Breakdown by severity (critical, urgent, warning, info)"
    )
    by_status: Dict[str, int] = Field(
        description="Breakdown by status (NEW, ALERTED, CONTACTED, TO_RETURN)"
    )


class OrderResponse(BaseModel):
    """Single order response."""

    erp_order_id: str
    customer_name: str
    customer_phone: Optional[str]
    product_description: str
    product_cn: Optional[str]
    cached_pvl: float = Field(description="Unit PVP value")
    cached_units: int
    economic_value: float = Field(description="Total value (pvl * units)")
    status: str
    severity: str
    days_pending: int
    is_spam: bool
    first_detected_at: datetime
    last_seen_in_erp: datetime
    contacted_at: Optional[datetime]
    marked_return_at: Optional[datetime]
    resolved_at: Optional[datetime]
    notes: Optional[str]

    class Config:
        from_attributes = True


class OrderListResponse(BaseModel):
    """List of orders response."""

    orders: List[OrderResponse]
    total: int
    filters_applied: Dict[str, Any]


class SyncResultResponse(BaseModel):
    """Response from sync operation."""

    new: int = Field(description="New orders detected")
    updated: int = Field(description="Existing orders updated")
    resolved: int = Field(description="Orders auto-resolved (disappeared from ERP)")
    total_pending: int = Field(description="Current pending count")
    total_value: float = Field(description="Total value of pending orders")
    sync_timestamp: str


class ActionRequest(BaseModel):
    """Request for marking an order."""

    notes: Optional[str] = Field(None, description="Optional notes about the action")


class ActionResponse(BaseModel):
    """Response after an action."""

    success: bool
    erp_order_id: str
    new_status: str
    message: str


class NotesRequest(BaseModel):
    """Request for adding notes."""

    notes: str = Field(..., min_length=1, max_length=1000)


# =============================================================================
# Helper Functions
# =============================================================================


def _get_service(db: Session) -> CustomerOrderService:
    """Get CustomerOrderService instance."""
    return CustomerOrderService(session=db)


def _order_to_response(order: CustomerOrderTracking) -> OrderResponse:
    """Convert model to response."""
    return OrderResponse(
        erp_order_id=order.erp_order_id,
        customer_name=order.customer_name,
        customer_phone=order.customer_phone,
        product_description=order.product_description,
        product_cn=order.product_cn,
        cached_pvl=order.cached_pvl or 0.0,
        cached_units=order.cached_units or 1,
        economic_value=order.economic_value,
        status=order.status,
        severity=order.severity,
        days_pending=order.days_pending,
        is_spam=order.is_spam,
        first_detected_at=order.first_detected_at,
        last_seen_in_erp=order.last_seen_in_erp,
        contacted_at=order.contacted_at,
        marked_return_at=order.marked_return_at,
        resolved_at=order.resolved_at,
        notes=order.notes,
    )


# =============================================================================
# Endpoints
# =============================================================================


@router.get("/summary", response_model=OrderSummaryResponse)
def get_summary(
    db: Session = Depends(get_db),
    _session=Depends(require_unlocked_session),
) -> Dict[str, Any]:
    """
    Get summary statistics for dashboard widget.

    Returns total pending orders, value, and breakdown by severity/status.
    """
    service = _get_service(db)
    summary = service.get_summary()

    logger.info(
        "customer_orders.summary",
        total_pending=summary["total_pending"],
        total_value=summary["total_value"],
    )

    return summary


@router.get("/pending", response_model=OrderListResponse)
def get_pending_orders(
    severity: Optional[List[str]] = Query(
        None,
        description="Filter by severity (critical, urgent, warning, info)",
    ),
    exclude_spam: bool = Query(
        True,
        description="Exclude recently contacted orders",
    ),
    limit: int = Query(100, ge=1, le=500),
    db: Session = Depends(get_db),
    _session=Depends(require_unlocked_session),
) -> Dict[str, Any]:
    """
    Get list of pending orders for dashboard.

    Sorted by severity (critical first) and days pending.
    """
    service = _get_service(db)
    orders = service.get_pending_orders(
        severity_filter=severity,
        exclude_spam=exclude_spam,
        limit=limit,
    )

    logger.info(
        "customer_orders.pending.fetched",
        count=len(orders),
        severity_filter=severity,
        exclude_spam=exclude_spam,
    )

    return {
        "orders": [_order_to_response(o) for o in orders],
        "total": len(orders),
        "filters_applied": {
            "severity": severity,
            "exclude_spam": exclude_spam,
            "limit": limit,
        },
    }


@router.get("/order/{erp_order_id}", response_model=OrderResponse)
def get_order(
    erp_order_id: str,
    db: Session = Depends(get_db),
    _session=Depends(require_unlocked_session),
) -> OrderResponse:
    """
    Get a specific order by ERP ID.
    """
    service = _get_service(db)
    order = service.get_order(erp_order_id)

    if order is None:
        raise HTTPException(
            status_code=404,
            detail=f"Order not found: {erp_order_id}",
        )

    return _order_to_response(order)


@router.get("/to-return", response_model=OrderListResponse)
def get_orders_to_return(
    db: Session = Depends(get_db),
    _session=Depends(require_unlocked_session),
) -> Dict[str, Any]:
    """
    Get orders marked for return to supplier.

    Use this for generating return lists/reports.
    """
    service = _get_service(db)
    orders = service.get_orders_to_return()

    logger.info("customer_orders.to_return.fetched", count=len(orders))

    return {
        "orders": [_order_to_response(o) for o in orders],
        "total": len(orders),
        "filters_applied": {"status": "TO_RETURN"},
    }


@router.get("/recently-resolved", response_model=OrderListResponse)
def get_recently_resolved(
    days: int = Query(7, ge=1, le=90),
    limit: int = Query(50, ge=1, le=200),
    db: Session = Depends(get_db),
    _session=Depends(require_unlocked_session),
) -> Dict[str, Any]:
    """
    Get recently resolved orders for history view.
    """
    service = _get_service(db)
    orders = service.get_recently_resolved(days=days, limit=limit)

    logger.info(
        "customer_orders.resolved.fetched",
        count=len(orders),
        days=days,
    )

    return {
        "orders": [_order_to_response(o) for o in orders],
        "total": len(orders),
        "filters_applied": {"status": "RESOLVED", "days": days, "limit": limit},
    }


# =============================================================================
# Sync Endpoints
# =============================================================================


@router.post("/sync", response_model=SyncResultResponse)
def sync_from_erp(
    db: Session = Depends(get_db),
    _session=Depends(require_unlocked_session),
) -> Dict[str, Any]:
    """
    Sync pending orders from ERP.

    This fetches current pending orders from Farmanager and:
    - Creates tracking records for new orders
    - Updates cached data for existing orders
    - Auto-resolves orders that disappeared from ERP

    Note: Runs synchronously. For large datasets, use /sync-async.
    """
    # Get ERP adapter
    erp_type = os.getenv("ERP_TYPE", "").lower()
    if erp_type != "farmanager":
        raise HTTPException(
            status_code=400,
            detail=f"Customer order sync only supported for Farmanager. Current ERP: {erp_type or 'not configured'}",
        )

    from app.erp_adapters.farmanager.adapter import FarmanagerAdapter

    adapter = FarmanagerAdapter(
        host=os.getenv("ERP_HOST", "localhost"),
        port=int(os.getenv("ERP_PORT", "3306")),
        user=os.getenv("ERP_USER", "administrador"),
        password=os.getenv("ERP_PASSWORD", ""),
        database=os.getenv("ERP_DATABASE", "gesql"),
    )

    service = _get_service(db)
    service.set_erp_adapter(adapter)

    logger.info("customer_orders.sync.started")

    try:
        result = service.sync_from_erp()
        logger.info("customer_orders.sync.completed", **result)
        return result
    except Exception as e:
        logger.error("customer_orders.sync.failed", error=str(e))
        raise HTTPException(
            status_code=500,
            detail=f"Sync failed: {str(e)}",
        )


@router.post("/sync-async")
def sync_from_erp_async(
    background_tasks: BackgroundTasks,
    db: Session = Depends(get_db),
    _session=Depends(require_unlocked_session),
) -> Dict[str, str]:
    """
    Sync pending orders from ERP in background.

    Returns immediately. Check /summary for updated data.
    """
    erp_type = os.getenv("ERP_TYPE", "").lower()
    if erp_type != "farmanager":
        raise HTTPException(
            status_code=400,
            detail=f"Customer order sync only supported for Farmanager. Current ERP: {erp_type or 'not configured'}",
        )

    def _run_sync():
        from app.database import SessionLocal
        from app.erp_adapters.farmanager.adapter import FarmanagerAdapter

        adapter = FarmanagerAdapter(
            host=os.getenv("ERP_HOST", "localhost"),
            port=int(os.getenv("ERP_PORT", "3306")),
            user=os.getenv("ERP_USER", "administrador"),
            password=os.getenv("ERP_PASSWORD", ""),
            database=os.getenv("ERP_DATABASE", "gesql"),
        )

        with SessionLocal() as session:
            service = CustomerOrderService(session=session, erp_adapter=adapter)
            service.sync_from_erp()

    background_tasks.add_task(_run_sync)

    logger.info("customer_orders.sync.async_started")

    return {
        "status": "started",
        "message": "Sync started in background. Check /summary for updated data.",
    }


# =============================================================================
# Action Endpoints
# =============================================================================


@router.post("/order/{erp_order_id}/contacted", response_model=ActionResponse)
def mark_as_contacted(
    erp_order_id: str,
    request: ActionRequest = None,
    db: Session = Depends(get_db),
    _session=Depends(require_unlocked_session),
) -> Dict[str, Any]:
    """
    Mark order as contacted (pharmacist called the customer).

    This updates status to CONTACTED and records the timestamp.
    """
    service = _get_service(db)
    notes = request.notes if request else None

    try:
        order = service.mark_as_contacted(erp_order_id, notes=notes)
        logger.info(
            "customer_orders.action.contacted",
            erp_order_id=erp_order_id,
            notes=notes,
        )
        return {
            "success": True,
            "erp_order_id": erp_order_id,
            "new_status": order.status,
            "message": "Marcado como contactado",
        }
    except ValueError as e:
        raise HTTPException(status_code=404, detail=str(e))


@router.post("/order/{erp_order_id}/to-return", response_model=ActionResponse)
def mark_for_return(
    erp_order_id: str,
    request: ActionRequest = None,
    db: Session = Depends(get_db),
    _session=Depends(require_unlocked_session),
) -> Dict[str, Any]:
    """
    Mark order for return to supplier.

    Use this when the customer won't pick up and the product should be returned.
    """
    service = _get_service(db)
    notes = request.notes if request else None

    try:
        order = service.mark_for_return(erp_order_id, notes=notes)
        logger.info(
            "customer_orders.action.to_return",
            erp_order_id=erp_order_id,
            notes=notes,
        )
        return {
            "success": True,
            "erp_order_id": erp_order_id,
            "new_status": order.status,
            "message": "Marcado para devolucion",
        }
    except ValueError as e:
        raise HTTPException(status_code=404, detail=str(e))


@router.post("/order/{erp_order_id}/resolved", response_model=ActionResponse)
def mark_as_resolved(
    erp_order_id: str,
    db: Session = Depends(get_db),
    _session=Depends(require_unlocked_session),
) -> Dict[str, Any]:
    """
    Manually mark order as resolved.

    Use this when the order was picked up but not detected automatically.
    """
    service = _get_service(db)

    try:
        order = service.mark_as_resolved(erp_order_id)
        logger.info(
            "customer_orders.action.resolved",
            erp_order_id=erp_order_id,
        )
        return {
            "success": True,
            "erp_order_id": erp_order_id,
            "new_status": order.status,
            "message": "Marcado como resuelto",
        }
    except ValueError as e:
        raise HTTPException(status_code=404, detail=str(e))


@router.put("/order/{erp_order_id}/notes", response_model=ActionResponse)
def update_notes(
    erp_order_id: str,
    request: NotesRequest,
    db: Session = Depends(get_db),
    _session=Depends(require_unlocked_session),
) -> Dict[str, Any]:
    """
    Add or update notes for an order.
    """
    service = _get_service(db)

    try:
        order = service.add_notes(erp_order_id, request.notes)
        logger.info(
            "customer_orders.action.notes_updated",
            erp_order_id=erp_order_id,
        )
        return {
            "success": True,
            "erp_order_id": erp_order_id,
            "new_status": order.status,
            "message": "Notas actualizadas",
        }
    except ValueError as e:
        raise HTTPException(status_code=404, detail=str(e))


# =============================================================================
# Stats Endpoints
# =============================================================================


@router.get("/stats/by-severity")
def get_stats_by_severity(
    db: Session = Depends(get_db),
    _session=Depends(require_unlocked_session),
) -> Dict[str, Any]:
    """
    Get order statistics grouped by severity.

    Returns count and value for each severity level.
    """
    service = _get_service(db)
    summary = service.get_summary()

    return {
        "by_severity": summary["by_severity"],
        "total_pending": summary["total_pending"],
        "total_value": summary["total_value"],
    }


@router.get("/stats/critical")
def get_critical_stats(
    db: Session = Depends(get_db),
    _session=Depends(require_unlocked_session),
) -> Dict[str, Any]:
    """
    Get statistics for critical orders only (30+ days).

    Quick endpoint for dashboard alerts.
    """
    service = _get_service(db)
    critical_orders = service.get_pending_orders(
        severity_filter=["critical"],
        exclude_spam=False,
    )

    total_value = sum(o.economic_value for o in critical_orders)

    return {
        "count": len(critical_orders),
        "total_value": total_value,
        "oldest_days": max((o.days_pending for o in critical_orders), default=0),
        "orders": [
            {
                "erp_order_id": o.erp_order_id,
                "customer_name": o.customer_name,
                "product_description": o.product_description[:50],
                "days_pending": o.days_pending,
                "economic_value": o.economic_value,
            }
            for o in critical_orders[:10]  # Top 10 only
        ],
    }
