# backend/app/api/auth_local.py
"""
Local Authentication API for kaiFarma Desktop (Pivot 2026).

PIN-based terminal lock/unlock for local Windows installation.
Replaces JWT authentication for local mode.

Endpoints:
- GET  /api/v1/auth/local/status  - Check lock status (for polling)
- POST /api/v1/auth/local/unlock  - Unlock with PIN
- POST /api/v1/auth/local/lock    - Lock terminal manually
- GET  /api/v1/license           - Get license status
- POST /api/v1/license/revalidate - Force license revalidation
"""

import logging
import os
from datetime import datetime
from typing import Literal, Optional

from fastapi import APIRouter, Depends, HTTPException, Request, status
from pydantic import BaseModel, Field
from sqlalchemy.orm import Session

from app.core.security_local.local_state import security_manager
from app.database import get_db

logger = logging.getLogger(__name__)

router = APIRouter(prefix="/auth/local", tags=["auth-local"])


# =============================================================================
# Request/Response Models
# =============================================================================


class UnlockRequest(BaseModel):
    """Request to unlock the terminal."""

    pin: str = Field(..., min_length=4, max_length=6, description="4-6 digit PIN")


class StatusResponse(BaseModel):
    """Current terminal status."""

    is_unlocked: bool
    pharmacy_name: str
    auto_lock_minutes: int
    state: Literal["ready", "locked", "uninitialized"]
    role: str = "operativo"


class MessageResponse(BaseModel):
    """Simple message response."""

    message: str


# =============================================================================
# Endpoints
# =============================================================================


@router.get("/status", response_model=StatusResponse)
async def get_status():
    """
    Get current terminal lock status.

    Frontend polls this endpoint every 5 seconds to:
    - Know if lock screen should be shown
    - Detect if auto-lock occurred
    - Get pharmacy name for display
    """
    session = security_manager.get_session()

    if not session:
        logger.warning("[AUTH_LOCAL] Status check on uninitialized manager")
        return StatusResponse(
            is_unlocked=False,
            pharmacy_name="",
            auto_lock_minutes=5,
            state="uninitialized",
            role="operativo",
        )

    return StatusResponse(
        is_unlocked=session.is_unlocked,
        pharmacy_name=session.pharmacy_name,
        auto_lock_minutes=session.auto_lock_minutes,
        state="ready" if session.is_unlocked else "locked",
        role=session.role,
    )


@router.post("/unlock", response_model=MessageResponse)
async def unlock(payload: UnlockRequest):
    """
    Unlock the terminal with PIN.

    Returns 200 on success, 401 on incorrect PIN.
    """
    success = security_manager.unlock(payload.pin)

    if not success:
        logger.warning("[AUTH_LOCAL] Failed unlock attempt")
        raise HTTPException(
            status_code=status.HTTP_401_UNAUTHORIZED,
            detail="PIN incorrecto",
        )

    logger.info("[AUTH_LOCAL] Terminal unlocked")
    return MessageResponse(message="Terminal desbloqueado")


@router.post("/lock", response_model=MessageResponse)
async def lock():
    """
    Lock the terminal manually.

    Called when user clicks "Lock" button or from keyboard shortcut.
    """
    security_manager.lock()
    logger.info("[AUTH_LOCAL] Terminal locked manually")
    return MessageResponse(message="Terminal bloqueado")


@router.post("/set-pin", response_model=MessageResponse)
async def set_pin(payload: UnlockRequest):
    """
    Change the terminal PIN.

    Security (REGLA #7):
    - Terminal must be unlocked
    - Only 'titular' role can change PIN (not 'operativo')

    Operativo users can unlock/lock but cannot modify security settings.
    This prevents staff from locking out the pharmacy owner.
    """
    session = security_manager.get_session()

    if not session or not session.is_unlocked:
        raise HTTPException(
            status_code=status.HTTP_401_UNAUTHORIZED,
            detail="Terminal debe estar desbloqueado para cambiar PIN",
        )

    # REGLA #7: Only titular can change security settings
    if session.role != "titular":
        logger.warning(
            "[AUTH_LOCAL] Operativo user attempted to change PIN",
            extra={"role": session.role},
        )
        raise HTTPException(
            status_code=status.HTTP_403_FORBIDDEN,
            detail="Solo el titular puede cambiar el PIN",
        )

    security_manager.set_pin(payload.pin)
    logger.info("[AUTH_LOCAL] PIN changed by titular")
    return MessageResponse(message="PIN actualizado")


# =============================================================================
# License Router (separate prefix for clarity)
# =============================================================================

license_router = APIRouter(prefix="/license", tags=["license"])


class LicenseStatusResponse(BaseModel):
    """Current license status."""

    is_valid: bool
    tier: str
    pharmacy_id: Optional[str] = None
    pharmacy_name: Optional[str] = None
    validated_at: Optional[datetime] = None
    expires_at: Optional[datetime] = None
    in_grace_period: bool = False
    grace_period_ends: Optional[datetime] = None
    days_remaining: Optional[int] = None
    kill_switch_active: bool = False
    security_status: str = "ok"  # ok, time_travel, machine_mismatch
    needs_activation: bool = False


class RevalidateResponse(BaseModel):
    """Response from license revalidation."""

    success: bool
    message: str
    is_valid: bool
    kill_switch_active: bool = False
    in_grace_period: bool = False
    days_remaining: Optional[int] = None


@license_router.get("", response_model=LicenseStatusResponse)
async def get_license_status(db: Session = Depends(get_db)):
    """
    Get current license status.

    Returns license validation state including:
    - Is the license valid
    - License tier (free, standard, premium)
    - Grace period status if offline
    - Kill-switch status if grace period expired
    - Security status (time travel, machine mismatch)

    Frontend uses this to:
    - Show license status in footer/header
    - Display grace period warnings
    - Show read-only mode banner when kill-switch active
    """
    try:
        from app.services.license_client import LicenseClientService

        license_client = LicenseClientService(db)
        status = license_client.get_license_status()

        return LicenseStatusResponse(
            is_valid=status.get("is_valid", False),
            tier=status.get("tier", "unknown"),
            pharmacy_id=status.get("pharmacy_id"),
            pharmacy_name=status.get("pharmacy_name"),
            validated_at=status.get("validated_at"),
            expires_at=status.get("expires_at"),
            in_grace_period=status.get("in_grace_period", False),
            grace_period_ends=status.get("grace_period_ends"),
            days_remaining=status.get("days_remaining"),
            kill_switch_active=status.get("kill_switch_active", False),
            security_status=status.get("security_status", "ok"),
            needs_activation=status.get("needs_activation", False),
        )

    except Exception as e:
        logger.error(f"[LICENSE] Failed to get status: {e}")
        return LicenseStatusResponse(
            is_valid=False,
            tier="unknown",
            kill_switch_active=False,
            needs_activation=True,
        )


@license_router.post("/revalidate", response_model=RevalidateResponse)
async def revalidate_license(request: Request, db: Session = Depends(get_db)):
    """
    Force license revalidation against Hub.

    Use this when:
    - User has reconnected to internet after being offline
    - User has renewed their license and wants immediate effect
    - Grace period is running and user wants to stop it

    This endpoint requires the terminal to be unlocked.
    """
    # Check if terminal is unlocked
    session = security_manager.get_session()
    if not session or not session.is_unlocked:
        raise HTTPException(
            status_code=status.HTTP_401_UNAUTHORIZED,
            detail="Terminal debe estar desbloqueado para revalidar licencia",
        )

    try:
        from app.services.license_client import LicenseClientService

        license_key = os.getenv("KAIFARMA_LICENSE_KEY")
        if not license_key:
            logger.error("[LICENSE] KAIFARMA_LICENSE_KEY not configured")
            raise HTTPException(
                status_code=status.HTTP_500_INTERNAL_SERVER_ERROR,
                detail="License key not configured. Contact administrator.",
            )

        license_client = LicenseClientService(db)
        result = await license_client.validate_license(license_key)

        # Update app state
        if hasattr(request.app.state, "read_only_mode"):
            request.app.state.read_only_mode = result.kill_switch_active

        if result.is_valid:
            if result.in_grace_period:
                message = f"Licencia en periodo de gracia - {result.days_remaining} días restantes"
            else:
                message = "Licencia validada correctamente"
        else:
            message = result.error_message or "Error de validación de licencia"

        logger.info(
            "[LICENSE] Revalidation completed",
            extra={
                "is_valid": result.is_valid,
                "hub_reachable": result.hub_reachable,
            },
        )

        return RevalidateResponse(
            success=result.hub_reachable,
            message=message,
            is_valid=result.is_valid,
            kill_switch_active=result.kill_switch_active,
            in_grace_period=result.in_grace_period,
            days_remaining=result.days_remaining,
        )

    except Exception as e:
        logger.error(f"[LICENSE] Revalidation failed: {e}")
        return RevalidateResponse(
            success=False,
            message=f"Error de revalidación: {str(e)}",
            is_valid=False,
        )
