# backend/app/api/auth.py
"""
API endpoints para autenticación OAuth2 + Social Login
Soporta login tradicional y OAuth con Google/Microsoft

Pivot 2026: Conditional imports for local mode (no authlib/OAuth needed)
"""

import logging
import os
from typing import Any, Dict, Optional

from fastapi import APIRouter, Depends, HTTPException, Request, status
from fastapi.responses import RedirectResponse
from slowapi import Limiter
from slowapi.util import get_remote_address
from sqlalchemy.orm import Session

# Pivot 2026: Detect local mode BEFORE importing OAuth modules
IS_LOCAL_MODE = os.getenv("KAIFARMA_LOCAL", "").lower() == "true"

if IS_LOCAL_MODE:
    # Local mode: Provide stubs for OAuth components (not used)
    OAuthError = Exception  # Stub class
    oauth = None
    GOOGLE_CLIENT_ID = None
    MICROSOFT_CLIENT_ID = None
    from app.core.security import (
        get_current_active_user,
        get_password_hash,
        verify_password,
    )
else:
    # Cloud mode: Full OAuth support
    from authlib.integrations.starlette_client import OAuthError
    from app.core.security import (
        GOOGLE_CLIENT_ID,
        MICROSOFT_CLIENT_ID,
        get_current_active_user,
        get_password_hash,
        oauth,
        verify_password,
    )

from app.database import get_db
from app.models import User
from app.schemas.auth import (
    EmailVerification,
    LoginResponse,
    OAuthProviders,
    PasswordReset,
    PasswordResetRequest,
    RefreshTokenRequest,
    Token,
    UserLogin,
    UserResponse,
    UserUpdate,
)
from app.schemas.user_pharmacy import UserStorageUsageResponse
from app.services.audit_service import AuditAction, AuditService
from app.services.auth_service import auth_service

logger = logging.getLogger(__name__)

# Create rate limiter
limiter = Limiter(key_func=get_remote_address)

router = APIRouter(prefix="/auth", tags=["auth"])

# Base URL for OAuth redirects
BASE_URL = os.getenv("BASE_URL", "http://localhost:8000")
FRONTEND_URL = os.getenv("FRONTEND_URL", "http://localhost:8050")


# Public registration has been removed - users must be invited
# Use POST /invitations endpoint to create invitations
# Use POST /invitations/{token}/accept to register from invitation


@router.post("/login", response_model=LoginResponse)
@limiter.limit("10/minute")  # 10 login attempts per minute per IP
async def login(request: Request, credentials: UserLogin, db: Session = Depends(get_db)) -> Dict[str, Any]:
    """
    Login with email/password

    Features:
    - Validates credentials
    - Updates last_login timestamp
    - Returns access and refresh tokens
    """
    try:
        # Authenticate user
        user = auth_service.authenticate_user(db=db, email=credentials.email, password=credentials.password)

        if not user:
            raise HTTPException(status_code=status.HTTP_401_UNAUTHORIZED, detail="Invalid email or password")

        if not user.is_active:
            raise HTTPException(status_code=status.HTTP_403_FORBIDDEN, detail="Account is inactive")

        # Create tokens
        tokens = auth_service.create_tokens(user)

        logger.info(f"[AUTH] Successful login: {user.email}")
        logger.info(f"[ISSUE#398] Login response includes user info: {bool(user)}")

        return LoginResponse(
            user=UserResponse.from_orm(user),
            access_token=tokens["access_token"],
            refresh_token=tokens["refresh_token"],
            token_type=tokens["token_type"],
        )

    except HTTPException:
        raise
    except Exception as e:
        logger.error(f"Login error: {e}")
        raise HTTPException(status_code=status.HTTP_500_INTERNAL_SERVER_ERROR, detail="Login failed")


@router.post("/refresh", response_model=Token)
async def refresh_token(token_request: RefreshTokenRequest, db: Session = Depends(get_db)) -> Dict[str, Any]:
    """
    Refresh access token using refresh token
    """
    try:
        result = auth_service.refresh_access_token(db=db, refresh_token=token_request.refresh_token)

        logger.info("[AUTH] Token refreshed successfully")
        return result

    except ValueError as e:
        raise HTTPException(status_code=status.HTTP_401_UNAUTHORIZED, detail=str(e))
    except Exception as e:
        logger.error(f"Token refresh error: {e}")
        raise HTTPException(status_code=status.HTTP_500_INTERNAL_SERVER_ERROR, detail="Token refresh failed")


@router.get("/me", response_model=UserResponse)
async def get_current_user_profile(current_user: User = Depends(get_current_active_user)) -> User:
    """
    Get current authenticated user profile
    """
    return current_user


@router.put("/me", response_model=UserResponse)
@limiter.limit("5/minute")  # Rate limiting más restrictivo para cambio de contraseña
async def update_current_user(
    request: Request,  # Requerido para rate limiter y audit logging
    user_update: UserUpdate,
    current_user: User = Depends(get_current_active_user),
    db: Session = Depends(get_db),
) -> User:
    """
    Update current user profile

    Supports updating:
    - Basic info (full_name, phone, dni_nie)
    - Notification preferences
    - Password (requires current_password verification)
    """
    try:
        # Update basic user fields
        if user_update.full_name is not None:
            current_user.full_name = user_update.full_name
        if user_update.phone is not None:
            current_user.phone = user_update.phone
        if user_update.dni_nie is not None:
            current_user.dni_nie = user_update.dni_nie
        if user_update.notify_uploads is not None:
            current_user.notify_uploads = user_update.notify_uploads
        if user_update.notify_errors is not None:
            current_user.notify_errors = user_update.notify_errors
        if user_update.notify_analysis is not None:
            current_user.notify_analysis = user_update.notify_analysis

        # Handle password change (Issue: Ajustes usuario)
        if user_update.new_password is not None:
            # Verificar que se proporcionó la contraseña actual
            if not user_update.current_password:
                raise HTTPException(
                    status_code=status.HTTP_400_BAD_REQUEST,
                    detail="Se requiere la contraseña actual para cambiarla"
                )

            # Verificar que la contraseña actual es correcta
            if not verify_password(user_update.current_password, current_user.hashed_password):
                raise HTTPException(
                    status_code=status.HTTP_401_UNAUTHORIZED,
                    detail="La contraseña actual es incorrecta"
                )

            # Actualizar con nueva contraseña hasheada
            current_user.hashed_password = get_password_hash(user_update.new_password)
            logger.info(f"[AUTH] Password updated for user: {current_user.email}")

            # Audit logging para compliance (GDPR Article 30)
            audit_service = AuditService(db)
            audit_service.log_action(
                action=AuditAction.UPDATE,
                method="PUT",
                endpoint="/api/v1/auth/me",
                user=current_user,
                request=request,
                resource_type="user",
                resource_id=str(current_user.id),
                description=f"Usuario cambió su contraseña: {current_user.email}",
                details={"password_changed": True, "self_service": True},
            )

        db.commit()
        db.refresh(current_user)

        logger.info(f"[AUTH] Profile updated for user: {current_user.email}")
        return current_user

    except HTTPException:
        raise  # Re-raise HTTP exceptions as-is
    except Exception as e:
        logger.error(f"Profile update error: {e}")
        raise HTTPException(status_code=status.HTTP_500_INTERNAL_SERVER_ERROR, detail="Profile update failed")


@router.post("/verify-email")
async def verify_email(verification: EmailVerification, db: Session = Depends(get_db)) -> Dict[str, str]:
    """
    Verify user email with verification token
    """
    try:
        success = auth_service.verify_email(
            db=db, user_id=str(verification.user_id), token=verification.verification_token
        )

        if not success:
            raise HTTPException(status_code=status.HTTP_400_BAD_REQUEST, detail="Invalid verification token")

        return {"message": "Email verified successfully"}

    except HTTPException:
        raise
    except Exception as e:
        logger.error(f"Email verification error: {e}")
        raise HTTPException(status_code=status.HTTP_500_INTERNAL_SERVER_ERROR, detail="Verification failed")


@router.post("/forgot-password")
@limiter.limit("3/hour")  # 3 password reset requests per hour per IP
async def forgot_password(
    request: Request, reset_request: PasswordResetRequest, db: Session = Depends(get_db)
) -> Dict[str, str]:
    """
    Initiate password reset process

    NOTE: Email service is temporarily disabled. Password reset functionality
    will be available once email service is configured.
    """
    # Password reset is temporarily disabled until email service is configured
    logger.warning(f"[AUTH] Password reset requested for {reset_request.email} but email service is disabled")

    # Return standard message for security (don't reveal if email exists)
    return {"message": "Password reset functionality is temporarily unavailable. Please contact your administrator."}


@router.post("/reset-password")
async def reset_password(reset_data: PasswordReset, db: Session = Depends(get_db)) -> Dict[str, str]:
    """
    Reset password with reset token

    NOTE: This functionality is temporarily disabled until email service is configured.
    """
    # Password reset is temporarily disabled
    logger.warning("[AUTH] Password reset attempted but functionality is disabled")

    raise HTTPException(
        status_code=status.HTTP_503_SERVICE_UNAVAILABLE,
        detail="Password reset functionality is temporarily unavailable. Please contact your administrator.",
    )


# OAuth endpoints
@router.get("/oauth/providers", response_model=OAuthProviders)
async def get_oauth_providers() -> Dict[str, Any]:
    """
    Get available OAuth providers and their login URLs
    """
    providers = OAuthProviders(
        google_enabled=bool(GOOGLE_CLIENT_ID),
        microsoft_enabled=bool(MICROSOFT_CLIENT_ID),
        google_url=f"{BASE_URL}/auth/oauth/google" if GOOGLE_CLIENT_ID else None,
        microsoft_url=f"{BASE_URL}/auth/oauth/microsoft" if MICROSOFT_CLIENT_ID else None,
    )

    return providers


@router.get("/oauth/google")
async def oauth_google_login(request: Request):
    """
    Initiate Google OAuth login
    """
    if not GOOGLE_CLIENT_ID:
        raise HTTPException(status_code=status.HTTP_503_SERVICE_UNAVAILABLE, detail="Google OAuth not configured")

    redirect_uri = f"{BASE_URL}/auth/oauth/google/callback"
    return await oauth.google.authorize_redirect(request, redirect_uri)


@router.get("/oauth/google/callback")
async def oauth_google_callback(request: Request, db: Session = Depends(get_db)):
    """
    Handle Google OAuth callback
    """
    try:
        # Get token from Google
        token = await oauth.google.authorize_access_token(request)

        # Get user info from Google
        user_info = token.get("userinfo")

        if not user_info:
            # Fetch user info if not in token
            resp = await oauth.google.get("https://www.googleapis.com/oauth2/v2/userinfo", token=token)
            user_info = resp.json()

        # Create or get user
        user = auth_service.create_user_from_oauth(
            db=db,
            email=user_info["email"],
            full_name=user_info.get("name", ""),
            provider="google",
            provider_user_id=user_info["id"],
        )

        # Create tokens
        tokens = auth_service.create_tokens(user)

        # Store tokens securely and get state key (Issue #398: Include user info)
        state_key = auth_service.store_oauth_tokens(tokens, user=user)

        logger.info(f"[ISSUE#398] Google OAuth callback stored user info: {bool(user)}")

        # Redirect to frontend with state key only (secure)
        redirect_url = f"{FRONTEND_URL}/auth/callback?state={state_key}&provider=google"

        return RedirectResponse(url=redirect_url)

    except OAuthError as e:
        logger.error(f"Google OAuth error: {e}")
        redirect_url = f"{FRONTEND_URL}/auth/error?message=OAuth+failed"
        return RedirectResponse(url=redirect_url)


@router.get("/oauth/microsoft")
async def oauth_microsoft_login(request: Request):
    """
    Initiate Microsoft OAuth login
    """
    if not MICROSOFT_CLIENT_ID:
        raise HTTPException(status_code=status.HTTP_503_SERVICE_UNAVAILABLE, detail="Microsoft OAuth not configured")

    redirect_uri = f"{BASE_URL}/auth/oauth/microsoft/callback"
    return await oauth.microsoft.authorize_redirect(request, redirect_uri)


@router.get("/oauth/microsoft/callback")
async def oauth_microsoft_callback(request: Request, db: Session = Depends(get_db)):
    """
    Handle Microsoft OAuth callback
    """
    try:
        # Get token from Microsoft
        token = await oauth.microsoft.authorize_access_token(request)

        # Get user info from Microsoft
        # Microsoft Graph API endpoint
        resp = await oauth.microsoft.get("https://graph.microsoft.com/v1.0/me", token=token)
        user_info = resp.json()

        # Create or get user
        user = auth_service.create_user_from_oauth(
            db=db,
            email=user_info["mail"] or user_info["userPrincipalName"],
            full_name=user_info.get("displayName", ""),
            provider="microsoft",
            provider_user_id=user_info["id"],
        )

        # Create tokens
        tokens = auth_service.create_tokens(user)

        # Store tokens securely and get state key (Issue #398: Include user info)
        state_key = auth_service.store_oauth_tokens(tokens, user=user)

        logger.info(f"[ISSUE#398] Microsoft OAuth callback stored user info: {bool(user)}")

        # Redirect to frontend with state key only (secure)
        redirect_url = f"{FRONTEND_URL}/auth/callback?state={state_key}&provider=microsoft"

        return RedirectResponse(url=redirect_url)

    except OAuthError as e:
        logger.error(f"Microsoft OAuth error: {e}")
        redirect_url = f"{FRONTEND_URL}/auth/error?message=OAuth+failed"
        return RedirectResponse(url=redirect_url)


@router.post("/oauth/exchange", response_model=LoginResponse)
async def exchange_oauth_state(state_key: str) -> LoginResponse:
    """
    Exchange OAuth state key for actual tokens

    This endpoint is used by the frontend after OAuth redirect
    to securely retrieve the tokens without exposing them in URL

    Returns LoginResponse with user info (Issue #398)
    """
    try:
        result = auth_service.retrieve_oauth_tokens(state_key)

        if not result:
            raise HTTPException(status_code=status.HTTP_400_BAD_REQUEST, detail="Invalid or expired state key")

        tokens = result.get("tokens", {})
        user = result.get("user")

        # Issue #398: Fail fast if user missing (indicates backend bug)
        if not user:
            logger.error("[ISSUE#398] OAuth exchange missing user - possible data corruption")
            raise HTTPException(
                status_code=status.HTTP_500_INTERNAL_SERVER_ERROR, detail="User information missing from OAuth state"
            )

        logger.info("[AUTH] OAuth tokens exchanged successfully")
        logger.info(f"[ISSUE#398] OAuth exchange includes user info: {bool(user)}")

        # Return LoginResponse with user info (Issue #398)
        return LoginResponse(
            user=UserResponse.from_orm(user),  # Guaranteed non-None
            access_token=tokens["access_token"],  # Fail fast if missing
            refresh_token=tokens["refresh_token"],  # Fail fast if missing
            token_type=tokens.get("token_type", "bearer"),
        )

    except HTTPException:
        raise
    except Exception as e:
        logger.error(f"OAuth exchange error: {e}")
        raise HTTPException(status_code=status.HTTP_500_INTERNAL_SERVER_ERROR, detail="Token exchange failed")


@router.post("/logout")
async def logout() -> Dict[str, str]:
    """
    Logout user (client should discard tokens)
    """
    # In a stateless JWT system, logout is handled client-side
    # The client should remove tokens from storage
    # For enhanced security, you could implement token blacklisting

    logger.info("[AUTH] User logged out")

    return {"message": "Logout successful"}


# =============================================================================
# SUBSCRIPTION & STORAGE ENDPOINTS (Issue #420: FREE tier experience)
# =============================================================================

@router.get("/storage-usage", response_model=UserStorageUsageResponse)
async def get_storage_usage(
    current_user: User = Depends(get_current_active_user),
    db: Session = Depends(get_db),
) -> Dict[str, Any]:
    """
    Get storage usage for current authenticated user.

    Issue #420: FREE tier experience - Subscription page.

    Returns:
        - total_used_mb: Total MB used by file uploads
        - limit_mb: Storage limit based on subscription plan
        - percentage: Usage percentage (0-100)
        - plan: Current subscription plan
        - warning_threshold: Threshold percentage for warning (default 80)
    """
    from sqlalchemy import func

    from app.core.subscription_limits import get_storage_limit_mb
    from app.models import FileUpload

    try:
        # Calculate bytes used (FileUpload files for user's pharmacy)
        # Note: FileUpload model doesn't have soft delete - all records are active
        used_bytes = (
            db.query(func.coalesce(func.sum(FileUpload.file_size), 0))
            .filter(FileUpload.pharmacy_id == current_user.pharmacy_id)
            .scalar()
        ) or 0

        used_mb = used_bytes / (1024 * 1024)
        limit_mb = get_storage_limit_mb(current_user.subscription_plan)
        percentage = round((used_mb / limit_mb) * 100, 1) if limit_mb > 0 else 0

        logger.debug(
            f"[STORAGE] User {current_user.email}: {used_mb:.2f} MB / {limit_mb} MB ({percentage}%)"
        )

        return {
            "total_used_mb": round(used_mb, 2),
            "limit_mb": limit_mb,
            "percentage": min(percentage, 100),  # Cap at 100%
            "plan": current_user.subscription_plan,
            "warning_threshold": 80,
        }

    except Exception as e:
        logger.error(f"[STORAGE] Error getting storage usage: {e}")
        raise HTTPException(
            status_code=status.HTTP_500_INTERNAL_SERVER_ERROR,
            detail="Error al obtener uso de almacenamiento",
        )
