# backend/app/api/v1/endpoints/admin_users.py
"""
API endpoints para administración de usuarios.

Issue #348 FASE 2: Backend - Services & APIs
Incluye RBAC, rate limiting, y audit logging.
"""
import logging
from typing import Optional
from uuid import UUID

from fastapi import APIRouter, Depends, HTTPException, status, Query, Request
from sqlalchemy.orm import Session
from slowapi import Limiter
from slowapi.util import get_remote_address

from app.api.deps import get_db, get_current_user
from app.models.user import User
from app.core.subscription_limits import Permission
from app.core.security import require_permissions
from app.services.user_pharmacy_service import UserPharmacyService
from app.services.audit_service import AuditService, AuditAction
from app.schemas.user_pharmacy import (
    UserWithPharmacyCreate,
    UserResponse,
    UserUpdate,
    UserListResponse,
    StorageUsageResponse
)

logger = logging.getLogger(__name__)
router = APIRouter()

# Rate limiter para operaciones destructivas
limiter = Limiter(key_func=get_remote_address)


@router.post(
    "/users",
    response_model=UserResponse,
    status_code=status.HTTP_201_CREATED,
    dependencies=[Depends(require_permissions(Permission.MANAGE_USERS.value))]
)
@limiter.limit("10/minute")
async def create_user_with_pharmacy(
    request: Request,
    user_data: UserWithPharmacyCreate,
    db: Session = Depends(get_db),
    current_user: User = Depends(get_current_user)
):
    """
    Crear un nuevo usuario con su farmacia.

    **Permisos requeridos:** MANAGE_USERS

    **Rate limit:** 10 requests por minuto

    **REGLA #10:** Relación 1:1 User-Pharmacy obligatoria

    Args:
        user_data: Datos del usuario y farmacia a crear

    Returns:
        Usuario creado con información de la farmacia

    Raises:
        400: Si el email/username ya existe o la farmacia ya tiene usuario
        403: Sin permisos MANAGE_USERS
        429: Rate limit excedido
    """
    try:
        # Crear usuario con farmacia (transacción atómica)
        user, pharmacy = UserPharmacyService.create_user_with_pharmacy(
            db=db,
            user_data=user_data.user,
            pharmacy_data=user_data.pharmacy,
            role=user_data.role.value,
            subscription_plan=user_data.subscription_plan.value
        )

        # Audit logging
        audit_service = AuditService(db)
        audit_service.log_action(
            action=AuditAction.CREATE,
            method="POST",
            endpoint="/api/v1/admin/users",
            user=current_user,
            request=request,
            resource_type="user",
            resource_id=str(user.id),
            description=f"Usuario creado: {user.email}",
            details={
                "email": user.email,
                "pharmacy_name": pharmacy.name,
                "pharmacy_nif": pharmacy.nif,
                "role": user.role,
                "subscription_plan": user.subscription_plan
            }
        )

        # Preparar respuesta con farmacia incluida
        user.pharmacy = pharmacy
        return user

    except HTTPException:
        raise  # Re-raise HTTPExceptions preserving their status codes
    except ValueError as e:
        logger.error(f"Error al crear usuario: {str(e)}")
        raise HTTPException(
            status_code=status.HTTP_400_BAD_REQUEST,
            detail=str(e)
        )
    except Exception as e:
        logger.error(f"Error inesperado al crear usuario: {str(e)}")
        raise HTTPException(
            status_code=status.HTTP_500_INTERNAL_SERVER_ERROR,
            detail="Error al crear usuario y farmacia"
        )


@router.get(
    "/users",
    response_model=UserListResponse,
    dependencies=[Depends(require_permissions(Permission.VIEW_USERS.value))]
)
async def list_users(
    skip: int = Query(0, ge=0, description="Número de registros a saltar"),
    limit: int = Query(100, ge=1, le=500, description="Máximo de registros a devolver"),
    role: Optional[str] = Query(None, description="Filtrar por rol (user/admin)"),
    subscription: Optional[str] = Query(None, description="Filtrar por plan de suscripción"),
    include_deleted: bool = Query(False, description="Incluir usuarios eliminados"),
    db: Session = Depends(get_db),
    current_user: User = Depends(get_current_user)
):
    """
    Listar usuarios con filtros y paginación.

    **Permisos requeridos:** VIEW_USERS

    Args:
        skip: Offset para paginación
        limit: Límite de resultados
        role: Filtrar por rol
        subscription: Filtrar por plan
        include_deleted: Incluir soft deleted

    Returns:
        Lista paginada de usuarios con total
    """
    try:
        users, total = UserPharmacyService.get_users_paginated(
            db=db,
            skip=skip,
            limit=limit,
            role_filter=role,
            subscription_filter=subscription,
            include_deleted=include_deleted
        )

        return UserListResponse(
            users=users,
            total=total,
            skip=skip,
            limit=limit
        )

    except HTTPException:
        raise  # Re-raise HTTPExceptions preserving their status codes
    except Exception as e:
        logger.error(f"Error al listar usuarios: {str(e)}")
        raise HTTPException(
            status_code=status.HTTP_500_INTERNAL_SERVER_ERROR,
            detail="Error al obtener lista de usuarios"
        )


@router.get(
    "/users/{user_id}",
    response_model=UserResponse,
    dependencies=[Depends(require_permissions(Permission.VIEW_USERS.value))]
)
async def get_user(
    user_id: UUID,
    db: Session = Depends(get_db),
    current_user: User = Depends(get_current_user)
):
    """
    Obtener información de un usuario específico.

    **Permisos requeridos:** VIEW_USERS

    Args:
        user_id: ID del usuario

    Returns:
        Información del usuario con su farmacia

    Raises:
        404: Usuario no encontrado
    """
    from sqlalchemy import select
    from sqlalchemy.orm import selectinload

    result = db.execute(
        select(User)
        .options(selectinload(User.pharmacy))
        .where(User.id == user_id)
    )
    user = result.scalar_one_or_none()

    if not user:
        raise HTTPException(
            status_code=status.HTTP_404_NOT_FOUND,
            detail=f"Usuario con ID {user_id} no encontrado"
        )

    return user


@router.put(
    "/users/{user_id}",
    response_model=UserResponse,
    dependencies=[Depends(require_permissions(Permission.MANAGE_USERS.value))]
)
@limiter.limit("10/minute")
async def update_user(
    request: Request,
    user_id: UUID,
    user_update: UserUpdate,
    db: Session = Depends(get_db),
    current_user: User = Depends(get_current_user)
):
    """
    Actualizar un usuario.

    **Permisos requerido:** MANAGE_USERS

    **Rate limit:** 10 requests por minuto

    Args:
        user_id: ID del usuario
        user_update: Campos a actualizar

    Returns:
        Usuario actualizado

    Raises:
        400: Usuario eliminado o datos inválidos
        404: Usuario no encontrado
        403: Sin permisos MANAGE_USERS
    """
    logger.info(f"[UPDATE_USER] Updating user {user_id} with data: {user_update.model_dump(exclude_unset=True)}")
    try:
        # No permitir que un usuario se modifique a sí mismo el rol (solo si está cambiando)
        if user_id == current_user.id and user_update.role:
            # Obtener el rol actual del usuario para comparar (Session síncrona, NO usar await)
            from sqlalchemy import select
            result = db.execute(
                select(User).where(User.id == user_id)
            )
            existing_user = result.scalar_one_or_none()

            if existing_user:
                # Convertir enum a string si es necesario
                new_role = user_update.role.value if hasattr(user_update.role, 'value') else user_update.role

                # Solo bloquear si el rol es DIFERENTE al actual
                if new_role != existing_user.role:
                    raise HTTPException(
                        status_code=status.HTTP_400_BAD_REQUEST,
                        detail="No puedes cambiar tu propio rol"
                    )

        user = UserPharmacyService.update_user(
            db=db,
            user_id=user_id,
            user_update=user_update
        )

        # Audit logging
        audit_service = AuditService(db)
        audit_service.log_action(
            action=AuditAction.UPDATE,
            method="PUT",
            endpoint=f"/api/v1/admin/users/{user_id}",
            user=current_user,
            request=request,
            resource_type="user",
            resource_id=str(user_id),
            description=f"Usuario actualizado: {user.email}",
            details=user_update.model_dump(exclude_unset=True)
        )

        return user

    except HTTPException:
        raise  # Re-raise HTTPExceptions preserving their status codes
    except ValueError as e:
        logger.error(f"Error al actualizar usuario: {str(e)}")
        raise HTTPException(
            status_code=status.HTTP_400_BAD_REQUEST,
            detail=str(e)
        )
    except Exception as e:
        logger.error(f"Error inesperado al actualizar usuario: {str(e)}")
        raise HTTPException(
            status_code=status.HTTP_500_INTERNAL_SERVER_ERROR,
            detail="Error al actualizar usuario"
        )


@router.delete(
    "/users/{user_id}",
    status_code=status.HTTP_204_NO_CONTENT,
    dependencies=[Depends(require_permissions(Permission.MANAGE_USERS.value))]
)
@limiter.limit("10/minute")
async def soft_delete_user(
    request: Request,
    user_id: UUID,
    db: Session = Depends(get_db),
    current_user: User = Depends(get_current_user)
):
    """
    Soft delete de un usuario (GDPR Article 17).

    **Permisos requeridos:** MANAGE_USERS

    **Rate limit:** 10 requests por minuto

    Marca el usuario como eliminado pero mantiene datos para auditoría.

    Args:
        user_id: ID del usuario a eliminar

    Raises:
        400: Usuario ya eliminado o es el usuario actual
        404: Usuario no encontrado
        403: Sin permisos MANAGE_USERS
    """
    try:
        # No permitir auto-eliminación
        if user_id == current_user.id:
            raise HTTPException(
                status_code=status.HTTP_400_BAD_REQUEST,
                detail="No puedes eliminar tu propia cuenta"
            )

        UserPharmacyService.soft_delete_user(
            db=db,
            user_id=user_id
        )

        # Audit logging
        audit_service = AuditService(db)
        audit_service.log_action(
            action=AuditAction.DELETE,
            method="DELETE",
            endpoint=f"/api/v1/admin/users/{user_id}",
            user=current_user,
            request=request,
            resource_type="user",
            resource_id=str(user_id),
            description="Usuario eliminado (soft delete)",
            details={"soft_delete": True, "gdpr_compliance": True}
        )

        return None  # 204 No Content

    except HTTPException:
        raise  # Re-raise HTTPExceptions preserving their status codes
    except ValueError as e:
        logger.error(f"Error al eliminar usuario: {str(e)}")
        raise HTTPException(
            status_code=status.HTTP_400_BAD_REQUEST,
            detail=str(e)
        )
    except Exception as e:
        logger.error(f"Error inesperado al eliminar usuario: {str(e)}")
        raise HTTPException(
            status_code=status.HTTP_500_INTERNAL_SERVER_ERROR,
            detail="Error al eliminar usuario"
        )


@router.post(
    "/users/{user_id}/restore",
    response_model=UserResponse,
    dependencies=[Depends(require_permissions(Permission.MANAGE_USERS.value))]
)
@limiter.limit("10/minute")
async def restore_user(
    request: Request,
    user_id: UUID,
    db: Session = Depends(get_db),
    current_user: User = Depends(get_current_user)
):
    """
    Restaurar un usuario con soft delete.

    **Permisos requeridos:** MANAGE_USERS

    **Rate limit:** 10 requests por minuto

    Args:
        user_id: ID del usuario a restaurar

    Returns:
        Usuario restaurado

    Raises:
        400: Usuario no está eliminado
        404: Usuario no encontrado
        403: Sin permisos MANAGE_USERS
    """
    try:
        user = UserPharmacyService.restore_user(
            db=db,
            user_id=user_id
        )

        # Audit logging
        await AuditService.log_action(
            db=db,
            user_id=current_user.id,
            action="USER_RESTORED",
            entity_type="user",
            entity_id=str(user_id),
            details={"restored_from_soft_delete": True}
        )

        return user

    except HTTPException:
        raise  # Re-raise HTTPExceptions preserving their status codes
    except ValueError as e:
        logger.error(f"Error al restaurar usuario: {str(e)}")
        raise HTTPException(
            status_code=status.HTTP_400_BAD_REQUEST,
            detail=str(e)
        )
    except Exception as e:
        logger.error(f"Error inesperado al restaurar usuario: {str(e)}")
        raise HTTPException(
            status_code=status.HTTP_500_INTERNAL_SERVER_ERROR,
            detail="Error al restaurar usuario"
        )


@router.get(
    "/storage/usage",
    response_model=list[StorageUsageResponse],
    dependencies=[Depends(require_permissions(Permission.VIEW_SYSTEM_STATS.value))]
)
async def get_storage_usage(
    pharmacy_id: Optional[UUID] = Query(None, description="ID de farmacia específica"),
    db: Session = Depends(get_db),
    current_user: User = Depends(get_current_user)
):
    """
    Obtener estadísticas de uso de almacenamiento.

    **Permisos requeridos:** VIEW_SYSTEM_STATS

    Consulta la vista materializada `pharmacy_storage_stats`.

    Args:
        pharmacy_id: Filtrar por farmacia específica

    Returns:
        Lista de estadísticas de uso por farmacia
    """
    try:
        storage_stats = UserPharmacyService.get_storage_usage(
            db=db,
            pharmacy_id=pharmacy_id
        )

        return storage_stats

    except HTTPException:
        raise  # Re-raise HTTPExceptions preserving their status codes
    except Exception as e:
        logger.error(f"Error al obtener uso de almacenamiento: {str(e)}")
        raise HTTPException(
            status_code=status.HTTP_500_INTERNAL_SERVER_ERROR,
            detail="Error al obtener estadísticas de almacenamiento"
        )
