﻿"""
Endpoints para herramientas administrativas consolidadas.

Estos endpoints centralizan las operaciones peligrosas y de mantenimiento
que anteriormente estaban dispersas entre diferentes secciones.
"""

import logging
import os
from typing import Any, Dict

from fastapi import APIRouter, BackgroundTasks, Depends, HTTPException, Request
from sqlalchemy import text
from sqlalchemy.orm import Session

from app.api.deps import get_current_active_user, get_current_admin_user
from app.database import get_db
from app.models import FileUpload, ProductCatalog, SalesData
from app.models.user import User
from app.services.catalog_maintenance_service import CatalogMaintenanceService
from app.utils.datetime_utils import utc_now
from app.utils.safety_guards import AuditLogger

# Response models are handled as plain dicts

router = APIRouter(prefix="/api/v1/admin-tools", tags=["admin-tools"])
logger = logging.getLogger(__name__)


@router.post("/maintenance/clear-cache")
async def clear_system_cache(
    current_user: User = Depends(get_current_admin_user),  # SECURITY: Admin required (Issue #452)
) -> Dict[str, Any]:
    """
    Limpia todas las cachés del sistema.

    Operación segura que limpia cachés en memoria y Redis.
    Requires: Admin authentication
    """
    try:
        # TODO: Implementar limpieza de caché Redis
        # await redis_client.flushdb()

        return {
            "success": True,
            "message": "Caché del sistema limpiada exitosamente",
            "data": {"cleared_at": utc_now().isoformat()},
        }

    except Exception as e:
        logger.error(f"Error limpiando caché: {e}")
        raise HTTPException(status_code=500, detail=f"Error limpiando caché: {str(e)}")


@router.post("/maintenance/verify-connectivity")
async def verify_system_connectivity(
    current_user: User = Depends(get_current_admin_user),  # SECURITY: Admin required (Issue #452)
    db: Session = Depends(get_db),
) -> Dict[str, Any]:
    """
    Verifica la conectividad de todos los sistemas críticos.

    Prueba conexiones a base de datos, APIs externas y servicios.
    Requires: Admin authentication
    """
    try:
        connectivity_results = {}

        # Test de base de datos
        try:
            db.execute(text("SELECT 1")).fetchone()
            connectivity_results["database"] = {
                "status": "operational",
                "response_time_ms": 25.4,  # Simulado
                "last_check": utc_now().isoformat(),
            }
        except Exception as e:
            connectivity_results["database"] = {
                "status": "error",
                "error": str(e),
                "last_check": utc_now().isoformat(),
            }

        # Test de CIMA API (simulado)
        connectivity_results["cima_api"] = {
            "status": "operational",
            "response_time_ms": 156.7,
            "last_check": utc_now().isoformat(),
        }

        # Test de Nomenclator (simulado)
        connectivity_results["nomenclator"] = {
            "status": "operational",
            "data_accessible": True,
            "last_check": utc_now().isoformat(),
        }

        # Test de Redis (simulado)
        connectivity_results["redis"] = {
            "status": "operational",
            "response_time_ms": 3.2,
            "last_check": utc_now().isoformat(),
        }

        return {
            "success": True,
            "message": "Verificación de conectividad completada",
            "data": connectivity_results,
        }

    except Exception as e:
        logger.error(f"Error verificando conectividad: {e}")
        raise HTTPException(status_code=500, detail=f"Error verificando conectividad: {str(e)}")


@router.get("/dangerous-operations")
async def list_dangerous_operations() -> Dict[str, Any]:
    """
    Lista todas las operaciones peligrosas disponibles.

    Estas operaciones requieren confirmación especial y pueden afectar datos.
    """
    dangerous_operations = [
        {
            "id": "delete_all_sales_data",
            "name": "Eliminar todos los datos de ventas",
            "description": "Elimina TODOS los registros de ventas de TODAS las farmacias",
            "risk_level": "CRITICAL",
            "confirmation_required": "ELIMINAR VENTAS",
            "estimated_time": "30 segundos",
            "reversible": False,
        },
        {
            "id": "clean_catalog_full",
            "name": "Limpiar catálogo completo",
            "description": "Elimina todo el catálogo y fuerza re-sincronización",
            "risk_level": "HIGH",
            "confirmation_required": "LIMPIAR CATALOGO",
            "estimated_time": "2-3 horas",
            "reversible": False,
        },
        {
            "id": "reset_system_configuration",
            "name": "Resetear configuración del sistema",
            "description": "Restaura todas las configuraciones a valores por defecto",
            "risk_level": "MEDIUM",
            "confirmation_required": "RESET CONFIG",
            "estimated_time": "5 minutos",
            "reversible": True,
        },
        {
            "id": "vacuum_database",
            "name": "VACUUM completo de base de datos",
            "description": "Optimiza y reorganiza toda la base de datos",
            "risk_level": "LOW",
            "confirmation_required": "VACUUM DB",
            "estimated_time": "10-15 minutos",
            "reversible": True,
        },
    ]

    return {
        "success": True,
        "message": "Operaciones peligrosas listadas",
        "data": {"operations": dangerous_operations},
    }


@router.post("/dangerous-operations/{operation_id}/execute")
async def execute_dangerous_operation(
    operation_id: str,
    confirmation_text: str,
    background_tasks: BackgroundTasks,
    request: Request,
    current_user: User = Depends(get_current_active_user),
    db: Session = Depends(get_db),
) -> Dict[str, Any]:
    """
    Ejecuta una operación peligrosa después de verificar confirmación.

    SECURITY:
        - Layer 1: JWT authentication via get_current_active_user (CRITICAL)
        - Layer 2: RBAC - Admin role required (CRITICAL)
        - Layer 3: Pharmacy scope - NO (operations are system-wide by design)
        - Layer 4: Audit logging with user_id, IP, timestamp (CRITICAL)

    Security Fix:
        Vulnerability discovered during review of commit 0f1dacb.
        Endpoint was PUBLIC without authentication, allowing ANYONE to:
        - Delete ALL sales data from ALL pharmacies
        - Wipe entire product catalog (67k products)
        - Corrupt system configuration
        - Lock database with VACUUM operations

        Fix implements same security pattern as /api/v1/upload/cleanup/zombies.

    Args:
        operation_id: ID de la operación a ejecutar
        confirmation_text: Texto de confirmación requerido
        request: FastAPI request object (for IP address logging)
        current_user: Usuario autenticado (JWT token validated)
        db: Database session

    Raises:
        HTTPException 401: If user not authenticated (JWT missing/invalid)
        HTTPException 403: If user not admin role
        HTTPException 404: If operation_id not found
        HTTPException 400: If confirmation text incorrect

    Returns:
        Dict con status de la operación iniciada en background
    """
    try:
        # ===================================================================
        # SECURITY LAYER 2: RBAC - Verify admin role
        # ===================================================================
        if current_user.role != "admin":
            logger.warning(
                "SECURITY: Non-admin user attempted dangerous operation",
                extra={
                    "user_id": str(current_user.id),
                    "user_email": current_user.email,
                    "user_role": current_user.role,
                    "operation_id": operation_id,
                    "security_event": "unauthorized_dangerous_operation_attempt",
                },
            )
            raise HTTPException(
                status_code=403, detail="Admin role required for dangerous operations"
            )
        # Mapeo de operaciones y sus textos de confirmación
        operation_confirmations = {
            "delete_all_sales_data": "ELIMINAR VENTAS",
            "clean_catalog_full": "LIMPIAR CATALOGO",
            "reset_system_configuration": "RESET CONFIG",
            "vacuum_database": "VACUUM DB",
        }

        # Verificar que la operación existe
        if operation_id not in operation_confirmations:
            raise HTTPException(status_code=404, detail=f"Operación '{operation_id}' no encontrada")

        # Verificar texto de confirmación
        required_confirmation = operation_confirmations[operation_id]
        if confirmation_text.strip().upper() != required_confirmation:
            logger.warning(
                "Incorrect confirmation text for dangerous operation",
                extra={
                    "user_id": str(current_user.id),
                    "operation_id": operation_id,
                    "expected": required_confirmation,
                    "received": confirmation_text.strip().upper(),
                },
            )
            raise HTTPException(
                status_code=400,
                detail=f"Texto de confirmación incorrecto. Se requiere: '{required_confirmation}'",
            )

        # ===================================================================
        # SECURITY LAYER 4: AUDIT LOGGING
        # ===================================================================
        client_ip = request.client.host if request.client else "unknown"
        user_agent = request.headers.get("user-agent", "unknown")

        logger.warning(
            f"DANGEROUS OPERATION INITIATED: {operation_id} by {current_user.email}",
            extra={
                "user_id": str(current_user.id),
                "user_email": current_user.email,
                "user_role": current_user.role,
                "operation_id": operation_id,
                "confirmation_text": confirmation_text,
                "ip_address": client_ip,
                "user_agent": user_agent,
                "timestamp": utc_now().isoformat(),
                "security_event": "dangerous_operation_executed",
            },
        )

        # Audit logging via AuditLogger
        AuditLogger.log_operation_attempt(
            db=db,
            operation_id=operation_id,
            user_id=str(current_user.id),
            environment="production" if os.getenv("ENVIRONMENT") == "production" else "development",
            confirmation_provided=confirmation_text,
            success=True,
            additional_context={
                "user_email": current_user.email,
                "ip_address": client_ip,
                "user_agent": user_agent,
            },
        )

        # Ejecutar la operación en background
        if operation_id == "delete_all_sales_data":
            background_tasks.add_task(_delete_all_sales_data, db)
        elif operation_id == "clean_catalog_full":
            background_tasks.add_task(_clean_catalog_full, db)
        elif operation_id == "reset_system_configuration":
            background_tasks.add_task(_reset_system_configuration, db)
        elif operation_id == "vacuum_database":
            background_tasks.add_task(_vacuum_database, db)

        return {
            "success": True,
            "message": f"Operación '{operation_id}' iniciada en segundo plano",
            "data": {
                "operation_id": operation_id,
                "started_at": utc_now().isoformat(),
                "status": "in_progress",
                "initiated_by": current_user.email,
            },
        }

    except HTTPException:
        raise
    except Exception as e:
        # Log error with user context for forensics
        logger.error(
            f"Error ejecutando operación peligrosa {operation_id}: {e}",
            extra={
                "user_id": str(current_user.id),
                "user_email": current_user.email,
                "operation_id": operation_id,
                "error": str(e),
            },
        )
        raise HTTPException(status_code=500, detail=f"Error ejecutando operación: {str(e)}")


@router.get("/system-analysis")
async def get_system_analysis(
    analysis_type: str = "full",  # full, performance, data, security
    db: Session = Depends(get_db),
) -> Dict[str, Any]:
    """
    Realiza análisis del sistema para identificar problemas.

    Args:
        analysis_type: Tipo de análisis a realizar
    """
    try:
        if analysis_type == "full":
            analysis_result = await _run_full_system_analysis(db)
        elif analysis_type == "performance":
            analysis_result = await _run_performance_analysis(db)
        elif analysis_type == "data":
            analysis_result = await _run_data_analysis(db)
        elif analysis_type == "security":
            analysis_result = await _run_security_analysis(db)
        else:
            raise HTTPException(
                status_code=400,
                detail="Tipo de análisis debe ser: full, performance, data, o security",
            )

        return {
            "success": True,
            "message": f"Análisis del sistema ({analysis_type}) completado",
            "data": analysis_result,
        }

    except Exception as e:
        logger.error(f"Error en análisis del sistema: {e}")
        raise HTTPException(status_code=500, detail=f"Error en análisis: {str(e)}")


# Funciones de background para operaciones peligrosas


async def _delete_all_sales_data(db: Session):
    """
    Elimina todos los datos de ventas y sus enriquecimientos.

    Issue #330: Orden correcto para respetar foreign key constraints:
    1. sales_enrichment (depende de sales_data)
    2. sales_data
    """
    try:
        logger.warning("INICIANDO: Eliminación de todos los datos de ventas")

        # Paso 1: Eliminar enriquecimientos primero (foreign key a sales_data)
        from app.models.sales_enrichment import SalesEnrichment
        enrichment_count = db.query(SalesEnrichment).count()
        db.query(SalesEnrichment).delete()
        logger.warning(f"Eliminados {enrichment_count} registros de enriquecimiento")

        # Paso 2: Eliminar datos de ventas
        sales_count = db.query(SalesData).count()
        db.query(SalesData).delete()
        db.commit()

        logger.warning(f"COMPLETADO: Eliminados {sales_count} registros de ventas y {enrichment_count} enriquecimientos")

    except Exception as e:
        logger.error(f"Error eliminando datos de ventas: {e}")
        db.rollback()
        raise  # Re-raise para que el endpoint retorne error


async def _clean_catalog_full(db: Session):
    """Limpia completamente el catálogo"""
    try:
        logger.warning("INICIANDO: Limpieza completa del catálogo")

        # Eliminar todo el catálogo
        deleted_count = db.query(ProductCatalog).count()
        db.query(ProductCatalog).delete()
        db.commit()

        logger.warning(f"COMPLETADO: Eliminados {deleted_count} productos del catálogo")

        # Trigger re-sincronización
        maintenance_service = CatalogMaintenanceService(db)
        await maintenance_service.force_full_sync()

    except Exception as e:
        logger.error(f"Error limpiando catálogo: {e}")
        db.rollback()


async def _reset_system_configuration(db: Session):
    """Resetea la configuración del sistema"""
    try:
        logger.warning("INICIANDO: Reset de configuración del sistema")

        # TODO: Implementar reset de configuraciones
        # Por ahora solo log

        logger.warning("COMPLETADO: Configuración del sistema reseteada")

    except Exception as e:
        logger.error(f"Error reseteando configuración: {e}")


async def _vacuum_database(db: Session):
    """Realiza VACUUM completo de la base de datos"""
    try:
        logger.info("INICIANDO: VACUUM completo de base de datos")

        # Ejecutar VACUUM
        db.execute(text("VACUUM ANALYZE"))
        db.commit()

        logger.info("COMPLETADO: VACUUM de base de datos")

    except Exception as e:
        logger.error(f"Error en VACUUM de BD: {e}")


# Funciones de análisis del sistema


async def _run_full_system_analysis(db: Session) -> Dict[str, Any]:
    """Análisis completo del sistema"""
    performance = await _run_performance_analysis(db)
    data = await _run_data_analysis(db)
    security = await _run_security_analysis(db)

    return {
        "analysis_type": "full",
        "timestamp": utc_now().isoformat(),
        "performance": performance,
        "data": data,
        "security": security,
        "overall_score": 85,  # Simulado
    }


async def _run_performance_analysis(db: Session) -> Dict[str, Any]:
    """Análisis de rendimiento"""
    return {
        "database_performance": "good",
        "query_optimization_needed": False,
        "index_recommendations": [],
        "memory_usage": "normal",
        "score": 90,
    }


async def _run_data_analysis(db: Session) -> Dict[str, Any]:
    """Análisis de datos"""
    try:
        # Contar registros
        catalog_count = db.query(ProductCatalog).count()
        sales_count = db.query(SalesData).count()
        uploads_count = db.query(FileUpload).count()

        return {
            "catalog_integrity": "good",
            "data_consistency": "good",
            "orphaned_records": 0,
            "duplicate_detection": "clean",
            "statistics": {
                "catalog_products": catalog_count,
                "sales_records": sales_count,
                "file_uploads": uploads_count,
            },
            "score": 88,
        }

    except Exception as e:
        return {"error": str(e), "score": 0}


async def _run_security_analysis(db: Session) -> Dict[str, Any]:
    """Análisis de seguridad"""
    return {
        "authentication": "configured",
        "authorization": "basic",
        "data_encryption": "in_transit",
        "audit_logging": "enabled",
        "vulnerabilities": [],
        "recommendations": [
            "Implementar autenticación de dos factores",
            "Añadir encriptación de datos en reposo",
        ],
        "score": 82,
    }
