"""
Router para endpoints admin de clasificacion de prescripcion.

Issue #16 Fase 2: APIs para clasificacion masiva y estadisticas.
Issue #445: Endpoint para carga de listados de referencia via API.
"""
import time
import logging
from typing import List
from datetime import datetime, timedelta
from fastapi import APIRouter, Depends, File, HTTPException, Query, UploadFile, status
from sqlalchemy.orm import Session
from sqlalchemy import func

from app.database import get_db
from app.api.deps import get_current_active_user
from app.models.user import User
from app.models.product_catalog import ProductCatalog
from app.models.sales_data import SalesData
from app.models.sales_enrichment import SalesEnrichment
from app.models.prescription_reference_list import PrescriptionReferenceList
from app.models.enums import PrescriptionCategory
from app.services.prescription_classification_service import (
    PrescriptionClassificationService
)
from app.services.prescription_reference_service import PrescriptionReferenceService
from app.schemas.prescription import (
    BulkClassifyRequest,
    BulkClassifyResponse,
    BulkUpdateRequest,
    BulkUpdateResponse,
    ClassificationStats,
    DetailedClassificationStats,
    CategoryBreakdown,
    UnclassifiedSummary,
    TopUnclassifiedProduct,
    ProductWithCategory,
    ReferenceListUploadResponse,
    ReferenceListStats,
)

logger = logging.getLogger(__name__)

router = APIRouter(
    prefix="/admin/prescription",
    tags=["admin", "prescription"]
)


# ============================================================================
# ENDPOINTS
# ============================================================================

@router.post("/classify", response_model=BulkClassifyResponse)
async def bulk_classify_products(
    request: BulkClassifyRequest,
    db: Session = Depends(get_db),
    current_user: User = Depends(get_current_active_user)
):
    """
    Clasificar productos de prescripción en batch.

    **Parámetros**:
    - `filters`: Filtros opcionales (ej: {"nomen_estado": "ALTA"})
    - `dry_run`: Si True, solo retorna estadísticas sin persistir (default: True)
    - `limit`: Máximo de productos a clasificar (max 10,000)

    **Respuesta**:
    - Estadísticas de clasificación
    - Desglose por categoría
    - Tiempo de ejecución

    **Seguridad**: Solo usuarios admin pueden ejecutar este endpoint.
    """
    # Verificar permisos admin
    if current_user.role != "admin":
        raise HTTPException(
            status_code=status.HTTP_403_FORBIDDEN,
            detail="Solo usuarios admin pueden clasificar productos en batch"
        )

    start_time = time.time()

    try:
        # Query productos de prescripción no clasificados (o todos si force_reclassify)
        query = db.query(ProductCatalog).filter(
            (ProductCatalog.nomen_codigo_homogeneo.isnot(None)) |
            (ProductCatalog.cima_requiere_receta == True)
        )

        # Aplicar filtros opcionales
        if request.filters:
            # Ejemplo: filters={"nomen_estado": "ALTA"}
            for field, value in request.filters.items():
                if hasattr(ProductCatalog, field):
                    query = query.filter(getattr(ProductCatalog, field) == value)

        # Filtrar solo no clasificados (si no es reclasificación forzada)
        if not request.filters or not request.filters.get("force_reclassify"):
            query = query.filter(ProductCatalog.xfarma_prescription_category.is_(None))

        # Aplicar límite
        if request.limit:
            query = query.limit(request.limit)

        productos = query.all()

        # Ejecutar clasificación
        service = PrescriptionClassificationService(db=db)
        resultado = service.bulk_classify(productos, dry_run=request.dry_run)

        execution_time = time.time() - start_time

        # Convertir category_breakdown a formato Pydantic
        category_breakdown = [
            CategoryBreakdown(
                category=cat,
                count=count,
                percentage=(count / resultado["classified_count"] * 100)
                if resultado["classified_count"] > 0 else 0.0
            )
            for cat, count in resultado["category_breakdown"].items()
        ]

        logger.info(
            f"[ADMIN_PRESCRIPTION] Bulk classify completed",
            extra={
                "user_id": str(current_user.id),
                "total_products": resultado["total_products"],
                "classified": resultado["classified_count"],
                "dry_run": request.dry_run,
                "execution_time": execution_time
            }
        )

        return BulkClassifyResponse(
            total_products=resultado["total_products"],
            classified_count=resultado["classified_count"],
            skipped_otc_count=resultado["skipped_otc_count"],
            category_breakdown=category_breakdown,
            dry_run=request.dry_run,
            execution_time_seconds=round(execution_time, 2)
        )

    except Exception as e:
        logger.error(
            f"[ADMIN_PRESCRIPTION] Error in bulk classify: {e}",
            extra={"user_id": str(current_user.id)},
            exc_info=True
        )
        raise HTTPException(
            status_code=status.HTTP_500_INTERNAL_SERVER_ERROR,
            detail=f"Error clasificando productos: {str(e)}"
        )


@router.get("/stats", response_model=DetailedClassificationStats)
async def get_classification_stats(
    include_top_unclassified: bool = True,
    db: Session = Depends(get_db),
    current_user: User = Depends(get_current_active_user)
):
    """
    Obtener estadísticas globales de clasificación del catálogo.

    **Parámetros**:
    - `include_top_unclassified`: Si True, incluye top 10 productos sin clasificar por ventas

    **Respuesta**:
    - Total de productos (prescripción vs OTC)
    - Tasa de clasificación
    - Distribución por categoría
    - Top productos sin clasificar (opcional)

    **Seguridad**: Solo usuarios admin.
    """
    # Verificar permisos admin
    if current_user.role != "admin":
        raise HTTPException(
            status_code=status.HTTP_403_FORBIDDEN,
            detail="Solo usuarios admin pueden ver estadísticas de clasificación"
        )

    try:
        # Total de productos en catálogo
        total_products = db.query(func.count(ProductCatalog.id)).scalar()

        # Productos de prescripción (nomenclator o requieren receta)
        prescription_products = db.query(func.count(ProductCatalog.id)).filter(
            (ProductCatalog.nomen_codigo_homogeneo.isnot(None)) |
            (ProductCatalog.cima_requiere_receta == True)
        ).scalar()

        # Productos OTC/parafarmacia
        otc_products = total_products - prescription_products

        # Productos clasificados (SOLO de prescripción)
        classified_count = db.query(func.count(ProductCatalog.id)).filter(
            ProductCatalog.xfarma_prescription_category.isnot(None),
            (ProductCatalog.nomen_codigo_homogeneo.isnot(None)) |
            (ProductCatalog.cima_requiere_receta == True)
        ).scalar()

        # Productos sin clasificar (de prescripción)
        unclassified_count = prescription_products - classified_count

        # Tasa de clasificación (nunca puede ser > 100%)
        classification_rate = min(
            (classified_count / prescription_products * 100) if prescription_products > 0 else 0.0,
            100.0
        )

        # Distribución por categoría
        category_counts = db.query(
            ProductCatalog.xfarma_prescription_category,
            func.count(ProductCatalog.id)
        ).filter(
            ProductCatalog.xfarma_prescription_category.isnot(None)
        ).group_by(
            ProductCatalog.xfarma_prescription_category
        ).all()

        category_breakdown = [
            CategoryBreakdown(
                category=cat,
                count=count,
                percentage=(count / classified_count * 100) if classified_count > 0 else 0.0
            )
            for cat, count in category_counts
        ]

        # Última clasificación (producto más reciente con categoría)
        last_classified = db.query(ProductCatalog.updated_at).filter(
            ProductCatalog.xfarma_prescription_category.isnot(None)
        ).order_by(ProductCatalog.updated_at.desc()).first()

        last_classification_at = last_classified[0] if last_classified else None

        # Resumen de productos sin clasificar (opcional)
        unclassified_summary = None
        if include_top_unclassified:
            # Con nomenclator pero sin categoría
            with_nomenclator = db.query(func.count(ProductCatalog.id)).filter(
                ProductCatalog.nomen_codigo_homogeneo.isnot(None),
                ProductCatalog.xfarma_prescription_category.is_(None)
            ).scalar()

            # Solo CIMA requiere receta (sin nomenclator)
            only_cima = db.query(func.count(ProductCatalog.id)).filter(
                ProductCatalog.cima_requiere_receta == True,
                ProductCatalog.nomen_codigo_homogeneo.is_(None),
                ProductCatalog.xfarma_prescription_category.is_(None)
            ).scalar()

            # Top 10 sin clasificar por ventas del último mes
            last_month = datetime.utcnow() - timedelta(days=30)

            top_unclassified_query = db.query(
                ProductCatalog.national_code,
                ProductCatalog.nomen_nombre,
                func.sum(SalesData.total_amount).label("total_sales")
            ).join(
                SalesEnrichment,
                SalesEnrichment.product_catalog_id == ProductCatalog.id
            ).join(
                SalesData,
                SalesData.id == SalesEnrichment.sales_data_id
            ).filter(
                ProductCatalog.xfarma_prescription_category.is_(None),
                (ProductCatalog.nomen_codigo_homogeneo.isnot(None)) |
                (ProductCatalog.cima_requiere_receta == True),
                SalesData.sale_date >= last_month
            ).group_by(
                ProductCatalog.national_code,
                ProductCatalog.nomen_nombre
            ).order_by(
                func.sum(SalesData.total_amount).desc()
            ).limit(10).all()

            # Clasificar on-the-fly para sugerencias
            service = PrescriptionClassificationService()
            top_unclassified = []
            for nc, name, sales in top_unclassified_query:
                producto = db.query(ProductCatalog).filter(
                    ProductCatalog.national_code == nc
                ).first()
                suggested_category = service.classify_product(producto) if producto else None

                top_unclassified.append(TopUnclassifiedProduct(
                    national_code=nc,
                    product_name=name or "Sin nombre",
                    suggested_category=suggested_category,
                    total_sales_last_month=float(sales)
                ))

            unclassified_summary = UnclassifiedSummary(
                with_nomenclator=with_nomenclator,
                only_cima=only_cima,
                top_unclassified_by_sales=top_unclassified
            )

        logger.info(
            f"[ADMIN_PRESCRIPTION] Classification stats retrieved",
            extra={
                "user_id": str(current_user.id),
                "classification_rate": classification_rate,
                "classified_count": classified_count,
                "unclassified_count": unclassified_count
            }
        )

        return DetailedClassificationStats(
            total_products=total_products,
            prescription_products=prescription_products,
            otc_parapharmacy_products=otc_products,
            classified_count=classified_count,
            unclassified_count=unclassified_count,
            classification_rate=round(classification_rate, 2),
            category_distribution=category_breakdown,
            last_classification_at=last_classification_at,
            unclassified_summary=unclassified_summary
        )

    except Exception as e:
        logger.error(
            f"[ADMIN_PRESCRIPTION] Error getting classification stats: {e}",
            extra={"user_id": str(current_user.id)},
            exc_info=True
        )
        raise HTTPException(
            status_code=status.HTTP_500_INTERNAL_SERVER_ERROR,
            detail=f"Error obteniendo estadísticas de clasificación: {str(e)}"
        )


@router.get("/products", response_model=List[ProductWithCategory])
async def get_all_prescription_products(
    limit: int = Query(default=10000, ge=1, le=10000, description="Máximo de productos a retornar"),
    db: Session = Depends(get_db),
    current_user: User = Depends(get_current_active_user)
):
    """
    Obtener todos los productos de prescripción con sus categorías.

    Issue #408: Endpoint para tab Prescripción del admin dashboard.

    **Performance Optimization (Issue #408 Code Review)**:
    - Eager load de prescription_reference_list (1 query única)
    - N+1 query fix: 10,000 queries → 1-2 queries (99.6% reducción)
    - Expected time: ~1s para 10,000 productos (vs ~50s sin optimización)

    **Parámetros**:
    - `limit`: Máximo de productos a retornar (default: 10,000, max: 10,000)
              FastAPI valida: 422 Unprocessable Entity si < 1 o > 10,000

    **Respuesta**:
    - Lista de productos con:
      - national_code: Código nacional
      - product_name: Nombre del producto
      - category: Categoría asignada (None si no clasificado)
      - laboratory: Laboratorio fabricante
      - suggested_category: Categoría sugerida automáticamente (solo si NO clasificado)

    **Seguridad**: Solo usuarios admin.

    **Performance Metrics**:
    - Database query: ~200ms (con índices)
    - Classification (con reference_map): ~500ms (10k × 0.05ms)
    - Schema serialization: ~300ms
    - Total: ~1s
    """
    # Verificar permisos admin
    if current_user.role != "admin":
        raise HTTPException(
            status_code=status.HTTP_403_FORBIDDEN,
            detail="Solo usuarios admin pueden ver todos los productos"
        )

    start_time = time.time()

    try:
        # ===================================================================
        # PASO 1: EAGER LOAD REFERENCE MAP (1 query única - N+1 fix)
        # ===================================================================
        # Cargar TODA la prescription_reference_list en memoria (única vez)
        # Evita 10,000+ queries individuales dentro del loop
        reference_list = db.query(PrescriptionReferenceList).all()
        reference_map = {
            ref.national_code: PrescriptionCategory(ref.category)
            for ref in reference_list
        }

        logger.info(
            f"[ADMIN_PRESCRIPTION] Reference map loaded",
            extra={
                "user_id": str(current_user.id),
                "reference_entries": len(reference_map)
            }
        )

        # ===================================================================
        # PASO 2: QUERY PRODUCTOS DE PRESCRIPCIÓN
        # ===================================================================
        query = db.query(ProductCatalog).filter(
            (ProductCatalog.nomen_codigo_homogeneo.isnot(None)) |
            (ProductCatalog.cima_requiere_receta == True)
        )

        # Aplicar límite (FastAPI ya validó que 1 <= limit <= 10000)
        query = query.limit(limit)

        productos = query.all()

        # ===================================================================
        # PASO 3: CLASIFICACIÓN CON REFERENCE MAP (sin queries adicionales)
        # ===================================================================
        # Inicializar servicio con reference_map (evita N+1 queries)
        service = PrescriptionClassificationService(
            db=db,
            reference_map=reference_map  # ← Key optimization
        )

        # Construir respuesta con categorías y sugerencias
        result = []
        for producto in productos:
            # Sugerir categoría SOLO si no tiene una asignada
            # classify_product() ahora usa reference_map (O(1) lookup) en lugar de DB query
            suggested_category = None
            if not producto.xfarma_prescription_category:
                suggested_category = service.classify_product(producto)

            result.append(ProductWithCategory(
                national_code=producto.national_code,
                product_name=producto.nomen_nombre or producto.cima_nombre_comercial,
                category=producto.xfarma_prescription_category,
                laboratory=producto.nomen_laboratorio or producto.cima_laboratorio_titular,
                suggested_category=suggested_category
            ))

        # ===================================================================
        # PASO 4: PERFORMANCE LOGGING
        # ===================================================================
        execution_time_ms = int((time.time() - start_time) * 1000)

        logger.info(
            f"[ADMIN_PRESCRIPTION] Products retrieved",
            extra={
                "user_id": str(current_user.id),
                "total_products": len(result),
                "limit": limit,
                "execution_time_ms": execution_time_ms,  # ← Performance tracking
                "reference_map_size": len(reference_map)
            }
        )

        return result

    except Exception as e:
        logger.error(
            f"[ADMIN_PRESCRIPTION] Error getting products: {e}",
            extra={"user_id": str(current_user.id)},
            exc_info=True
        )
        raise HTTPException(
            status_code=status.HTTP_500_INTERNAL_SERVER_ERROR,
            detail=f"Error obteniendo productos: {str(e)}"
        )


@router.post("/bulk-update", response_model=BulkUpdateResponse)
async def bulk_update_classifications(
    request: BulkUpdateRequest,
    db: Session = Depends(get_db),
    current_user: User = Depends(get_current_active_user)
):
    """
    Actualizar clasificaciones de productos desde archivo Excel.

    **Parámetros**:
    - `classifications`: Lista de objetos {codigo_nacional, categoria}

    **Respuesta**:
    - Conteo de productos actualizados y errores

    **Seguridad**: Solo usuarios admin pueden ejecutar este endpoint.

    Issue #408: Endpoint para importación masiva desde Excel.
    """
    # Verificar permisos admin
    if current_user.role != "admin":
        raise HTTPException(
            status_code=status.HTTP_403_FORBIDDEN,
            detail="Solo usuarios admin pueden actualizar clasificaciones"
        )

    start_time = time.time()
    updated_count = 0
    failed_count = 0
    errors = []

    # Validar categorías permitidas
    valid_categories = {cat.value for cat in PrescriptionCategory}

    try:
        for item in request.classifications:
            try:
                # Validar categoría
                categoria_upper = item.categoria.upper().strip()
                if categoria_upper not in valid_categories:
                    failed_count += 1
                    errors.append(f"Categoría inválida: {item.categoria} (CN: {item.codigo_nacional})")
                    continue

                # Buscar producto por código nacional
                product = db.query(ProductCatalog).filter(
                    ProductCatalog.codigo_nacional == str(item.codigo_nacional)
                ).first()

                if not product:
                    failed_count += 1
                    errors.append(f"Producto no encontrado: CN {item.codigo_nacional}")
                    continue

                # Actualizar clasificación
                product.xfarma_prescription_category = PrescriptionCategory(categoria_upper)
                updated_count += 1

            except Exception as e:
                failed_count += 1
                errors.append(f"Error CN {item.codigo_nacional}: {str(e)}")

        # Commit cambios
        db.commit()

        execution_time = time.time() - start_time

        logger.info(
            f"[ADMIN_PRESCRIPTION] Bulk update completed",
            extra={
                "user_id": str(current_user.id),
                "updated": updated_count,
                "failed": failed_count,
                "execution_time": execution_time
            }
        )

        # Limitar errores en respuesta (máximo 50)
        return BulkUpdateResponse(
            updated_count=updated_count,
            failed_count=failed_count,
            errors=errors[:50]
        )

    except Exception as e:
        db.rollback()
        logger.error(
            f"[ADMIN_PRESCRIPTION] Error in bulk update: {e}",
            extra={"user_id": str(current_user.id)},
            exc_info=True
        )
        raise HTTPException(
            status_code=status.HTTP_500_INTERNAL_SERVER_ERROR,
            detail=f"Error actualizando clasificaciones: {str(e)}"
        )


# ============================================================================
# REFERENCE LIST ENDPOINTS (Issue #445)
# ============================================================================

@router.post("/reference-list/upload", response_model=ReferenceListUploadResponse)
async def upload_reference_list(
    file: UploadFile = File(..., description="Excel con listados oficiales de prescripcion"),
    truncate: bool = Query(
        default=False,
        description="Si True, elimina registros existentes antes de cargar"
    ),
    db: Session = Depends(get_db),
    current_user: User = Depends(get_current_active_user)
):
    """
    Cargar listados de referencia de prescripcion desde Excel oficial.

    Issue #445: Endpoint sostenible para cargar listados oficiales (dietas,
    tiras reactivas, efectos y accesorios) en lugar de scripts manuales.

    **Excel Esperado**:
    Archivo con hojas:
    - 'Dietas': Productos dieteticos (DIETOTERAPICOS)
    - 'Tiras reactivas': Tiras glucosa (TIRAS_REACTIVAS_GLUCOSA)
    - 'Efectos y accesorios': Clasificados automaticamente en:
      - ABSORB INC* -> INCONTINENCIA_FINANCIADA
      - MEDIA*/MUNEQUERA*/etc -> ORTOPEDIA_FINANCIADA
      - Resto -> EFECTOS_FINANCIADOS

    **Columnas esperadas**: CN, DESCRIPCION, PF, PMF/PVP+iva, situacion

    **Comportamiento**:
    - Sin truncate: Agrega nuevos registros (puede duplicar si ya existen)
    - Con truncate=true: Reemplaza todos los registros existentes

    **Side effects**:
    - Crea entradas en ProductCatalog para productos nuevos (necesario para joins)
    - Agrega codigos especiales (500017 formulas, 500009 vacunas)

    **Seguridad**: Solo usuarios admin pueden ejecutar este endpoint.

    **Ejemplo de uso** (curl):
    ```bash
    curl -X POST "http://localhost:8000/api/v1/admin/prescription/reference-list/upload?truncate=true" \
      -H "Authorization: Bearer {token}" \
      -F "file=@pf-fichero-1-noviembrev2xlsx.xlsx"
    ```
    """
    # Verificar permisos admin
    if current_user.role != "admin":
        raise HTTPException(
            status_code=status.HTTP_403_FORBIDDEN,
            detail="Solo usuarios admin pueden cargar listados de referencia"
        )

    # Validar tipo de archivo
    if not file.filename.endswith(('.xlsx', '.xls')):
        raise HTTPException(
            status_code=status.HTTP_400_BAD_REQUEST,
            detail="El archivo debe ser Excel (.xlsx o .xls)"
        )

    try:
        # Leer contenido del archivo
        file_content = await file.read()

        if len(file_content) == 0:
            raise HTTPException(
                status_code=status.HTTP_400_BAD_REQUEST,
                detail="El archivo esta vacio"
            )

        # Cargar usando el servicio
        service = PrescriptionReferenceService(db=db)
        result = service.load_from_excel(
            file_content=file_content,
            truncate_before=truncate
        )

        logger.info(
            f"[ADMIN_PRESCRIPTION] Reference list uploaded",
            extra={
                "user_id": str(current_user.id),
                "filename": file.filename,
                "total_loaded": result['total_loaded'],
                "catalog_entries_created": result['catalog_entries_created'],
                "truncated": truncate
            }
        )

        return ReferenceListUploadResponse(**result)

    except ValueError as e:
        raise HTTPException(
            status_code=status.HTTP_400_BAD_REQUEST,
            detail=str(e)
        )
    except Exception as e:
        db.rollback()
        logger.error(
            f"[ADMIN_PRESCRIPTION] Error uploading reference list: {e}",
            extra={"user_id": str(current_user.id)},
            exc_info=True
        )
        raise HTTPException(
            status_code=status.HTTP_500_INTERNAL_SERVER_ERROR,
            detail=f"Error cargando listados de referencia: {str(e)}"
        )


@router.get("/reference-list/stats", response_model=ReferenceListStats)
async def get_reference_list_stats(
    db: Session = Depends(get_db),
    current_user: User = Depends(get_current_active_user)
):
    """
    Obtener estadisticas de listados de referencia cargados.

    Issue #445: Endpoint para monitorear estado de listados de prescripcion.

    **Respuesta**:
    - total_entries: Total de productos en prescription_reference_list
    - category_breakdown: Desglose por categoria (DIETOTERAPICOS, TIRAS_REACTIVAS, etc)
    - source_breakdown: Desglose por fuente (Dietas, Tiras reactivas, Efectos)
    - last_loaded_at: Fecha de ultima carga
    - catalog_coverage: Productos que existen en ProductCatalog (para joins)

    **Uso**: Verificar que los listados estan cargados antes de procesar ventas.

    **Seguridad**: Solo usuarios admin.
    """
    # Verificar permisos admin
    if current_user.role != "admin":
        raise HTTPException(
            status_code=status.HTTP_403_FORBIDDEN,
            detail="Solo usuarios admin pueden ver estadisticas de listados"
        )

    try:
        service = PrescriptionReferenceService(db=db)
        stats = service.get_stats()

        logger.info(
            f"[ADMIN_PRESCRIPTION] Reference list stats retrieved",
            extra={
                "user_id": str(current_user.id),
                "total_entries": stats['total_entries']
            }
        )

        return ReferenceListStats(**stats)

    except Exception as e:
        logger.error(
            f"[ADMIN_PRESCRIPTION] Error getting reference list stats: {e}",
            extra={"user_id": str(current_user.id)},
            exc_info=True
        )
        raise HTTPException(
            status_code=status.HTTP_500_INTERNAL_SERVER_ERROR,
            detail=f"Error obteniendo estadisticas: {str(e)}"
        )
