# backend/app/api/admin_keywords.py
"""
Admin API for keyword management (Issue #449).

Endpoints:
- GET /admin/keywords: List all keywords with filters
- POST /admin/keywords: Create new keyword
- PUT /admin/keywords/{id}: Update keyword
- DELETE /admin/keywords/{id}: Soft delete (is_active=False) or hard delete
- POST /admin/keywords/preview: Preview impact of changes
- GET /admin/keywords/export: Export keywords as JSON
- POST /admin/keywords/import: Bulk import keywords
- GET /admin/keywords/categories: Available NECESIDAD categories
"""

from typing import Optional

from fastapi import APIRouter, Depends, HTTPException, Query
from fastapi.responses import JSONResponse
from sqlalchemy.orm import Session

from app.api.deps import get_current_user, get_db
from app.models.user import User
from app.schemas.keyword_override import (
    ApplyKeywordsRequest,
    ApplyKeywordsResponse,
    AvailableCategoryResponse,
    KeywordBulkImportRequest,
    KeywordBulkImportResponse,
    KeywordOverrideCreate,
    KeywordOverrideListResponse,
    KeywordOverrideResponse,
    KeywordOverrideUpdate,
    KeywordPreviewRequest,
    KeywordPreviewResponse,
)
from app.services.keyword_override_service import KeywordOverrideService

router = APIRouter(prefix="/admin/keywords")


@router.get("", response_model=KeywordOverrideListResponse)
def list_keywords(
    keyword_type: Optional[str] = Query(
        None, description="Filter by type: keyword, brand, blacklist"
    ),
    category: Optional[str] = Query(None, description="Filter by category"),
    is_active: Optional[bool] = Query(None, description="Filter by active status"),
    search: Optional[str] = Query(None, description="Search in keyword text"),
    limit: int = Query(50, ge=1, le=200),
    offset: int = Query(0, ge=0),
    current_user: User = Depends(get_current_user),
    db: Session = Depends(get_db),
):
    """
    List all keyword overrides with filters.

    Requires admin role.
    """
    # Verificar permisos (solo admin puede gestionar keywords)
    if current_user.role != "admin":
        raise HTTPException(
            status_code=403,
            detail="Solo administradores pueden gestionar keywords",
        )

    service = KeywordOverrideService(db)
    return service.list_keywords(
        keyword_type=keyword_type,
        category=category,
        is_active=is_active,
        search=search,
        limit=limit,
        offset=offset,
    )


@router.post("", response_model=KeywordOverrideResponse)
def create_keyword(
    request: KeywordOverrideCreate,
    current_user: User = Depends(get_current_user),
    db: Session = Depends(get_db),
):
    """
    Create a new keyword override.

    The keyword takes effect immediately in classification.
    Requires admin role.
    """
    if current_user.role != "admin":
        raise HTTPException(
            status_code=403,
            detail="Solo administradores pueden crear keywords",
        )

    service = KeywordOverrideService(db)
    try:
        override = service.create_keyword(request, created_by=current_user.id)
        # Return as dict for response model
        return {
            "id": override.id,
            "keyword": override.keyword,
            "category": override.category,
            "keyword_type": override.keyword_type,
            "priority": override.priority,
            "is_active": override.is_active,
            "created_by": override.created_by,
            "created_at": override.created_at,
            "updated_at": override.updated_at,
            "source": override.source,
            "notes": override.notes,
            "creator_email": current_user.email,
        }
    except ValueError as e:
        raise HTTPException(status_code=400, detail=str(e))


@router.put("/{keyword_id}", response_model=KeywordOverrideResponse)
def update_keyword(
    keyword_id: int,
    request: KeywordOverrideUpdate,
    current_user: User = Depends(get_current_user),
    db: Session = Depends(get_db),
):
    """
    Update an existing keyword override.

    Changes take effect immediately.
    Requires admin role.
    """
    if current_user.role != "admin":
        raise HTTPException(
            status_code=403,
            detail="Solo administradores pueden modificar keywords",
        )

    service = KeywordOverrideService(db)
    try:
        override = service.update_keyword(keyword_id, request)
        return {
            "id": override.id,
            "keyword": override.keyword,
            "category": override.category,
            "keyword_type": override.keyword_type,
            "priority": override.priority,
            "is_active": override.is_active,
            "created_by": override.created_by,
            "created_at": override.created_at,
            "updated_at": override.updated_at,
            "source": override.source,
            "notes": override.notes,
            "creator_email": override.creator.email if override.creator else None,
        }
    except ValueError as e:
        raise HTTPException(status_code=404, detail=str(e))


@router.delete("/{keyword_id}")
def delete_keyword(
    keyword_id: int,
    hard_delete: bool = Query(
        False, description="If true, permanently delete instead of deactivate"
    ),
    current_user: User = Depends(get_current_user),
    db: Session = Depends(get_db),
):
    """
    Delete (deactivate) or hard delete a keyword override.

    By default, keywords are soft-deleted (is_active=False).
    Use hard_delete=true for permanent deletion.
    Requires admin role.
    """
    if current_user.role != "admin":
        raise HTTPException(
            status_code=403,
            detail="Solo administradores pueden eliminar keywords",
        )

    service = KeywordOverrideService(db)
    try:
        return service.delete_keyword(keyword_id, hard_delete=hard_delete)
    except ValueError as e:
        raise HTTPException(status_code=404, detail=str(e))


@router.post("/preview", response_model=KeywordPreviewResponse)
def preview_keyword_impact(
    request: KeywordPreviewRequest,
    current_user: User = Depends(get_current_user),
    db: Session = Depends(get_db),
):
    """
    Preview impact of adding/changing a keyword.

    Returns products that would be reclassified if this keyword is applied.
    Use this before creating/updating keywords to understand the impact.
    Requires admin role.
    """
    if current_user.role != "admin":
        raise HTTPException(
            status_code=403,
            detail="Solo administradores pueden previsualizar keywords",
        )

    service = KeywordOverrideService(db)
    return service.preview_impact(request)


@router.get("/export")
def export_keywords(
    format: str = Query("json", pattern="^(json)$"),
    include_inactive: bool = Query(False),
    current_user: User = Depends(get_current_user),
    db: Session = Depends(get_db),
):
    """
    Export all keywords as JSON.

    Useful for backup or transfer between environments.
    Requires admin role.
    """
    if current_user.role != "admin":
        raise HTTPException(
            status_code=403,
            detail="Solo administradores pueden exportar keywords",
        )

    service = KeywordOverrideService(db)
    keywords = service.export_keywords(format=format, include_inactive=include_inactive)

    return JSONResponse(
        content={"keywords": keywords, "count": len(keywords)},
        headers={"Content-Disposition": "attachment; filename=keywords_export.json"},
    )


@router.post("/import", response_model=KeywordBulkImportResponse)
def import_keywords(
    request: KeywordBulkImportRequest,
    current_user: User = Depends(get_current_user),
    db: Session = Depends(get_db),
):
    """
    Bulk import keywords from JSON.

    Set overwrite_existing=true to update existing keywords.
    Requires admin role.
    """
    if current_user.role != "admin":
        raise HTTPException(
            status_code=403,
            detail="Solo administradores pueden importar keywords",
        )

    service = KeywordOverrideService(db)
    return service.import_keywords(request, created_by=current_user.id)


@router.get("/categories", response_model=list[AvailableCategoryResponse])
def get_available_categories(
    current_user: User = Depends(get_current_user),
):
    """
    Get list of available NECESIDAD categories for dropdowns.

    Returns all valid categories from the symptom taxonomy.
    Requires admin role for consistency with other keyword endpoints.
    """
    if current_user.role != "admin":
        raise HTTPException(
            status_code=403,
            detail="Solo administradores pueden acceder a categorías",
        )

    from app.schemas.symptom_taxonomy import NecesidadEspecifica

    categories = []
    for e in NecesidadEspecifica:
        categories.append(
            {
                "value": e.value,
                "label": e.value.replace("_", " ").title(),
                "description": None,
            }
        )

    # Sort alphabetically by label
    categories.sort(key=lambda x: x["label"])

    return categories


@router.post("/apply", response_model=ApplyKeywordsResponse)
def apply_keywords_to_products(
    request: ApplyKeywordsRequest,
    current_user: User = Depends(get_current_user),
    db: Session = Depends(get_db),
):
    """
    Apply keyword overrides to existing products (Issue #449 Phase 3).

    Scans all products in ProductCatalogVentaLibre and applies matching
    keyword overrides to update their ml_category.

    Use dry_run=true (default) to preview changes before applying.
    Set dry_run=false to actually update products.

    Requires admin role.

    Returns:
        ApplyKeywordsResponse with count and sample of affected products
    """
    if current_user.role != "admin":
        raise HTTPException(
            status_code=403,
            detail="Solo administradores pueden aplicar keywords",
        )

    service = KeywordOverrideService(db)
    return service.apply_keywords_to_products(
        dry_run=request.dry_run,
        keyword_ids=request.keyword_ids,
    )
