# backend/app/api/v1/endpoints/admin_database.py
"""
API endpoints para administración de base de datos.

Issue #348 FASE 2: Backend - Services & APIs
Incluye gestión de backups y vistas materializadas.
"""
import logging
from typing import Optional

from fastapi import APIRouter, Depends, HTTPException, status, Query, Request
from sqlalchemy.ext.asyncio import AsyncSession
from slowapi import Limiter
from slowapi.util import get_remote_address
from pydantic import BaseModel

from app.api.deps import get_db, get_current_user
from app.models.user import User
from app.core.subscription_limits import Permission
from app.core.security import require_permissions
from app.services.database_management_service import DatabaseManagementService
from app.services.audit_service import AuditService

logger = logging.getLogger(__name__)
router = APIRouter()

# Rate limiter para operaciones costosas
limiter = Limiter(key_func=get_remote_address)


class BackupRequest(BaseModel):
    """Request para crear backup"""
    path: Optional[str] = None  # Ruta opcional para el backup


class BackupResponse(BaseModel):
    """Response de backup creado"""
    id: int
    file_path: str
    file_size: int
    sha256_hash: Optional[str]
    hmac_signature: Optional[str]
    status: str
    created_at: str

    class Config:
        from_attributes = True


class BackupListResponse(BaseModel):
    """Response de lista de backups"""
    id: int
    file_path: str
    file_size: int
    sha256_hash: Optional[str]
    status: str
    created_at: str
    is_verified: Optional[bool] = None
    error_message: Optional[str] = None

    class Config:
        from_attributes = True


class DatabaseSizeResponse(BaseModel):
    """Response de tamaño de base de datos"""
    database_size: int
    database_size_pretty: str
    top_tables: list


@router.post(
    "/backups",
    response_model=BackupResponse,
    status_code=status.HTTP_201_CREATED,
    dependencies=[Depends(require_permissions(Permission.MANAGE_DATABASE.value))]
)
@limiter.limit("3/hour")  # Limitado a 3 backups por hora
async def create_backup(
    request: Request,
    backup_request: Optional[BackupRequest] = None,
    db: AsyncSession = Depends(get_db),
    current_user: User = Depends(get_current_user)
):
    """
    Crear un backup de la base de datos.

    **Permisos requeridos:** MANAGE_DATABASE

    **Rate limit:** 3 requests por hora

    El backup incluye:
    - Hash SHA-256 para verificación de integridad
    - Firma HMAC para detección de manipulación
    - Todas las tablas del sistema

    Args:
        backup_request: Configuración opcional del backup

    Returns:
        Metadatos del backup creado

    Raises:
        403: Sin permisos MANAGE_DATABASE
        429: Rate limit excedido
        500: Error al crear backup
    """
    try:
        # Crear backup
        backup_path = backup_request.path if backup_request else None
        backup_metadata = await DatabaseManagementService.create_backup(
            db=db,
            backup_path=backup_path
        )

        # Audit logging
        await AuditService.log_action(
            db=db,
            user_id=current_user.id,
            action="DATABASE_BACKUP_CREATED",
            entity_type="backup",
            entity_id=str(backup_metadata.id),
            details={
                "file_path": backup_metadata.file_path,
                "file_size": backup_metadata.file_size,
                "sha256_hash": backup_metadata.sha256_hash[:8] + "..." if backup_metadata.sha256_hash else None
            }
        )

        return BackupResponse(
            id=backup_metadata.id,
            file_path=backup_metadata.file_path,
            file_size=backup_metadata.file_size,
            sha256_hash=backup_metadata.sha256_hash,
            hmac_signature=backup_metadata.hmac_signature,
            status=backup_metadata.status,
            created_at=backup_metadata.created_at.isoformat()
        )

    except RuntimeError as e:
        logger.error(f"Error al crear backup: {str(e)}")
        raise HTTPException(
            status_code=status.HTTP_500_INTERNAL_SERVER_ERROR,
            detail=str(e)
        )
    except Exception as e:
        logger.error(f"Error inesperado al crear backup: {str(e)}")
        raise HTTPException(
            status_code=status.HTTP_500_INTERNAL_SERVER_ERROR,
            detail="Error al crear backup de base de datos"
        )


@router.get(
    "/backups",
    response_model=list[BackupListResponse],
    dependencies=[Depends(require_permissions(Permission.VIEW_SYSTEM_STATS.value))]
)
async def list_backups(
    limit: int = Query(20, ge=1, le=100, description="Máximo de backups a devolver"),
    db: AsyncSession = Depends(get_db),
    current_user: User = Depends(get_current_user)
):
    """
    Listar backups disponibles con verificación de integridad.

    **Permisos requeridos:** VIEW_SYSTEM_STATS

    Para cada backup se verifica:
    - Existencia del archivo
    - Integridad del hash SHA-256
    - Validez de la firma HMAC

    Args:
        limit: Número máximo de backups a listar

    Returns:
        Lista de backups con estado de verificación
    """
    try:
        backups = await DatabaseManagementService.list_backups(
            db=db,
            limit=limit
        )

        return [
            BackupListResponse(
                id=backup.id,
                file_path=backup.file_path,
                file_size=backup.file_size,
                sha256_hash=backup.sha256_hash,
                status=backup.status,
                created_at=backup.created_at.isoformat(),
                is_verified=getattr(backup, 'is_verified', None),
                error_message=backup.error_message if hasattr(backup, 'error_message') else None
            )
            for backup in backups
        ]

    except HTTPException:
        raise  # Re-raise HTTPExceptions preserving their status codes
    except Exception as e:
        logger.error(f"Error al listar backups: {str(e)}")
        raise HTTPException(
            status_code=status.HTTP_500_INTERNAL_SERVER_ERROR,
            detail="Error al obtener lista de backups"
        )


@router.post(
    "/storage/refresh",
    status_code=status.HTTP_204_NO_CONTENT,
    dependencies=[Depends(require_permissions(Permission.MANAGE_DATABASE.value))]
)
@limiter.limit("10/hour")  # Limitado a 10 refreshes por hora
async def refresh_storage_stats(
    request: Request,
    db: AsyncSession = Depends(get_db),
    current_user: User = Depends(get_current_user)
):
    """
    Actualizar vista materializada de estadísticas de almacenamiento.

    **Permisos requeridos:** MANAGE_DATABASE

    **Rate limit:** 10 requests por hora

    Ejecuta `REFRESH MATERIALIZED VIEW pharmacy_storage_stats` para
    actualizar las estadísticas de uso de almacenamiento por farmacia.

    Returns:
        204 No Content si la actualización es exitosa

    Raises:
        403: Sin permisos MANAGE_DATABASE
        429: Rate limit excedido
        500: Error al actualizar vista
    """
    try:
        await DatabaseManagementService.refresh_storage_stats(db)

        # Audit logging
        await AuditService.log_action(
            db=db,
            user_id=current_user.id,
            action="MATERIALIZED_VIEW_REFRESHED",
            entity_type="database",
            entity_id="pharmacy_storage_stats",
            details={"view_name": "pharmacy_storage_stats"}
        )

        return None  # 204 No Content

    except RuntimeError as e:
        logger.error(f"Error al actualizar vista materializada: {str(e)}")
        raise HTTPException(
            status_code=status.HTTP_500_INTERNAL_SERVER_ERROR,
            detail=str(e)
        )
    except Exception as e:
        logger.error(f"Error inesperado al actualizar vista: {str(e)}")
        raise HTTPException(
            status_code=status.HTTP_500_INTERNAL_SERVER_ERROR,
            detail="Error al actualizar estadísticas de almacenamiento"
        )


@router.post(
    "/partner-analytics/refresh",
    status_code=status.HTTP_204_NO_CONTENT,
    dependencies=[Depends(require_permissions(Permission.MANAGE_DATABASE.value))]
)
@limiter.limit("10/hour")  # Limitado a 10 refreshes por hora
async def refresh_partner_analytics(
    request: Request,
    db: AsyncSession = Depends(get_db),
    current_user: User = Depends(get_current_user)
):
    """
    Actualizar vista materializada de análisis de partners.

    **Permisos requeridos:** MANAGE_DATABASE

    **Rate limit:** 10 requests por hora

    Ejecuta `REFRESH MATERIALIZED VIEW CONCURRENTLY pharmacy_lab_sales_summary`
    para actualizar las ventas pre-calculadas por (pharmacy_id, laboratory).

    Esta view optimiza queries de /initialize de 180s → < 2s.

    Returns:
        204 No Content si la actualización es exitosa

    Raises:
        403: Sin permisos MANAGE_DATABASE
        429: Rate limit excedido
        500: Error al actualizar vista

    Note:
        - View debe refreshearse diariamente para mantener partners suggestions actuales
        - Usa CONCURRENTLY para no bloquear lecturas (producción-safe)
        - View contiene últimos 13 meses de ventas agregadas
    """
    try:
        await DatabaseManagementService.refresh_partner_analytics(db)

        # Audit logging
        await AuditService.log_action(
            db=db,
            user_id=current_user.id,
            action="MATERIALIZED_VIEW_REFRESHED",
            entity_type="database",
            entity_id="pharmacy_lab_sales_summary",
            details={"view_name": "pharmacy_lab_sales_summary", "purpose": "partner_analytics_optimization"}
        )

        return None  # 204 No Content

    except RuntimeError as e:
        logger.error(f"Error al actualizar vista materializada partner analytics: {str(e)}")
        raise HTTPException(
            status_code=status.HTTP_500_INTERNAL_SERVER_ERROR,
            detail=str(e)
        )
    except Exception as e:
        logger.error(f"Error inesperado al actualizar vista partner analytics: {str(e)}")
        raise HTTPException(
            status_code=status.HTTP_500_INTERNAL_SERVER_ERROR,
            detail="Error al actualizar análisis de partners"
        )


@router.get(
    "/size",
    response_model=DatabaseSizeResponse,
    dependencies=[Depends(require_permissions(Permission.VIEW_SYSTEM_STATS.value))]
)
def get_database_size(
    db=Depends(get_db),
    current_user: User = Depends(get_current_user)
):
    """
    Obtener el tamaño de la base de datos y tablas principales.

    **Permisos requeridos:** VIEW_SYSTEM_STATS

    Returns:
        Información de tamaño de base de datos y top 10 tablas

    Raises:
        403: Sin permisos VIEW_SYSTEM_STATS
        500: Error al obtener información
    """
    try:
        size_info = DatabaseManagementService.get_database_size(db)

        return DatabaseSizeResponse(
            database_size=size_info["database_size"],
            database_size_pretty=size_info["database_size_pretty"],
            top_tables=size_info["top_tables"]
        )

    except HTTPException:
        raise  # Re-raise HTTPExceptions preserving their status codes
    except Exception as e:
        logger.error(f"Error al obtener tamaño de base de datos: {str(e)}")
        raise HTTPException(
            status_code=status.HTTP_500_INTERNAL_SERVER_ERROR,
            detail="Error al obtener información de tamaño"
        )
