# backend/app/api/admin_curated_groups.py
"""
Admin API endpoints for InterchangeableCuratedGroup management.

Issue #521: Admin UI para gestión de grupos curados.
ADR-004: Simplificación del Sistema de Clasificación

Endpoints:
- GET  /admin/curated-groups         - Listar grupos con filtros
- GET  /admin/curated-groups/stats   - Estadísticas generales
- POST /admin/curated-groups         - Crear grupo
- GET  /admin/curated-groups/{id}    - Obtener grupo por ID
- PUT  /admin/curated-groups/{id}    - Actualizar grupo
- DELETE /admin/curated-groups/{id}  - Eliminar grupo

Members:
- GET  /admin/curated-groups/{id}/members  - Listar miembros
- POST /admin/curated-groups/{id}/members  - Añadir miembro
- DELETE /admin/curated-groups/{id}/members/{member_id} - Eliminar miembro
"""

from typing import Optional
from uuid import UUID

import structlog
from fastapi import APIRouter, Depends, HTTPException, Query
from sqlalchemy.orm import Session

from app.api.deps import get_current_user, get_db
from app.models.user import User
from app.schemas.curated_groups import (
    CuratedGroupCreate,
    CuratedGroupDetailResponse,
    CuratedGroupFilter,
    CuratedGroupListResponse,
    CuratedGroupResponse,
    CuratedGroupStats,
    CuratedGroupUpdate,
    GroupMemberCreate,
    GroupMemberListResponse,
    GroupMemberResponse,
)
from app.services.curated_groups_service import CuratedGroupsService

logger = structlog.get_logger(__name__)

router = APIRouter(prefix="/admin/curated-groups", tags=["admin-curated-groups"])


def _require_admin(current_user: User) -> None:
    """Verificar que el usuario es administrador."""
    if current_user.role != "admin":
        raise HTTPException(
            status_code=403,
            detail="Solo administradores pueden gestionar grupos curados"
        )


# === GROUPS ENDPOINTS ===

@router.get("", response_model=CuratedGroupListResponse)
def list_curated_groups(
    necesidad_l1: Optional[str] = Query(None, description="Filtrar por categoría L1"),
    is_active: Optional[bool] = Query(None, description="Filtrar por estado activo"),
    source: Optional[str] = Query(None, description="Filtrar por origen (legacy_clustering, manual_curated)"),
    search: Optional[str] = Query(None, description="Buscar en nombre/descripción"),
    order_by: str = Query("total_sales_amount", description="Campo para ordenar"),
    order_desc: bool = Query(True, description="Orden descendente"),
    page: int = Query(1, ge=1, description="Página"),
    page_size: int = Query(20, ge=1, le=100, description="Tamaño de página"),
    db: Session = Depends(get_db),
    current_user: User = Depends(get_current_user),
):
    """
    Lista grupos curados con filtros y paginación.

    Solo administradores.
    """
    _require_admin(current_user)

    filter_params = CuratedGroupFilter(
        necesidad_l1=necesidad_l1,
        is_active=is_active,
        source=source,
        search=search,
        order_by=order_by,
        order_desc=order_desc,
        page=page,
        page_size=page_size,
    )

    service = CuratedGroupsService(db)
    groups, total = service.list_groups(filter_params)

    return CuratedGroupListResponse(
        groups=[CuratedGroupResponse.model_validate(g) for g in groups],
        total=total,
        page=page,
        page_size=page_size,
    )


@router.get("/stats", response_model=CuratedGroupStats)
def get_curated_groups_stats(
    db: Session = Depends(get_db),
    current_user: User = Depends(get_current_user),
):
    """
    Obtener estadísticas generales de grupos curados.

    Solo administradores.
    """
    _require_admin(current_user)

    service = CuratedGroupsService(db)
    return service.get_stats()


@router.post("", response_model=CuratedGroupResponse, status_code=201)
def create_curated_group(
    data: CuratedGroupCreate,
    db: Session = Depends(get_db),
    current_user: User = Depends(get_current_user),
):
    """
    Crear un nuevo grupo curado.

    El slug se genera automáticamente si no se proporciona.
    Solo administradores.
    """
    _require_admin(current_user)

    service = CuratedGroupsService(db)
    group = service.create_group(data, created_by=current_user.email)

    logger.info(
        "admin.curated_group.created",
        group_id=str(group.id),
        group_name=group.group_name,
        created_by=current_user.email,
    )

    return CuratedGroupResponse.model_validate(group)


@router.get("/{group_id}", response_model=CuratedGroupDetailResponse)
def get_curated_group(
    group_id: UUID,
    db: Session = Depends(get_db),
    current_user: User = Depends(get_current_user),
):
    """
    Obtener un grupo curado por ID con sus miembros.

    Solo administradores.
    """
    _require_admin(current_user)

    service = CuratedGroupsService(db)
    group = service.get_group(group_id)

    if not group:
        raise HTTPException(
            status_code=404,
            detail=f"Grupo curado id={group_id} no encontrado"
        )

    # Get members
    members, _ = service.list_members(group_id)

    # Build response with members
    response = CuratedGroupDetailResponse.model_validate(group)
    response.members = [GroupMemberResponse.model_validate(m) for m in members]

    # Calculate top brands
    brand_counts: dict[str, int] = {}
    for member in members:
        if member.detected_brand:
            brand_counts[member.detected_brand] = brand_counts.get(member.detected_brand, 0) + 1

    response.top_brands = [
        {"brand": brand, "count": count}
        for brand, count in sorted(brand_counts.items(), key=lambda x: -x[1])[:5]
    ]

    return response


@router.put("/{group_id}", response_model=CuratedGroupResponse)
def update_curated_group(
    group_id: UUID,
    data: CuratedGroupUpdate,
    db: Session = Depends(get_db),
    current_user: User = Depends(get_current_user),
):
    """
    Actualizar un grupo curado existente.

    Solo administradores.
    """
    _require_admin(current_user)

    service = CuratedGroupsService(db)
    group = service.update_group(group_id, data)

    if not group:
        raise HTTPException(
            status_code=404,
            detail=f"Grupo curado id={group_id} no encontrado"
        )

    logger.info(
        "admin.curated_group.updated",
        group_id=str(group_id),
        updated_by=current_user.email,
    )

    return CuratedGroupResponse.model_validate(group)


@router.delete("/{group_id}")
def delete_curated_group(
    group_id: UUID,
    db: Session = Depends(get_db),
    current_user: User = Depends(get_current_user),
):
    """
    Eliminar un grupo curado y todos sus miembros.

    Solo administradores. Esta acción es irreversible.
    """
    _require_admin(current_user)

    service = CuratedGroupsService(db)
    deleted = service.delete_group(group_id)

    if not deleted:
        raise HTTPException(
            status_code=404,
            detail=f"Grupo curado id={group_id} no encontrado"
        )

    logger.info(
        "admin.curated_group.deleted",
        group_id=str(group_id),
        deleted_by=current_user.email,
    )

    return {"message": "Grupo curado eliminado exitosamente", "id": str(group_id)}


@router.post("/{group_id}/toggle", response_model=CuratedGroupResponse)
def toggle_curated_group(
    group_id: UUID,
    db: Session = Depends(get_db),
    current_user: User = Depends(get_current_user),
):
    """
    Alternar estado activo/inactivo de un grupo curado.

    Solo administradores.
    """
    _require_admin(current_user)

    service = CuratedGroupsService(db)
    group = service.get_group(group_id)

    if not group:
        raise HTTPException(
            status_code=404,
            detail=f"Grupo curado id={group_id} no encontrado"
        )

    # Toggle is_active
    update_data = CuratedGroupUpdate(is_active=not group.is_active)
    group = service.update_group(group_id, update_data)

    logger.info(
        "admin.curated_group.toggled",
        group_id=str(group_id),
        is_active=group.is_active,
        toggled_by=current_user.email,
    )

    return CuratedGroupResponse.model_validate(group)


# === MEMBERS ENDPOINTS ===

@router.get("/{group_id}/members", response_model=GroupMemberListResponse)
def list_group_members(
    group_id: UUID,
    db: Session = Depends(get_db),
    current_user: User = Depends(get_current_user),
):
    """
    Listar miembros de un grupo curado.

    Solo administradores.
    """
    _require_admin(current_user)

    service = CuratedGroupsService(db)

    # Verify group exists
    group = service.get_group(group_id)
    if not group:
        raise HTTPException(
            status_code=404,
            detail=f"Grupo curado id={group_id} no encontrado"
        )

    members, total = service.list_members(group_id)

    return GroupMemberListResponse(
        members=[GroupMemberResponse.model_validate(m) for m in members],
        total=total,
    )


@router.post("/{group_id}/members", response_model=GroupMemberResponse, status_code=201)
def add_group_member(
    group_id: UUID,
    data: GroupMemberCreate,
    db: Session = Depends(get_db),
    current_user: User = Depends(get_current_user),
):
    """
    Añadir un miembro a un grupo curado.

    Requiere al menos ean13 o codigo_nacional.
    Solo administradores.
    """
    _require_admin(current_user)

    # Note: Validation for at least one code is done in GroupMemberCreate schema

    service = CuratedGroupsService(db)
    member = service.add_member(group_id, data, added_by=current_user.email)

    if not member:
        raise HTTPException(
            status_code=404,
            detail=f"Grupo curado id={group_id} no encontrado"
        )

    logger.info(
        "admin.curated_group.member_added",
        group_id=str(group_id),
        member_id=str(member.id),
        added_by=current_user.email,
    )

    return GroupMemberResponse.model_validate(member)


@router.delete("/{group_id}/members/{member_id}")
def remove_group_member(
    group_id: UUID,
    member_id: UUID,
    db: Session = Depends(get_db),
    current_user: User = Depends(get_current_user),
):
    """
    Eliminar un miembro de un grupo curado.

    Solo administradores.
    """
    _require_admin(current_user)

    service = CuratedGroupsService(db)
    removed = service.remove_member(group_id, member_id)

    if not removed:
        raise HTTPException(
            status_code=404,
            detail=f"Miembro id={member_id} no encontrado en grupo id={group_id}"
        )

    logger.info(
        "admin.curated_group.member_removed",
        group_id=str(group_id),
        member_id=str(member_id),
        removed_by=current_user.email,
    )

    return {"message": "Miembro eliminado exitosamente", "id": str(member_id)}
