# backend/app/api/admin_category_aliases.py
"""
Admin API endpoints for CategoryAlias management.

Issue #459: Category normalization en base de datos.

Endpoints:
- GET  /admin/category-aliases         - Listar aliases
- GET  /admin/category-aliases/stats   - Estadísticas
- POST /admin/category-aliases         - Crear alias
- PUT  /admin/category-aliases/{id}    - Actualizar alias
- DELETE /admin/category-aliases/{id}  - Eliminar alias (soft/hard)
- POST /admin/category-aliases/{id}/toggle - Alternar activo/inactivo
"""

import logging
from typing import Optional

from fastapi import APIRouter, Depends, HTTPException, Query
from sqlalchemy.orm import Session

from app.api.deps import get_current_user, get_db
from app.models.user import User
from app.schemas.category_alias import (
    CategoryAliasCreate,
    CategoryAliasListResponse,
    CategoryAliasResponse,
    CategoryAliasStatsResponse,
    CategoryAliasUpdate,
)
from app.services.category_alias_service import CategoryAliasService

logger = logging.getLogger(__name__)

router = APIRouter(prefix="/admin/category-aliases", tags=["admin-category-aliases"])


@router.get("", response_model=CategoryAliasListResponse)
def list_category_aliases(
    is_active: Optional[bool] = Query(None, description="Filtrar por estado activo"),
    search: Optional[str] = Query(None, description="Buscar en source/target"),
    limit: int = Query(100, ge=1, le=500),
    offset: int = Query(0, ge=0),
    db: Session = Depends(get_db),
    current_user: User = Depends(get_current_user),
):
    """
    Lista todos los category aliases con filtros opcionales.

    Solo administradores.
    """
    if current_user.role != "admin":
        raise HTTPException(
            status_code=403,
            detail="Solo administradores pueden gestionar category aliases"
        )

    service = CategoryAliasService(db)
    result = service.list_aliases(
        is_active=is_active,
        search=search,
        limit=limit,
        offset=offset,
    )

    return result


@router.get("/stats", response_model=CategoryAliasStatsResponse)
def get_category_alias_stats(
    db: Session = Depends(get_db),
    current_user: User = Depends(get_current_user),
):
    """
    Obtener estadísticas de category aliases.

    Solo administradores.
    """
    if current_user.role != "admin":
        raise HTTPException(
            status_code=403,
            detail="Solo administradores pueden ver estadísticas"
        )

    service = CategoryAliasService(db)
    return service.get_stats()


@router.post("", response_model=CategoryAliasResponse, status_code=201)
def create_category_alias(
    data: CategoryAliasCreate,
    db: Session = Depends(get_db),
    current_user: User = Depends(get_current_user),
):
    """
    Crear un nuevo category alias.

    Solo administradores.
    """
    if current_user.role != "admin":
        raise HTTPException(
            status_code=403,
            detail="Solo administradores pueden crear category aliases"
        )

    service = CategoryAliasService(db)

    try:
        alias = service.create_alias(data)
        logger.info(
            f"Admin {current_user.email} created category alias: "
            f"{alias.source_category} → {alias.target_category}"
        )
        return alias
    except ValueError as e:
        raise HTTPException(status_code=400, detail=str(e))


@router.put("/{alias_id}", response_model=CategoryAliasResponse)
def update_category_alias(
    alias_id: int,
    data: CategoryAliasUpdate,
    db: Session = Depends(get_db),
    current_user: User = Depends(get_current_user),
):
    """
    Actualizar un category alias existente.

    Solo administradores.
    """
    if current_user.role != "admin":
        raise HTTPException(
            status_code=403,
            detail="Solo administradores pueden actualizar category aliases"
        )

    service = CategoryAliasService(db)

    try:
        alias = service.update_alias(alias_id, data)
        if not alias:
            raise HTTPException(
                status_code=404,
                detail=f"Category alias id={alias_id} no encontrado"
            )

        logger.info(
            f"Admin {current_user.email} updated category alias id={alias_id}"
        )
        return alias
    except ValueError as e:
        raise HTTPException(status_code=400, detail=str(e))


@router.delete("/{alias_id}")
def delete_category_alias(
    alias_id: int,
    hard: bool = Query(False, description="Si True, elimina permanentemente"),
    db: Session = Depends(get_db),
    current_user: User = Depends(get_current_user),
):
    """
    Eliminar un category alias.

    Por defecto hace soft delete (desactiva). Con hard=True elimina permanentemente.
    Solo administradores.
    """
    if current_user.role != "admin":
        raise HTTPException(
            status_code=403,
            detail="Solo administradores pueden eliminar category aliases"
        )

    service = CategoryAliasService(db)
    deleted = service.delete_alias(alias_id, hard=hard)

    if not deleted:
        raise HTTPException(
            status_code=404,
            detail=f"Category alias id={alias_id} no encontrado"
        )

    action = "hard deleted" if hard else "deactivated"
    logger.info(
        f"Admin {current_user.email} {action} category alias id={alias_id}"
    )

    return {"message": f"Category alias {action} successfully", "id": alias_id}


@router.post("/{alias_id}/toggle", response_model=CategoryAliasResponse)
def toggle_category_alias(
    alias_id: int,
    db: Session = Depends(get_db),
    current_user: User = Depends(get_current_user),
):
    """
    Alternar estado activo/inactivo de un category alias.

    Solo administradores.
    """
    if current_user.role != "admin":
        raise HTTPException(
            status_code=403,
            detail="Solo administradores pueden modificar category aliases"
        )

    service = CategoryAliasService(db)
    alias = service.toggle_active(alias_id)

    if not alias:
        raise HTTPException(
            status_code=404,
            detail=f"Category alias id={alias_id} no encontrado"
        )

    logger.info(
        f"Admin {current_user.email} toggled category alias id={alias_id} "
        f"to is_active={alias.is_active}"
    )

    return alias
