# backend/app/api/admin_brand_aliases.py
"""
Admin API endpoints for BrandAlias management.

Sistema de corrección de marcas para BrandDetectionService.

Endpoints:
- GET  /admin/brand-aliases           - Listar aliases
- GET  /admin/brand-aliases/stats     - Estadísticas
- GET  /admin/brand-aliases/{id}      - Obtener alias por ID
- POST /admin/brand-aliases           - Crear alias
- PUT  /admin/brand-aliases/{id}      - Actualizar alias
- DELETE /admin/brand-aliases/{id}    - Eliminar alias (soft/hard)
- POST /admin/brand-aliases/{id}/toggle - Alternar activo/inactivo
- POST /admin/brand-aliases/{id}/reprocess - Re-procesar productos afectados
"""

import logging
from typing import Optional

from fastapi import APIRouter, Depends, HTTPException, Query
from sqlalchemy.orm import Session

from app.api.deps import get_current_user, get_db
from app.models.brand_alias import BrandAliasAction
from app.models.user import User
from app.schemas.brand_alias import (
    BrandAliasCreate,
    BrandAliasListResponse,
    BrandAliasReprocessResponse,
    BrandAliasResponse,
    BrandAliasStatsResponse,
    BrandAliasUpdate,
)
from app.services.brand_alias_service import BrandAliasService

logger = logging.getLogger(__name__)

router = APIRouter(prefix="/admin/brand-aliases", tags=["admin-brand-aliases"])


@router.get("", response_model=BrandAliasListResponse)
def list_brand_aliases(
    is_active: Optional[bool] = Query(None, description="Filtrar por estado activo"),
    action: Optional[BrandAliasAction] = Query(None, description="Filtrar por tipo: alias/exclude"),
    search: Optional[str] = Query(None, description="Buscar en source/target"),
    limit: int = Query(100, ge=1, le=500),
    offset: int = Query(0, ge=0),
    db: Session = Depends(get_db),
    current_user: User = Depends(get_current_user),
):
    """
    Lista todos los brand aliases con filtros opcionales.

    Incluye affected_count (productos pendientes de re-proceso) para cada alias.

    Solo administradores.
    """
    if current_user.role != "admin":
        raise HTTPException(
            status_code=403,
            detail="Solo administradores pueden gestionar brand aliases",
        )

    service = BrandAliasService(db)
    result = service.list_aliases(
        is_active=is_active,
        action=action,
        search=search,
        limit=limit,
        offset=offset,
    )

    return result


@router.get("/stats", response_model=BrandAliasStatsResponse)
def get_brand_alias_stats(
    db: Session = Depends(get_db),
    current_user: User = Depends(get_current_user),
):
    """
    Obtener estadísticas de brand aliases.

    Incluye conteo de aliases vs excludes.

    Solo administradores.
    """
    if current_user.role != "admin":
        raise HTTPException(
            status_code=403,
            detail="Solo administradores pueden ver estadísticas",
        )

    service = BrandAliasService(db)
    return service.get_stats()


@router.get("/{alias_id}", response_model=BrandAliasResponse)
def get_brand_alias(
    alias_id: int,
    db: Session = Depends(get_db),
    current_user: User = Depends(get_current_user),
):
    """
    Obtener un brand alias por ID.

    Incluye affected_count (productos con detected_brand = source_brand).

    Solo administradores.
    """
    if current_user.role != "admin":
        raise HTTPException(
            status_code=403,
            detail="Solo administradores pueden ver brand aliases",
        )

    service = BrandAliasService(db)
    alias = service.get_alias(alias_id)

    if not alias:
        raise HTTPException(
            status_code=404,
            detail=f"Brand alias id={alias_id} no encontrado",
        )

    # Add affected_count for response
    affected_count = service._get_affected_count(alias.source_brand)
    response = BrandAliasResponse.model_validate(alias)
    response.affected_count = affected_count
    return response


@router.post("", response_model=BrandAliasResponse, status_code=201)
def create_brand_alias(
    data: BrandAliasCreate,
    db: Session = Depends(get_db),
    current_user: User = Depends(get_current_user),
):
    """
    Crear un nuevo brand alias.

    - action=alias: Fusiona marcas (source_brand → target_brand)
    - action=exclude: Excluye de detección (source_brand no es marca)

    Solo administradores.
    """
    if current_user.role != "admin":
        raise HTTPException(
            status_code=403,
            detail="Solo administradores pueden crear brand aliases",
        )

    service = BrandAliasService(db)

    try:
        alias = service.create_alias(data)

        if alias.action == BrandAliasAction.EXCLUDE:
            logger.info(
                f"Admin {current_user.email} created brand alias: "
                f"{alias.source_brand} → EXCLUDE"
            )
        else:
            logger.info(
                f"Admin {current_user.email} created brand alias: "
                f"{alias.source_brand} → {alias.target_brand}"
            )

        # Add affected_count for response
        affected_count = service._get_affected_count(alias.source_brand)
        response = BrandAliasResponse.model_validate(alias)
        response.affected_count = affected_count
        return response

    except ValueError as e:
        raise HTTPException(status_code=400, detail=str(e))


@router.put("/{alias_id}", response_model=BrandAliasResponse)
def update_brand_alias(
    alias_id: int,
    data: BrandAliasUpdate,
    db: Session = Depends(get_db),
    current_user: User = Depends(get_current_user),
):
    """
    Actualizar un brand alias existente.

    Solo administradores.
    """
    if current_user.role != "admin":
        raise HTTPException(
            status_code=403,
            detail="Solo administradores pueden actualizar brand aliases",
        )

    service = BrandAliasService(db)

    try:
        alias = service.update_alias(alias_id, data)
        if not alias:
            raise HTTPException(
                status_code=404,
                detail=f"Brand alias id={alias_id} no encontrado",
            )

        logger.info(
            f"Admin {current_user.email} updated brand alias id={alias_id}"
        )

        # Add affected_count for response
        affected_count = service._get_affected_count(alias.source_brand)
        response = BrandAliasResponse.model_validate(alias)
        response.affected_count = affected_count
        return response

    except ValueError as e:
        raise HTTPException(status_code=400, detail=str(e))


@router.delete("/{alias_id}")
def delete_brand_alias(
    alias_id: int,
    hard: bool = Query(False, description="Si True, elimina permanentemente"),
    db: Session = Depends(get_db),
    current_user: User = Depends(get_current_user),
):
    """
    Eliminar un brand alias.

    Por defecto hace soft delete (desactiva). Con hard=True elimina permanentemente.
    Solo administradores.
    """
    if current_user.role != "admin":
        raise HTTPException(
            status_code=403,
            detail="Solo administradores pueden eliminar brand aliases",
        )

    service = BrandAliasService(db)
    deleted = service.delete_alias(alias_id, hard=hard)

    if not deleted:
        raise HTTPException(
            status_code=404,
            detail=f"Brand alias id={alias_id} no encontrado",
        )

    action = "hard deleted" if hard else "deactivated"
    logger.info(
        f"Admin {current_user.email} {action} brand alias id={alias_id}"
    )

    return {"message": f"Brand alias {action} successfully", "id": alias_id}


@router.post("/{alias_id}/toggle", response_model=BrandAliasResponse)
def toggle_brand_alias(
    alias_id: int,
    db: Session = Depends(get_db),
    current_user: User = Depends(get_current_user),
):
    """
    Alternar estado activo/inactivo de un brand alias.

    Solo administradores.
    """
    if current_user.role != "admin":
        raise HTTPException(
            status_code=403,
            detail="Solo administradores pueden modificar brand aliases",
        )

    service = BrandAliasService(db)
    alias = service.toggle_active(alias_id)

    if not alias:
        raise HTTPException(
            status_code=404,
            detail=f"Brand alias id={alias_id} no encontrado",
        )

    logger.info(
        f"Admin {current_user.email} toggled brand alias id={alias_id} "
        f"to is_active={alias.is_active}"
    )

    # Add affected_count for response
    affected_count = service._get_affected_count(alias.source_brand)
    response = BrandAliasResponse.model_validate(alias)
    response.affected_count = affected_count
    return response


@router.post("/{alias_id}/reprocess", response_model=BrandAliasReprocessResponse)
def reprocess_brand_alias(
    alias_id: int,
    db: Session = Depends(get_db),
    current_user: User = Depends(get_current_user),
):
    """
    Re-procesa productos afectados por un brand alias.

    Actualiza todos los SalesEnrichment con detected_brand = source_brand:
    - Si action=alias: Cambia detected_brand a target_brand
    - Si action=exclude: Cambia detected_brand a NULL

    Solo administradores.
    """
    if current_user.role != "admin":
        raise HTTPException(
            status_code=403,
            detail="Solo administradores pueden re-procesar brand aliases",
        )

    service = BrandAliasService(db)

    try:
        result = service.reprocess_alias(alias_id)

        logger.info(
            f"Admin {current_user.email} reprocessed brand alias id={alias_id}: "
            f"{result['updated_count']} productos actualizados"
        )

        return result

    except ValueError as e:
        raise HTTPException(status_code=400, detail=str(e))
