﻿# backend/app/api/admin.py
"""
Endpoints de administración y mantenimiento del sistema.
Incluye operaciones peligrosas que requieren confirmaciones especiales y rate limiting.
FASE 5: Añadido modo desarrollador y sistema de notificaciones inteligentes.
"""

import logging
from datetime import timedelta
from typing import Dict, Literal, Optional

import structlog
from fastapi import APIRouter, BackgroundTasks, Depends, HTTPException, Query, Request
from pydantic import BaseModel
from sqlalchemy import text
from sqlalchemy.orm import Session

from app.utils.datetime_utils import utc_now

from ..api.deps import get_current_admin_user, get_current_user
from ..core.rate_limiting import admin_delete_limit, admin_general_limit, admin_vacuum_limit
from ..core.security import require_permissions
from ..database import get_db
from ..models.file_upload import FileUpload
from ..models.pharmacy import Pharmacy
from ..models.sales_data import SalesData
from ..models.sales_enrichment import SalesEnrichment
from ..models.system_health import DeveloperLog, SystemAlert
from ..models.system_status import SystemComponent, SystemStatus
from ..models.user import User
from ..services.health_monitoring_service import HealthMonitoringService, log_to_developer_mode

logger = logging.getLogger(__name__)
struct_logger = structlog.get_logger(__name__)

router = APIRouter(prefix="/admin", tags=["admin"])


# Modelos Pydantic para requests
class AlertAcknowledgeRequest(BaseModel):
    """Request para reconocer una alerta."""

    acknowledged_by: str
    note: Optional[str] = None


class DeveloperModeToggleRequest(BaseModel):
    """Request para activar/desactivar modo desarrollador."""

    enabled: bool
    user_id: str


@router.delete("/delete-all-data")
@admin_delete_limit
def delete_all_pharmacy_data(
    request: Request,
    pharmacy_id: Optional[str] = Query(None, description="ID de la farmacia (opcional)"),
    current_user: User = Depends(require_permissions("admin")),
    db: Session = Depends(get_db),
):
    """
    OPERACIÓN PELIGROSA: Borrar TODOS los datos de ventas de una farmacia.

    Esta operación elimina de forma PERMANENTE:
    - Todos los archivos de ventas (file_uploads)
    - Todos los datos de ventas procesados (sales_data)
    - Todos los datos de enriquecimiento (sales_enrichment)

    Se mantienen:
    - Configuración de la farmacia
    - Usuarios y configuraciones de la aplicación
    - Configuraciones de partners

    Args:
        pharmacy_id: ID de la farmacia (por defecto: primera farmacia encontrada)
        db: Sesión de base de datos

    Returns:
        Confirmación de borrado con estadísticas

    Raises:
        HTTPException: Si no se encuentra la farmacia o hay errores en la operación
    """

    try:
        # Encontrar la farmacia
        if pharmacy_id:
            pharmacy = db.query(Pharmacy).filter(Pharmacy.id == pharmacy_id).first()
            if not pharmacy:
                raise HTTPException(
                    status_code=404,
                    detail=f"No se encontró la farmacia con ID: {pharmacy_id}",
                )
        else:
            # Si no se especifica pharmacy_id, usar la primera farmacia
            pharmacy = db.query(Pharmacy).first()
            if not pharmacy:
                raise HTTPException(
                    status_code=404,
                    detail="No se encontró ninguna farmacia en el sistema",
                )

        logger.warning(
            f"INICIANDO BORRADO MASIVO para farmacia {pharmacy.id} - {pharmacy.name} por usuario {current_user.email}"
        )

        # Contar registros antes del borrado para estadísticas
        sales_count = db.query(SalesData).filter(SalesData.pharmacy_id == pharmacy.id).count()
        enrichment_count = (
            db.query(SalesEnrichment).join(SalesData).filter(SalesData.pharmacy_id == pharmacy.id).count()
        )
        uploads_count = db.query(FileUpload).filter(FileUpload.pharmacy_id == pharmacy.id).count()

        deleted_records = {
            "pharmacy_id": str(pharmacy.id),
            "pharmacy_name": pharmacy.name,
            "sales_data_deleted": sales_count,
            "enrichment_data_deleted": enrichment_count,
            "file_uploads_deleted": uploads_count,
            "total_deleted": sales_count + enrichment_count + uploads_count,
        }

        # Borrar en orden correcto para respetar foreign keys

        # 1. Borrar datos de enriquecimiento (referencias a sales_data)
        if enrichment_count > 0:
            logger.info(f"Borrando {enrichment_count} registros de enriquecimiento...")
            # Usar subconsulta para borrar enrichment relacionados con sales_data de esta farmacia
            enrichment_subquery = (
                db.query(SalesEnrichment.id).join(SalesData).filter(SalesData.pharmacy_id == pharmacy.id).subquery()
            )

            deleted_enrichment = (
                db.query(SalesEnrichment)
                .filter(SalesEnrichment.id.in_(enrichment_subquery))
                .delete(synchronize_session=False)
            )

            logger.info(f"Borrados {deleted_enrichment} registros de enriquecimiento")

        # 2. Borrar datos de ventas
        if sales_count > 0:
            logger.info(f"Borrando {sales_count} registros de ventas...")
            deleted_sales = (
                db.query(SalesData).filter(SalesData.pharmacy_id == pharmacy.id).delete(synchronize_session=False)
            )

            logger.info(f"Borrados {deleted_sales} registros de ventas")

        # 3. Borrar archivos de upload
        if uploads_count > 0:
            logger.info(f"Borrando {uploads_count} archivos de upload...")
            deleted_uploads = (
                db.query(FileUpload).filter(FileUpload.pharmacy_id == pharmacy.id).delete(synchronize_session=False)
            )

            logger.info(f"Borrados {deleted_uploads} archivos de upload")

        # Commit de la transacción
        db.commit()

        logger.warning(
            f"BORRADO MASIVO COMPLETADO para farmacia {pharmacy.id} - Total eliminado: {deleted_records['total_deleted']} registros"
        )

        return {
            "success": True,
            "message": "Todos los datos de ventas han sido eliminados exitosamente",
            "details": deleted_records,
            "warning": "Esta operación es irreversible",
        }

    except HTTPException:
        # Re-raise HTTP exceptions
        raise

    except Exception as e:
        db.rollback()
        logger.error(f"Error durante borrado masivo: {str(e)}")
        raise HTTPException(
            status_code=500,
            detail=f"Error interno durante el borrado de datos: {str(e)}",
        )


@router.get("/statistics")
def get_system_statistics(
    pharmacy_id: Optional[str] = Query(None, description="ID de la farmacia (opcional)"),
    current_user: User = Depends(get_current_admin_user),  # ✅ SECURITY FIX: Admin required
    db: Session = Depends(get_db),
):
    """
    Obtener estadísticas del sistema para auditoría antes de operaciones peligrosas.

    Args:
        pharmacy_id: ID de la farmacia específica
        db: Sesión de base de datos

    Returns:
        Estadísticas detalladas del sistema
    """
    try:
        if pharmacy_id:
            # Estadísticas de una farmacia específica
            pharmacy = db.query(Pharmacy).filter(Pharmacy.id == pharmacy_id).first()
            if not pharmacy:
                raise HTTPException(status_code=404, detail="Farmacia no encontrada")

            sales_count = db.query(SalesData).filter(SalesData.pharmacy_id == pharmacy.id).count()
            enrichment_count = (
                db.query(SalesEnrichment).join(SalesData).filter(SalesData.pharmacy_id == pharmacy.id).count()
            )
            uploads_count = db.query(FileUpload).filter(FileUpload.pharmacy_id == pharmacy.id).count()

            return {
                "pharmacy_id": str(pharmacy.id),
                "pharmacy_name": pharmacy.name,
                "sales_data_count": sales_count,
                "enrichment_data_count": enrichment_count,
                "file_uploads_count": uploads_count,
                "total_records": sales_count + enrichment_count + uploads_count,
            }
        else:
            # Estadísticas globales del sistema
            total_pharmacies = db.query(Pharmacy).count()
            total_sales = db.query(SalesData).count()
            total_enrichment = db.query(SalesEnrichment).count()
            total_uploads = db.query(FileUpload).count()

            return {
                "system_wide": True,
                "total_pharmacies": total_pharmacies,
                "total_sales_data": total_sales,
                "total_enrichment_data": total_enrichment,
                "total_file_uploads": total_uploads,
                "total_records": total_sales + total_enrichment + total_uploads,
            }

    except HTTPException:
        raise
    except Exception as e:
        logger.error(f"Error obteniendo estadísticas: {str(e)}")
        raise HTTPException(
            status_code=500,
            detail=f"Error obteniendo estadísticas del sistema: {str(e)}",
        )


@router.post("/vacuum-database")
@admin_vacuum_limit
def vacuum_database(
    request: Request,
    current_user: User = Depends(require_permissions("admin")),
    db: Session = Depends(get_db),
):
    """
    Ejecutar VACUUM en la base de datos para recuperar espacio después de borrados masivos.

    Args:
        db: Sesión de base de datos

    Returns:
        Confirmación de la operación de limpieza
    """
    try:
        logger.info(f"Ejecutando VACUUM en la base de datos por usuario {current_user.email}...")

        # Ejecutar VACUUM para recuperar espacio
        db.execute(text("VACUUM ANALYZE;"))
        db.commit()

        logger.info("VACUUM completado exitosamente")

        return {"success": True, "message": "Base de datos limpiada exitosamente"}

    except Exception as e:
        db.rollback()
        logger.error(f"Error durante VACUUM: {str(e)}")
        raise HTTPException(status_code=500, detail=f"Error limpiando la base de datos: {str(e)}")


# ===============================================
# FASE 5: MODO DESARROLLADOR Y NOTIFICACIONES
# ===============================================


@router.get("/health/overview")
def get_health_overview(
    current_user: User = Depends(get_current_admin_user),  # ✅ SECURITY FIX: Admin required
    db: Session = Depends(get_db),
) -> Dict:
    """
    Obtiene un resumen completo de la salud del sistema.
    Diseñado para administradores de farmacia.
    """
    try:
        health_service = HealthMonitoringService(db)

        # Tomar snapshot actual
        snapshot = health_service.take_performance_snapshot()

        # Ejecutar chequeos de salud
        health_status = health_service.check_system_health()

        # Obtener tendencias de las últimas 24 horas
        trends = health_service.get_performance_trends(hours=24)

        # Compilar resumen ejecutivo
        overview = {
            "timestamp": utc_now().isoformat(),
            "overall_status": health_status["overall_status"],
            # Métricas actuales
            "current_metrics": {
                "cpu_percent": snapshot.cpu_usage_percent,
                "memory_usage_mb": snapshot.memory_usage_mb,
                "memory_available_mb": snapshot.memory_available_mb,
                "catalog_products": snapshot.catalog_total_products,
                "catalog_with_cima": snapshot.catalog_with_cima,
                "catalog_with_nomenclator": snapshot.catalog_with_nomenclator,
                "catalog_freshness_hours": snapshot.catalog_freshness_hours,
            },
            # Resultados de chequeos específicos
            "health_checks": health_status["checks"],
            # Alertas activas
            "active_alerts": health_status["alerts"],
            # Tendencias para gráficos
            "trends_24h": trends,
            # Recomendaciones específicas para farmacia
            "recommendations": health_status.get("recommendations", []),
        }

        struct_logger.info(
            "Health overview generated",
            overall_status=overview["overall_status"],
            active_alerts=len(overview["active_alerts"]),
        )

        return overview

    except Exception as e:
        struct_logger.error("Error generating health overview", error=str(e))
        raise HTTPException(status_code=500, detail=f"Error obteniendo resumen de salud: {str(e)}")


@router.get("/developer/logs")
def get_developer_logs(
    limit: int = Query(default=100, le=500),
    level: Optional[Literal["DEBUG", "INFO", "WARNING", "ERROR"]] = None,
    logger_name: Optional[str] = None,
    since_hours: Optional[int] = Query(default=24, le=168),  # Máximo 1 semana
    current_user: User = Depends(require_permissions("admin")),
    db: Session = Depends(get_db),
) -> Dict:
    """
    Obtiene logs técnicos detallados para modo desarrollador.
    """
    try:
        health_service = HealthMonitoringService(db)

        since = None
        if since_hours:
            since = utc_now() - timedelta(hours=since_hours)

        logs = health_service.get_developer_logs(limit=limit, level=level, logger_name=logger_name, since=since)

        # Agregar estadísticas de logs
        total_count = (
            db.query(DeveloperLog).count()
            if not since
            else db.query(DeveloperLog).filter(DeveloperLog.timestamp >= since).count()
        )

        error_count = (
            db.query(DeveloperLog).filter(DeveloperLog.level == "ERROR").count()
            if not since
            else db.query(DeveloperLog).filter(DeveloperLog.timestamp >= since, DeveloperLog.level == "ERROR").count()
        )

        return {
            "logs": logs,
            "metadata": {
                "total_count": total_count,
                "error_count": error_count,
                "returned_count": len(logs),
                "filters": {
                    "level": level,
                    "logger_name": logger_name,
                    "since_hours": since_hours,
                },
            },
        }

    except Exception as e:
        struct_logger.error("Error retrieving developer logs", error=str(e))
        raise HTTPException(status_code=500, detail=f"Error obteniendo logs: {str(e)}")


@router.post("/developer/log")
def create_developer_log(
    level: Literal["DEBUG", "INFO", "WARNING", "ERROR"],
    logger_name: str,
    message: str,
    stack_trace: Optional[str] = None,
    context: Optional[Dict] = None,
    execution_time_ms: Optional[float] = None,
    memory_usage_mb: Optional[float] = None,
    current_user: User = Depends(require_permissions("admin")),
    db: Session = Depends(get_db),
) -> Dict:
    """
    Crea un log técnico en modo desarrollador.
    Usado por otros servicios para registrar información detallada.
    """
    try:
        log_to_developer_mode(
            db=db,
            level=level,
            logger_name=logger_name,
            message=message,
            stack_trace=stack_trace,
            context=context,
            execution_time_ms=execution_time_ms,
            memory_usage_mb=memory_usage_mb,
        )

        return {
            "success": True,
            "message": "Log registrado correctamente",
            "timestamp": utc_now().isoformat(),
        }

    except Exception as e:
        struct_logger.error("Error creating developer log", error=str(e))
        raise HTTPException(status_code=500, detail=f"Error creando log: {str(e)}")


@router.get("/alerts")
def get_system_alerts(
    active_only: bool = Query(default=True),
    severity: Optional[Literal["low", "medium", "high", "critical"]] = None,
    limit: int = Query(default=50, le=200),
    current_user: User = Depends(get_current_admin_user),  # ✅ SECURITY FIX: Admin required
    db: Session = Depends(get_db),
) -> Dict:
    """
    Obtiene alertas del sistema para administradores.
    """
    try:
        query = db.query(SystemAlert)

        if active_only:
            query = query.filter(SystemAlert.is_active.is_(True))

        if severity:
            query = query.filter(SystemAlert.severity == severity)

        alerts = query.order_by(SystemAlert.last_occurred.desc()).limit(limit).all()

        alerts_data = []
        for alert in alerts:
            alerts_data.append(
                {
                    "id": str(alert.id),
                    "type": alert.alert_type,
                    "severity": alert.severity,
                    "title": alert.title,
                    "message": alert.message,
                    "context": alert.context,
                    "is_active": alert.is_active,
                    "is_acknowledged": alert.is_acknowledged,
                    "acknowledged_by": alert.acknowledged_by,
                    "acknowledged_at": (alert.acknowledged_at.isoformat() if alert.acknowledged_at else None),
                    "occurrence_count": alert.occurrence_count,
                    "first_occurred": alert.first_occurred.isoformat(),
                    "last_occurred": alert.last_occurred.isoformat(),
                }
            )

        # Estadísticas de alertas
        total_active = db.query(SystemAlert).filter(SystemAlert.is_active.is_(True)).count()
        critical_count = (
            db.query(SystemAlert).filter(SystemAlert.is_active.is_(True), SystemAlert.severity == "critical").count()
        )

        return {
            "alerts": alerts_data,
            "metadata": {
                "total_active": total_active,
                "critical_count": critical_count,
                "returned_count": len(alerts_data),
                "filters": {"active_only": active_only, "severity": severity},
            },
        }

    except Exception as e:
        struct_logger.error("Error retrieving alerts", error=str(e))
        raise HTTPException(status_code=500, detail=f"Error obteniendo alertas: {str(e)}")


@router.post("/alerts/{alert_id}/acknowledge")
def acknowledge_alert(
    alert_id: str,
    request: AlertAcknowledgeRequest,
    current_user: User = Depends(require_permissions("admin")),
    db: Session = Depends(get_db),
) -> Dict:
    """
    Reconoce una alerta específica.
    """
    try:
        alert = db.query(SystemAlert).filter(SystemAlert.id == alert_id).first()

        if not alert:
            raise HTTPException(status_code=404, detail="Alerta no encontrada")

        alert.is_acknowledged = True
        alert.acknowledged_by = current_user.email  # Use authenticated user instead of request
        alert.acknowledged_at = utc_now()
        alert.updated_at = utc_now()

        # Registrar en logs de desarrollador
        log_to_developer_mode(
            db=db,
            level="INFO",
            logger_name="admin_api",
            message=f"Alert acknowledged: {alert.alert_type}",
            context={
                "alert_id": str(alert.id),
                "acknowledged_by": request.acknowledged_by,
                "note": request.note,
            },
        )

        db.commit()

        return {
            "success": True,
            "message": f"Alerta {alert.alert_type} reconocida correctamente",
            "acknowledged_at": alert.acknowledged_at.isoformat(),
        }

    except HTTPException:
        raise
    except Exception as e:
        struct_logger.error("Error acknowledging alert", error=str(e), alert_id=alert_id)
        db.rollback()
        raise HTTPException(status_code=500, detail=f"Error reconociendo alerta: {str(e)}")


@router.get("/system/performance/trends")
def get_performance_trends(
    hours: int = Query(default=24, le=168),  # Máximo 1 semana
    current_user: User = Depends(get_current_admin_user),  # ✅ SECURITY FIX: Admin required
    db: Session = Depends(get_db),
) -> Dict:
    """
    Obtiene tendencias de performance para gráficos del dashboard.
    """
    try:
        health_service = HealthMonitoringService(db)
        trends = health_service.get_performance_trends(hours=hours)

        return {
            "trends": trends,
            "metadata": {
                "hours_requested": hours,
                "data_points": len(trends["timestamps"]),
                "period": f"Últimas {hours} horas",
            },
        }

    except Exception as e:
        struct_logger.error("Error retrieving performance trends", error=str(e))
        raise HTTPException(status_code=500, detail=f"Error obteniendo tendencias: {str(e)}")


@router.get("/system/status/detailed")
def get_detailed_system_status(
    current_user: User = Depends(get_current_admin_user),  # ✅ SECURITY FIX: Admin required
    db: Session = Depends(get_db),
) -> Dict:
    """
    Obtiene estado detallado de todos los componentes del sistema.
    Específicamente para modo desarrollador.
    """
    try:
        components = [
            SystemComponent.CATALOG,
            SystemComponent.CIMA,
            SystemComponent.NOMENCLATOR,
            SystemComponent.DATABASE,
            SystemComponent.REDIS,
        ]

        component_status = {}

        for component in components:
            status = db.query(SystemStatus).filter_by(component=component).first()

            if status:
                component_status[component.value] = {
                    "status": status.status.value if status.status else "unknown",
                    "message": status.message,
                    "progress": status.progress,
                    "processed_items": status.processed_items,
                    "total_items": status.total_items,
                    "started_at": (status.started_at.isoformat() if status.started_at else None),
                    "updated_at": (status.updated_at.isoformat() if status.updated_at else None),
                    "details": status.details,
                }
            else:
                component_status[component.value] = {
                    "status": "not_initialized",
                    "message": f"Componente {component.value} no inicializado",
                }

        return {
            "components": component_status,
            "timestamp": utc_now().isoformat(),
        }

    except Exception as e:
        struct_logger.error("Error retrieving detailed system status", error=str(e))
        raise HTTPException(status_code=500, detail=f"Error obteniendo estado detallado: {str(e)}")


@router.post("/system/performance/snapshot")
def take_performance_snapshot(
    background_tasks: BackgroundTasks,
    current_user: User = Depends(require_permissions("admin")),
    db: Session = Depends(get_db),
) -> Dict:
    """
    Fuerza la captura de un snapshot de performance.
    """
    try:

        def capture_snapshot():
            health_service = HealthMonitoringService(db)
            snapshot = health_service.take_performance_snapshot()
            struct_logger.info("Manual performance snapshot captured", snapshot_id=str(snapshot.id))

        background_tasks.add_task(capture_snapshot)

        return {
            "success": True,
            "message": "Snapshot de performance programado",
            "timestamp": utc_now().isoformat(),
        }

    except Exception as e:
        struct_logger.error("Error scheduling performance snapshot", error=str(e))
        raise HTTPException(status_code=500, detail=f"Error programando snapshot: {str(e)}")


# Endpoint para testing de alertas (solo desarrollo)
@router.post("/cleanup-duplicates")
@admin_general_limit
def cleanup_duplicate_sales(
    request: Request,
    pharmacy_id: Optional[str] = Query(None, description="ID de la farmacia (opcional, por defecto: todas)"),
    current_user: User = Depends(require_permissions("admin")),
    db: Session = Depends(get_db),
) -> Dict:
    """
    OPERACIÓN DE MANTENIMIENTO: Limpia registros duplicados en sales_data.

    Criterios de duplicado:
    - pharmacy_id + sale_date + codigo_nacional + product_name + quantity + total_amount

    Comportamiento:
    - Para cada grupo de duplicados, mantiene el registro MÁS ANTIGUO (earliest created_at)
    - Elimina todos los demás registros del grupo
    - Respeta la integridad referencial con sales_enrichment (cascade delete)

    Args:
        pharmacy_id: ID de la farmacia específica (opcional, por defecto: todas las farmacias)
        request: Request object para rate limiting
        current_user: Usuario autenticado con permisos de admin
        db: Sesión de base de datos

    Returns:
        Estadísticas de la operación:
        - duplicates_found: Número de grupos de duplicados encontrados
        - records_deleted: Número total de registros eliminados
        - groups_processed: Número de grupos procesados exitosamente

    Raises:
        HTTPException: Si hay errores durante la operación
    """
    import time

    from sqlalchemy import func

    start_time = time.time()

    try:
        # Importar audit service
        from app.services.audit_service import AuditService

        audit_service = AuditService(db)

        struct_logger.info(
            "[CLEANUP_DUPLICATES] Iniciando limpieza de duplicados",
            extra={"pharmacy_id": pharmacy_id, "user_email": current_user.email, "timestamp": utc_now().isoformat()},
        )

        # Query base para encontrar duplicados
        duplicate_query = (
            db.query(
                SalesData.pharmacy_id,
                SalesData.sale_date,
                SalesData.codigo_nacional,
                SalesData.product_name,
                SalesData.quantity,
                SalesData.total_amount,
                func.count(SalesData.id).label("count"),
                func.min(SalesData.created_at).label("oldest_created_at"),
            )
            .group_by(
                SalesData.pharmacy_id,
                SalesData.sale_date,
                SalesData.codigo_nacional,
                SalesData.product_name,
                SalesData.quantity,
                SalesData.total_amount,
            )
            .having(func.count(SalesData.id) > 1)
        )

        # Filtrar por farmacia si se especifica
        if pharmacy_id:
            pharmacy = db.query(Pharmacy).filter(Pharmacy.id == pharmacy_id).first()
            if not pharmacy:
                raise HTTPException(status_code=404, detail=f"No se encontró la farmacia con ID: {pharmacy_id}")
            duplicate_query = duplicate_query.filter(SalesData.pharmacy_id == pharmacy_id)
            struct_logger.info(f"[CLEANUP_DUPLICATES] Filtrando por farmacia: {pharmacy.name}")

        # Obtener grupos de duplicados
        duplicate_groups = duplicate_query.all()
        duplicates_found = len(duplicate_groups)

        struct_logger.info(
            f"[CLEANUP_DUPLICATES] Encontrados {duplicates_found} grupos de duplicados",
            extra={"duplicates_found": duplicates_found},
        )

        if duplicates_found == 0:
            return {
                "success": True,
                "message": "No se encontraron duplicados",
                "duplicates_found": 0,
                "records_deleted": 0,
                "groups_processed": 0,
                "processing_time_seconds": round(time.time() - start_time, 2),
            }

        # Procesar cada grupo de duplicados
        groups_processed = 0
        records_deleted = 0
        errors = []

        for group in duplicate_groups:
            try:
                # Obtener todos los registros del grupo
                group_records = (
                    db.query(SalesData)
                    .filter(
                        SalesData.pharmacy_id == group.pharmacy_id,
                        SalesData.sale_date == group.sale_date,
                        SalesData.codigo_nacional == group.codigo_nacional,
                        SalesData.product_name == group.product_name,
                        SalesData.quantity == group.quantity,
                        SalesData.total_amount == group.total_amount,
                    )
                    .order_by(SalesData.created_at.asc())
                    .all()
                )

                # Mantener el primero (más antiguo), eliminar el resto
                if len(group_records) > 1:
                    records_to_delete = group_records[1:]  # Todos excepto el primero

                    for record in records_to_delete:
                        # Cascade delete se encargará de sales_enrichment
                        db.delete(record)
                        records_deleted += 1

                    groups_processed += 1

                    # Log cada 100 grupos procesados
                    if groups_processed % 100 == 0:
                        struct_logger.info(
                            f"[CLEANUP_DUPLICATES] Progreso: {groups_processed}/{duplicates_found} grupos procesados",
                            extra={"groups_processed": groups_processed, "records_deleted": records_deleted},
                        )
                        # Commit parcial cada 100 grupos para evitar transacciones muy largas
                        db.commit()

            except Exception as e:
                error_msg = f"Error procesando grupo: {str(e)}"
                errors.append(error_msg)
                struct_logger.error(
                    "[CLEANUP_DUPLICATES] Error procesando grupo de duplicados",
                    extra={
                        "error": str(e),
                        "group_info": {
                            "pharmacy_id": str(group.pharmacy_id),
                            "sale_date": str(group.sale_date),
                            "codigo_nacional": group.codigo_nacional,
                            "product_name": group.product_name,
                        },
                    },
                )
                # Continuar con el siguiente grupo
                continue

        # Commit final
        db.commit()

        processing_time = round(time.time() - start_time, 2)

        # Log en audit trail
        from app.models.audit_log import AuditAction

        audit_service.log_action(
            action=AuditAction.ADMIN_ACTION,
            method="POST",
            endpoint="/api/v1/admin/cleanup-duplicates",
            user=current_user,
            request=request,
            resource_type="sales_data",
            resource_id=pharmacy_id if pharmacy_id else "all",
            description=f"Limpieza de duplicados: {records_deleted} registros eliminados de {groups_processed} grupos",
            details={
                "pharmacy_id": pharmacy_id,
                "duplicates_found": duplicates_found,
                "groups_processed": groups_processed,
                "records_deleted": records_deleted,
                "processing_time_seconds": processing_time,
                "errors": errors if errors else None,
            },
            success=True,
            status_code=200,
            duration_ms=processing_time * 1000,
        )

        struct_logger.info(
            "[CLEANUP_DUPLICATES] Limpieza completada exitosamente",
            extra={
                "duplicates_found": duplicates_found,
                "groups_processed": groups_processed,
                "records_deleted": records_deleted,
                "processing_time_seconds": processing_time,
                "errors_count": len(errors),
            },
        )

        result = {
            "success": True,
            "message": f"Limpieza completada: {records_deleted} registros duplicados eliminados",
            "duplicates_found": duplicates_found,
            "groups_processed": groups_processed,
            "records_deleted": records_deleted,
            "processing_time_seconds": processing_time,
        }

        if errors:
            result["errors"] = errors
            result["partial_success"] = True

        return result

    except HTTPException:
        # Re-raise HTTP exceptions
        raise
    except Exception as e:
        db.rollback()
        processing_time = round(time.time() - start_time, 2)

        struct_logger.error(
            "[CLEANUP_DUPLICATES] Error durante limpieza de duplicados",
            extra={
                "error": str(e),
                "error_type": type(e).__name__,
                "pharmacy_id": pharmacy_id,
                "processing_time_seconds": processing_time,
            },
        )

        # Log error en audit trail
        from app.models.audit_log import AuditAction
        from app.services.audit_service import AuditService

        audit_service = AuditService(db)
        audit_service.log_action(
            action=AuditAction.ADMIN_ACTION,
            method="POST",
            endpoint="/api/v1/admin/cleanup-duplicates",
            user=current_user,
            request=request,
            resource_type="sales_data",
            resource_id=pharmacy_id if pharmacy_id else "all",
            description="Error durante limpieza de duplicados",
            details={"error": str(e), "error_type": type(e).__name__},
            success=False,
            status_code=500,
            error_message=str(e),
            duration_ms=processing_time * 1000,
        )

        raise HTTPException(status_code=500, detail=f"Error durante limpieza de duplicados: {str(e)}")


@router.post("/test/create-alert")
def create_test_alert(
    alert_type: str,
    severity: Literal["low", "medium", "high", "critical"],
    title: str,
    message: str,
    current_user: User = Depends(require_permissions("admin")),
    db: Session = Depends(get_db),
) -> Dict:
    """
    Crea una alerta de prueba para testing del sistema.
    Solo para entornos de desarrollo.
    """
    import os

    if os.getenv("ENVIRONMENT") == "production":
        raise HTTPException(status_code=403, detail="No disponible en producción")

    try:
        health_service = HealthMonitoringService(db)
        alert = health_service.create_alert(
            alert_type=alert_type,
            severity=severity,
            title=title,
            message=message,
            context={"test": True, "created_via": "api"},
        )

        return {
            "success": True,
            "alert_id": str(alert.id),
            "message": "Alerta de prueba creada correctamente",
        }

    except Exception as e:
        struct_logger.error("Error creating test alert", error=str(e))
        raise HTTPException(status_code=500, detail=f"Error creando alerta de prueba: {str(e)}")


# ===============================================
# CLASSIFIER MONITORING (Issue #458 - M6)
# ===============================================


@router.get("/classifier/metrics")
def get_classifier_metrics(
    current_user: User = Depends(get_current_admin_user),
    db: Session = Depends(get_db),
) -> Dict:
    """
    Obtiene métricas actuales del clasificador NECESIDAD.

    Incluye:
    - Entropía de clusters (distribución)
    - Tasa de outliers
    - Confianza promedio del modelo
    - Productos pendientes de validación
    - Decay del modelo (últimos 7 días)
    - Alertas activas
    """
    try:
        from app.services.classifier_monitoring_service import ClassifierMonitoringService

        monitoring_service = ClassifierMonitoringService(db)
        snapshot = monitoring_service.take_snapshot()

        return {
            "success": True,
            "metrics": snapshot,
            "timestamp": snapshot.get("timestamp"),
        }

    except Exception as e:
        struct_logger.error("Error getting classifier metrics", error=str(e))
        raise HTTPException(status_code=500, detail=f"Error obteniendo métricas: {str(e)}")


@router.get("/classifier/metrics/history")
def get_classifier_metrics_history(
    days: int = Query(default=30, le=90),
    metrics: Optional[str] = Query(None, description="Métricas separadas por coma"),
    current_user: User = Depends(get_current_admin_user),
    db: Session = Depends(get_db),
) -> Dict:
    """
    Obtiene historial de métricas del clasificador para gráficas de tendencias.
    """
    try:
        from app.services.classifier_monitoring_service import ClassifierMonitoringService

        monitoring_service = ClassifierMonitoringService(db)

        metric_names = metrics.split(",") if metrics else None
        history = monitoring_service.get_metrics_history(
            days=days,
            metric_names=metric_names,
        )

        return {
            "success": True,
            "history": history,
            "days": days,
        }

    except Exception as e:
        struct_logger.error("Error getting classifier metrics history", error=str(e))
        raise HTTPException(status_code=500, detail=f"Error obteniendo historial: {str(e)}")


@router.get("/classifier/distribution")
def get_category_distribution(
    current_user: User = Depends(get_current_admin_user),
    db: Session = Depends(get_db),
) -> Dict:
    """
    Obtiene distribución de productos por categoría NECESIDAD.
    """
    try:
        from app.services.classifier_monitoring_service import ClassifierMonitoringService

        monitoring_service = ClassifierMonitoringService(db)
        distribution = monitoring_service.get_category_distribution()

        return {
            "success": True,
            "categories": distribution,
            "total_categories": len(distribution),
        }

    except Exception as e:
        struct_logger.error("Error getting category distribution", error=str(e))
        raise HTTPException(status_code=500, detail=f"Error obteniendo distribución: {str(e)}")


@router.get("/classifier/high-risk")
def get_high_risk_products(
    limit: int = Query(default=20, le=100),
    z_threshold: float = Query(default=2.5),
    current_user: User = Depends(get_current_admin_user),
    db: Session = Depends(get_db),
) -> Dict:
    """
    Obtiene productos de alto riesgo (outliers) para revisión prioritaria.
    """
    try:
        from app.services.classifier_monitoring_service import ClassifierMonitoringService

        monitoring_service = ClassifierMonitoringService(db)
        products = monitoring_service.get_high_risk_products(
            limit=limit,
            z_score_threshold=z_threshold,
        )

        return {
            "success": True,
            "products": products,
            "count": len(products),
            "z_threshold": z_threshold,
        }

    except Exception as e:
        struct_logger.error("Error getting high risk products", error=str(e))
        raise HTTPException(status_code=500, detail=f"Error obteniendo productos: {str(e)}")


@router.post("/classifier/snapshot")
def take_classifier_snapshot(
    current_user: User = Depends(require_permissions("admin")),
    db: Session = Depends(get_db),
) -> Dict:
    """
    Fuerza la captura de un snapshot de métricas del clasificador.
    Guarda las métricas en SystemHealthMetric para historial.
    """
    try:
        from app.services.classifier_monitoring_service import ClassifierMonitoringService

        monitoring_service = ClassifierMonitoringService(db)
        snapshot = monitoring_service.take_snapshot()
        metrics = monitoring_service.record_metrics(snapshot)

        return {
            "success": True,
            "message": "Snapshot capturado y guardado",
            "metrics_recorded": len(metrics),
            "snapshot": snapshot,
        }

    except Exception as e:
        struct_logger.error("Error taking classifier snapshot", error=str(e))
        raise HTTPException(status_code=500, detail=f"Error capturando snapshot: {str(e)}")


@router.get("/classifier/alerts")
def get_classifier_alerts(
    include_resolved: bool = Query(default=False),
    days: int = Query(default=30, le=90),
    current_user: User = Depends(get_current_admin_user),
    db: Session = Depends(get_db),
) -> Dict:
    """
    Obtiene alertas del clasificador (activas e historial).
    """
    try:
        from app.services.clustering_alert_service import ClusteringAlertService

        alert_service = ClusteringAlertService(db)

        if include_resolved:
            alerts = alert_service.get_alert_history(
                days=days,
                include_resolved=True,
            )
        else:
            alerts = alert_service.get_active_alerts()

        return {
            "success": True,
            "alerts": alerts,
            "count": len(alerts),
            "active_count": alert_service.get_active_alerts_count(),
        }

    except Exception as e:
        struct_logger.error("Error getting classifier alerts", error=str(e))
        raise HTTPException(status_code=500, detail=f"Error obteniendo alertas: {str(e)}")


@router.post("/classifier/alerts/evaluate")
def evaluate_classifier_alerts(
    current_user: User = Depends(require_permissions("admin")),
    db: Session = Depends(get_db),
) -> Dict:
    """
    Evalúa métricas actuales y genera/actualiza alertas.
    """
    try:
        from app.services.clustering_alert_service import ClusteringAlertService

        alert_service = ClusteringAlertService(db)
        new_alerts = alert_service.evaluate_and_create_alerts()

        return {
            "success": True,
            "new_alerts": new_alerts,
            "active_count": alert_service.get_active_alerts_count(),
        }

    except Exception as e:
        struct_logger.error("Error evaluating classifier alerts", error=str(e))
        raise HTTPException(status_code=500, detail=f"Error evaluando alertas: {str(e)}")


@router.post("/classifier/alerts/{alert_id}/acknowledge")
def acknowledge_classifier_alert(
    alert_id: str,
    current_user: User = Depends(require_permissions("admin")),
    db: Session = Depends(get_db),
) -> Dict:
    """
    Reconoce una alerta del clasificador.
    """
    try:
        from app.services.clustering_alert_service import ClusteringAlertService

        alert_service = ClusteringAlertService(db)
        success = alert_service.acknowledge_alert(
            alert_id=alert_id,
            user_id=current_user.id,
        )

        if not success:
            raise HTTPException(status_code=404, detail="Alerta no encontrada o ya reconocida")

        return {
            "success": True,
            "message": "Alerta reconocida correctamente",
        }

    except HTTPException:
        raise
    except Exception as e:
        struct_logger.error("Error acknowledging classifier alert", error=str(e))
        raise HTTPException(status_code=500, detail=f"Error reconociendo alerta: {str(e)}")


@router.post("/classifier/alerts/{alert_id}/resolve")
def resolve_classifier_alert(
    alert_id: str,
    current_user: User = Depends(require_permissions("admin")),
    db: Session = Depends(get_db),
) -> Dict:
    """
    Resuelve manualmente una alerta del clasificador.
    """
    try:
        from app.services.clustering_alert_service import ClusteringAlertService

        alert_service = ClusteringAlertService(db)
        success = alert_service.resolve_alert(
            alert_id=alert_id,
            user_id=current_user.id,
        )

        if not success:
            raise HTTPException(status_code=404, detail="Alerta no encontrada o ya resuelta")

        return {
            "success": True,
            "message": "Alerta resuelta correctamente",
        }

    except HTTPException:
        raise
    except Exception as e:
        struct_logger.error("Error resolving classifier alert", error=str(e))
        raise HTTPException(status_code=500, detail=f"Error resolviendo alerta: {str(e)}")


@router.get("/classifier/precision")
def get_classifier_precision(
    days: int = Query(default=30, ge=1, le=365, description="Ventana temporal en días (1-365)"),
    db: Session = Depends(get_db),
    current_user: User = Depends(get_current_admin_user),  # ✅ SECURITY FIX: Admin required
):
    """
    Obtener precisión del clasificador basada en correcciones humanas (Issue #465).

    Precision = (predicciones correctas) / (total evaluadas)

    Args:
        days: Ventana temporal en días (default 30, range 1-365)

    Returns:
        Dict con precision, approve_count, evaluated_count
    """
    try:
        from app.services.classifier_monitoring_service import ClassifierMonitoringService

        monitoring_service = ClassifierMonitoringService(db)
        precision, approve_count, evaluated_count = monitoring_service.calculate_precision(days)

        return {
            "precision": precision,
            # Fix: precision 0.0 es válido (0%), solo N/A si no hay datos evaluados
            "precision_pct": f"{precision:.1%}" if evaluated_count > 0 else "N/A",
            "approve_count": approve_count,
            "evaluated_count": evaluated_count,
            "days": days,
            "status": (
                "healthy" if precision >= 0.85
                else "warning" if precision >= 0.70
                else "critical" if evaluated_count > 0
                else "no_data"
            ),
        }
    except Exception as e:
        struct_logger.error("Error getting classifier precision", error=str(e))
        raise HTTPException(status_code=500, detail=f"Error obteniendo precisión: {str(e)}")


@router.get("/classifier/drift")
def get_classifier_drift(
    days_current: int = Query(default=7, ge=1, le=90, description="Días semana actual (1-90)"),
    days_previous: int = Query(default=7, ge=1, le=90, description="Días semana anterior (1-90)"),
    db: Session = Depends(get_db),
    current_user: User = Depends(get_current_admin_user),  # ✅ SECURITY FIX: Admin required
):
    """
    Obtener drift de distribución de categorías (Issue #466).

    Compara distribución de categorías de esta semana vs la anterior.

    Args:
        days_current: Días de la semana actual (default 7, range 1-90)
        days_previous: Días de la semana anterior (default 7, range 1-90)

    Returns:
        Dict con has_drift, max_drift, drifts por categoría
    """
    try:
        from app.services.classifier_monitoring_service import ClassifierMonitoringService

        monitoring_service = ClassifierMonitoringService(db)
        drift_data = monitoring_service.calculate_distribution_drift(days_current, days_previous)

        return {
            **drift_data,
            "status": (
                "critical" if drift_data["max_drift"] > 0.20
                else "warning" if drift_data["has_drift"]
                else "healthy"
            ),
        }
    except Exception as e:
        struct_logger.error("Error getting classifier drift", error=str(e))
        raise HTTPException(status_code=500, detail=f"Error obteniendo drift: {str(e)}")
