"""add_pharmacy_lab_sales_summary_materialized_view

Revision ID: f3cc8610607e
Revises: 99538dc6edb3
Create Date: 2025-11-06 20:47:56.974966+01:00

MATERIALIZED VIEW para optimizar queries de partners (Issue #XXX).

Pre-calcula ventas agregadas por (pharmacy_id, laboratory) de los últimos 13 meses.
Reduce query time de /initialize de 180s → < 2s.

Refresh strategy: CONCURRENTLY para no bloquear lecturas (requiere UNIQUE index).
Refresh schedule: Diario via background job (suficiente para partners suggestion).

Performance impact:
- View size: ~5k rows (100 pharmacies × ~50 labs)
- Refresh time: ~5-10s (acceptable para job nocturno)
- Query speedup: 100x (elimina 3-way JOIN sobre 650k+ sales rows)
"""
from typing import Sequence, Union

from alembic import op
import sqlalchemy as sa


# revision identifiers, used by Alembic.
revision: str = 'f3cc8610607e'
down_revision: Union[str, None] = '99538dc6edb3'
branch_labels: Union[str, Sequence[str], None] = None
depends_on: Union[str, Sequence[str], None] = None


def upgrade() -> None:
    """
    Crear materialized view pharmacy_lab_sales_summary.

    Pre-calcula ventas por (pharmacy_id, laboratory) de últimos 13 meses.
    Incluye UNIQUE index para soporte de REFRESH CONCURRENTLY.
    """
    conn = op.get_bind()

    # ✅ IDEMPOTENT: Drop view si existe
    conn.execute(sa.text("DROP MATERIALIZED VIEW IF EXISTS pharmacy_lab_sales_summary CASCADE"))

    # Crear materialized view con query optimizado
    conn.execute(sa.text("""
        CREATE MATERIALIZED VIEW pharmacy_lab_sales_summary AS
        SELECT
            sd.pharmacy_id,
            pc.nomen_laboratorio AS laboratory_name,
            SUM(sd.total_amount) AS total_sales,
            SUM(sd.quantity) AS total_quantity,
            COUNT(sd.id) AS total_transactions,
            MAX(sd.sale_date) AS last_sale_date,
            -- Metadata para debugging/monitoring
            NOW() AS view_refreshed_at
        FROM sales_data sd
        INNER JOIN sales_enrichment se ON se.sales_data_id = sd.id
        INNER JOIN product_catalog pc ON pc.id = se.product_catalog_id
        WHERE
            -- Solo últimos 13 meses (filtro dinámico en query, no en view)
            sd.sale_date >= (NOW() - INTERVAL '13 months')
            AND pc.nomen_estado = 'ALTA'  -- Solo productos activos
            AND pc.nomen_laboratorio IS NOT NULL  -- Excluir nulls
        GROUP BY sd.pharmacy_id, pc.nomen_laboratorio
        HAVING SUM(sd.total_amount) > 0  -- Excluir labs sin ventas
    """))

    # ✅ CRITICAL: UNIQUE index para REFRESH CONCURRENTLY
    # Sin esto, REFRESH bloquea lecturas (inaceptable en producción)
    # IF NOT EXISTS garantiza idempotencia (REGLA #14)
    conn.execute(sa.text("""
        CREATE UNIQUE INDEX IF NOT EXISTS idx_pharmacy_lab_sales_summary_unique
        ON pharmacy_lab_sales_summary (pharmacy_id, laboratory_name)
    """))

    # Índice adicional para query /initialize (ORDER BY total_sales DESC LIMIT 8)
    # IF NOT EXISTS garantiza idempotencia (REGLA #14)
    conn.execute(sa.text("""
        CREATE INDEX IF NOT EXISTS idx_pharmacy_lab_sales_summary_sales
        ON pharmacy_lab_sales_summary (pharmacy_id, total_sales DESC)
    """))

    # Initial refresh (poblar view con datos actuales)
    conn.execute(sa.text("REFRESH MATERIALIZED VIEW pharmacy_lab_sales_summary"))


def downgrade() -> None:
    """
    Eliminar materialized view y sus índices.

    Rollback seguro: Los queries volverán a usar el JOIN directo (lento pero funcional).
    """
    conn = op.get_bind()

    # CASCADE elimina automáticamente los índices
    conn.execute(sa.text("DROP MATERIALIZED VIEW IF EXISTS pharmacy_lab_sales_summary CASCADE"))
