"""revert_to_1_1_user_pharmacy_relationship

Restore 1:1 User-Pharmacy relationship (fixes Issue #225)

This migration reverts the incorrect change made in edb51a5211e1 which
changed the architecture from 1:1 to 1:N User-Pharmacy relationship.

According to DATA_CATALOG.md (lines 220-249, 251-286), the correct design
is 1:1: each pharmacy has exactly ONE user.

SAFETY: This migration will fail if there are multiple users per pharmacy.
Clean duplicates manually first using:
  SELECT pharmacy_id, COUNT(*) FROM users GROUP BY pharmacy_id HAVING COUNT(*) > 1;

Revision ID: f2610595fadb
Revises: edb51a5211e1
Create Date: 2025-10-11 10:48:13.527844+02:00

"""
from typing import Sequence, Union
import logging

from alembic import op
from sqlalchemy import text

# Configure logging
logger = logging.getLogger(__name__)

# revision identifiers, used by Alembic.
revision: str = 'f2610595fadb'
down_revision: Union[str, None] = '000_initial_schema'
branch_labels: Union[str, Sequence[str], None] = None
depends_on: Union[str, Sequence[str], None] = None


def upgrade() -> None:
    """
    Restore 1:1 User-Pharmacy relationship.

    SAFETY: This migration will fail if there are multiple users
    per pharmacy in the database. Clean duplicates manually first.
    """
    # Pre-flight check: Verify no duplicate users per pharmacy
    conn = op.get_bind()

    # Check if users table exists (fresh DB won't have it yet from previous migration)
    table_check = conn.execute(text("""
        SELECT EXISTS (
            SELECT FROM information_schema.tables
            WHERE table_schema = 'public'
            AND table_name = 'users'
        )
    """))
    table_exists = table_check.scalar()

    if not table_exists:
        # Fresh database - table was just created by previous migration
        # No need to check duplicates or modify index - it's already correct
        logger.info("Fresh database detected - users table doesn't exist yet, skipping duplicate check")
        return

    # Check for duplicates
    result = conn.execute(text("""
        SELECT pharmacy_id, COUNT(*) as user_count
        FROM users
        WHERE pharmacy_id IS NOT NULL
        GROUP BY pharmacy_id
        HAVING COUNT(*) > 1
    """))

    duplicates = list(result)

    if duplicates:
        # Log detailed information about duplicates
        logger.error(f"Cannot upgrade: Found {len(duplicates)} pharmacies with multiple users")
        for pharmacy_id, count in duplicates:
            logger.error(f"  Pharmacy {pharmacy_id}: {count} users")

        # Get more details about the duplicate users
        for pharmacy_id, count in duplicates:
            users_result = conn.execute(text("""
                SELECT id, email, created_at
                FROM users
                WHERE pharmacy_id = :pharmacy_id
                ORDER BY created_at
            """), {"pharmacy_id": pharmacy_id})

            users = list(users_result)
            logger.error(f"  Users in pharmacy {pharmacy_id}:")
            for user_id, email, created_at in users:
                logger.error(f"    - User {user_id}: {email} (created: {created_at})")

        raise Exception(
            f"Cannot upgrade: Found {len(duplicates)} pharmacies with multiple users. "
            f"This violates the 1:1 User-Pharmacy relationship. "
            f"Please clean duplicates manually before running this migration. "
            f"Run this query to identify duplicates: "
            f"SELECT pharmacy_id, COUNT(*) FROM users GROUP BY pharmacy_id HAVING COUNT(*) > 1"
        )

    logger.info("Pre-flight check passed: No duplicate users per pharmacy found")

    # Drop the existing index (non-unique)
    op.drop_index(op.f('ix_users_pharmacy_id'), table_name='users')

    # Recreate index with unique=True (1:1 relationship)
    op.create_index(
        op.f('ix_users_pharmacy_id'),
        'users',
        ['pharmacy_id'],
        unique=True
    )

    logger.info("Successfully restored 1:1 User-Pharmacy relationship with UNIQUE constraint")


def downgrade() -> None:
    """
    Downgrade back to 1:N relationship (not recommended).

    ⚠️  CRITICAL WARNING: This downgrade restores the INCORRECT architecture!

    The 1:1 User-Pharmacy relationship is the correct design according to
    DATA_CATALOG.md. This downgrade should only be used for emergency rollback
    and the issue should be fixed immediately afterward.

    Downgrading will:
    - Remove the UNIQUE constraint that ensures data integrity
    - Allow multiple users per pharmacy (violating the design)
    - Potentially cause application errors that expect 1:1 relationship
    """
    logger.warning(
        "⚠️  CRITICAL: Downgrading to INCORRECT 1:N User-Pharmacy relationship! "
        "This violates the documented architecture in DATA_CATALOG.md. "
        "Only use for emergency rollback and fix immediately."
    )

    # Drop unique index
    op.drop_index(op.f('ix_users_pharmacy_id'), table_name='users')

    # Recreate non-unique index (allows duplicates - INCORRECT)
    op.create_index(
        op.f('ix_users_pharmacy_id'),
        'users',
        ['pharmacy_id'],
        unique=False
    )

    logger.warning(
        "Downgrade completed: UNIQUE constraint removed (INCORRECT STATE). "
        "The system is now in a state that violates the 1:1 design. "
        "Please fix this as soon as possible."
    )
