"""add_laboratory_index_for_partner_queries

Optimización crítica para queries de análisis de partners.

Issue: Performance degradation en producción
- Endpoint /api/v1/analysis/partner-dynamic timeout (2.5s)
- Filtro por nomen_codigo_laboratorio sin índice dedicado
- Full table scan en 72k registros

Impacto esperado: -30% tiempo de query

Revision ID: e3b7179c3f48
Revises: 50c50579fb27
Create Date: 2025-11-03 00:31:04.132934+01:00

"""
from typing import Sequence, Union

from alembic import op
import sqlalchemy as sa


# revision identifiers, used by Alembic.
revision: str = 'e3b7179c3f48'
down_revision: Union[str, None] = '50c50579fb27'
branch_labels: Union[str, Sequence[str], None] = None
depends_on: Union[str, Sequence[str], None] = None


def upgrade() -> None:
    """
    Crea índice para optimizar filtros por código de laboratorio en queries de partners.

    IMPORTANTE: Usa índice normal (NO CONCURRENTLY) en desarrollo por limitación de Alembic.
    En producción, este índice puede crearse CONCURRENTLY vía SQL directo.

    Partial index: Solo productos activos (reduce tamaño ~30%)
    """
    # Verificar si el índice ya existe
    conn = op.get_bind()
    result = conn.execute(sa.text(
        "SELECT indexname FROM pg_indexes "
        "WHERE tablename = 'product_catalog' "
        "AND indexname = 'idx_nomen_codigo_laboratorio'"
    ))

    if not result.fetchone():
        # Crear índice (normal en desarrollo, aplicar CONCURRENTLY manual en producción)
        op.create_index(
            'idx_nomen_codigo_laboratorio',
            'product_catalog',
            ['nomen_codigo_laboratorio'],
            unique=False,
            postgresql_where=sa.text("nomen_estado = 'ALTA'")
        )


def downgrade() -> None:
    """
    Elimina el índice de forma segura.
    """
    # Verificar si el índice existe antes de eliminar
    conn = op.get_bind()
    result = conn.execute(sa.text(
        "SELECT indexname FROM pg_indexes "
        "WHERE tablename = 'product_catalog' "
        "AND indexname = 'idx_nomen_codigo_laboratorio'"
    ))

    if result.fetchone():
        op.drop_index('idx_nomen_codigo_laboratorio', table_name='product_catalog')
