"""drop_unique_constraint_sales_data

Revision ID: df009b59a1e6
Revises: fc38b8d391de
Create Date: 2025-10-26 21:01:17.771856+01:00

"""
from typing import Sequence, Union

from alembic import op
import sqlalchemy as sa


# revision identifiers, used by Alembic.
revision: str = 'df009b59a1e6'
down_revision: Union[str, None] = 'fc38b8d391de'
branch_labels: Union[str, Sequence[str], None] = None
depends_on: Union[str, Sequence[str], None] = None


def upgrade() -> None:
    """
    Drop UNIQUE constraint uq_sales_data_composite_v2 from sales_data table.

    RATIONALE: El constraint UNIQUE bloqueaba ventas legítimas que parecen duplicadas
    (mismo producto, timestamp, cantidad, precio). La detección de duplicados se maneja
    ahora exclusivamente por ventana temporal, que es más flexible.

    IDEMPOTENCIA: Verifica que el constraint exista antes de eliminarlo.
    """
    conn = op.get_bind()

    # Verificar si el constraint existe
    result = conn.execute(sa.text("""
        SELECT constraint_name
        FROM information_schema.table_constraints
        WHERE table_name = 'sales_data'
        AND constraint_name = 'uq_sales_data_composite_v2'
        AND constraint_type = 'UNIQUE'
    """))

    if result.fetchone():
        # El constraint existe, eliminarlo
        op.drop_constraint('uq_sales_data_composite_v2', 'sales_data', type_='unique')
        print("✓ Constraint uq_sales_data_composite_v2 eliminado correctamente")
    else:
        print("✓ Constraint uq_sales_data_composite_v2 ya no existe (skip)")


def downgrade() -> None:
    """
    Recrear UNIQUE constraint uq_sales_data_composite_v2 en sales_data table.

    WARNING: Este downgrade puede fallar si existen duplicados en la base de datos.
    Si la migración forward ya se ejecutó y hay datos duplicados, no se puede revertir.

    IDEMPOTENCIA: Verifica que el constraint NO exista antes de crearlo.
    """
    conn = op.get_bind()

    # Verificar si el constraint ya existe
    result = conn.execute(sa.text("""
        SELECT constraint_name
        FROM information_schema.table_constraints
        WHERE table_name = 'sales_data'
        AND constraint_name = 'uq_sales_data_composite_v2'
        AND constraint_type = 'UNIQUE'
    """))

    if not result.fetchone():
        # El constraint no existe, crearlo
        try:
            op.create_unique_constraint(
                'uq_sales_data_composite_v2',
                'sales_data',
                ['pharmacy_id', 'sale_date', 'sale_time', 'codigo_nacional', 'product_name', 'quantity', 'total_amount']
            )
            print("✓ Constraint uq_sales_data_composite_v2 recreado correctamente")
        except Exception as e:
            print(f"✗ ERROR: No se pudo recrear el constraint (probablemente hay duplicados): {e}")
            print("  Para revertir manualmente, primero elimina duplicados en sales_data")
            raise
    else:
        print("✓ Constraint uq_sales_data_composite_v2 ya existe (skip)")
