"""add_product_corrections_table_issue_457

Revision ID: d9e81c815291
Revises: 20251218_01_llm_pipeline
Create Date: 2025-12-19 12:32:43.034760+01:00

Issue #457: M4 Feedback Loop - Human-in-the-Loop
Modelo ProductCorrection para almacenar correcciones humanas del clasificador.
Permite aprendizaje continuo y mining de reglas automáticas.
"""
from typing import Sequence, Union

from alembic import op
import sqlalchemy as sa
from sqlalchemy.dialects import postgresql

# revision identifiers, used by Alembic.
revision: str = 'd9e81c815291'
down_revision: Union[str, None] = '20251218_01_llm_pipeline'
branch_labels: Union[str, Sequence[str], None] = None
depends_on: Union[str, Sequence[str], None] = None


def upgrade() -> None:
    # Crear enums para tipos de corrección y fuentes de predicción
    correction_type_enum = postgresql.ENUM(
        'approve', 'correct', 'outlier',
        name='correction_type_enum',
        create_type=False
    )
    prediction_source_enum = postgresql.ENUM(
        'tier1_specific', 'tier1_generic', 'tier2_rules', 'brand', 'llm', 'fallback',
        name='prediction_source_enum',
        create_type=False
    )

    # Crear tipos si no existen
    conn = op.get_bind()

    # Check and create correction_type_enum
    result = conn.execute(sa.text(
        "SELECT 1 FROM pg_type WHERE typname = 'correction_type_enum'"
    ))
    if not result.fetchone():
        correction_type_enum.create(conn)

    # Check and create prediction_source_enum
    result = conn.execute(sa.text(
        "SELECT 1 FROM pg_type WHERE typname = 'prediction_source_enum'"
    ))
    if not result.fetchone():
        prediction_source_enum.create(conn)

    # Verificar si la tabla ya existe
    result = conn.execute(sa.text(
        "SELECT 1 FROM information_schema.tables WHERE table_name = 'product_corrections'"
    ))
    if result.fetchone():
        return  # Tabla ya existe, nada que hacer

    # Crear tabla product_corrections
    op.create_table(
        'product_corrections',
        sa.Column('id', sa.Integer(), nullable=False),
        sa.Column('sales_enrichment_id', postgresql.UUID(as_uuid=True), nullable=False),
        sa.Column('product_name', sa.String(length=500), nullable=False),
        sa.Column('predicted_category', sa.String(length=100), nullable=False),
        sa.Column('prediction_source', postgresql.ENUM(
            'tier1_specific', 'tier1_generic', 'tier2_rules', 'brand', 'llm', 'fallback',
            name='prediction_source_enum',
            create_type=False
        ), nullable=False),
        sa.Column('confidence_score', sa.Float(), nullable=False),
        sa.Column('correction_type', postgresql.ENUM(
            'approve', 'correct', 'outlier',
            name='correction_type_enum',
            create_type=False
        ), nullable=False),
        sa.Column('corrected_category', sa.String(length=100), nullable=True),
        sa.Column('reviewer_notes', sa.Text(), nullable=True),
        sa.Column('pharmacy_id', postgresql.UUID(as_uuid=True), nullable=True),
        sa.Column('created_at', sa.DateTime(timezone=True), nullable=False),
        sa.Column('processed_for_rules', sa.Boolean(), nullable=False, server_default='false'),
        sa.Column('processed_at', sa.DateTime(timezone=True), nullable=True),
        sa.Column('suggested_rule', sa.Text(), nullable=True),
        sa.ForeignKeyConstraint(['pharmacy_id'], ['pharmacies.id'], ondelete='SET NULL'),
        sa.ForeignKeyConstraint(['sales_enrichment_id'], ['sales_enrichment.id'], ondelete='CASCADE'),
        sa.PrimaryKeyConstraint('id')
    )

    # Crear índices
    op.create_index(
        op.f('ix_product_corrections_id'),
        'product_corrections',
        ['id'],
        unique=False
    )
    op.create_index(
        op.f('ix_product_corrections_sales_enrichment_id'),
        'product_corrections',
        ['sales_enrichment_id'],
        unique=False
    )


def downgrade() -> None:
    # Verificar si la tabla existe antes de intentar eliminarla
    conn = op.get_bind()
    result = conn.execute(sa.text(
        "SELECT 1 FROM information_schema.tables WHERE table_name = 'product_corrections'"
    ))
    if result.fetchone():
        op.drop_index(op.f('ix_product_corrections_sales_enrichment_id'), table_name='product_corrections')
        op.drop_index(op.f('ix_product_corrections_id'), table_name='product_corrections')
        op.drop_table('product_corrections')

    # No eliminamos los enums porque pueden ser usados por otras tablas en el futuro
    # y drop_type causa errores si hay dependencias
