"""add_prescription_indexes_issue_408

Issue #408 Code Review: Agregar índices para optimizar queries de prescripción.

Performance Improvement:
- idx_prescription_lookup: Optimiza query filter (nomen_codigo_homogeneo, cima_requiere_receta)
  Reduce tiempo de query de ~500ms a ~50ms en catálogo de 67k productos
- idx_prescription_category: Optimiza filtros por categoría asignada
  Útil para estadísticas y filtrado en admin dashboard

Revision ID: bca195003e19
Revises: 1931d756e755
Create Date: 2025-11-13 23:46:39.047440+01:00

"""
from typing import Sequence, Union

from alembic import op
import sqlalchemy as sa


# revision identifiers, used by Alembic.
revision: str = 'bca195003e19'
down_revision: Union[str, None] = '1931d756e755'
branch_labels: Union[str, Sequence[str], None] = None
depends_on: Union[str, Sequence[str], None] = None


def upgrade() -> None:
    """
    Agregar índices para optimizar queries de prescripción.

    REGLA #14: Migraciones idempotentes - Verificar existencia ANTES de crear.
    """
    conn = op.get_bind()

    # ===================================================================
    # ÍNDICE 1: idx_prescription_lookup
    # ===================================================================
    # Verifica si el índice ya existe
    result = conn.execute(sa.text(
        """
        SELECT indexname
        FROM pg_indexes
        WHERE tablename = 'product_catalog'
          AND indexname = 'idx_prescription_lookup'
        """
    ))

    if not result.fetchone():
        # Crear índice para lookup de productos de prescripción
        # Optimiza query: (nomen_codigo_homogeneo IS NOT NULL) OR (cima_requiere_receta = True)
        op.create_index(
            'idx_prescription_lookup',
            'product_catalog',
            ['nomen_codigo_homogeneo', 'cima_requiere_receta'],
            unique=False,
            postgresql_where=sa.text(
                "nomen_codigo_homogeneo IS NOT NULL OR cima_requiere_receta = true"
            )
        )
        print("[MIGRATION] ✅ Índice idx_prescription_lookup creado")
    else:
        print("[MIGRATION] ℹ️  Índice idx_prescription_lookup ya existe - skipping")

    # ===================================================================
    # ÍNDICE 2: idx_prescription_category
    # ===================================================================
    result = conn.execute(sa.text(
        """
        SELECT indexname
        FROM pg_indexes
        WHERE tablename = 'product_catalog'
          AND indexname = 'idx_prescription_category'
        """
    ))

    if not result.fetchone():
        # Crear índice para filtros por categoría de prescripción
        # Optimiza queries de estadísticas y filtrado en admin dashboard
        op.create_index(
            'idx_prescription_category',
            'product_catalog',
            ['xfarma_prescription_category'],
            unique=False
        )
        print("[MIGRATION] ✅ Índice idx_prescription_category creado")
    else:
        print("[MIGRATION] ℹ️  Índice idx_prescription_category ya existe - skipping")


def downgrade() -> None:
    """
    Eliminar índices de prescripción.

    REGLA #14: Downgrade también debe ser idempotente.
    """
    conn = op.get_bind()

    # Eliminar idx_prescription_category
    result = conn.execute(sa.text(
        """
        SELECT indexname
        FROM pg_indexes
        WHERE tablename = 'product_catalog'
          AND indexname = 'idx_prescription_category'
        """
    ))

    if result.fetchone():
        op.drop_index('idx_prescription_category', table_name='product_catalog')
        print("[MIGRATION] ✅ Índice idx_prescription_category eliminado")
    else:
        print("[MIGRATION] ℹ️  Índice idx_prescription_category no existe - skipping")

    # Eliminar idx_prescription_lookup
    result = conn.execute(sa.text(
        """
        SELECT indexname
        FROM pg_indexes
        WHERE tablename = 'product_catalog'
          AND indexname = 'idx_prescription_lookup'
        """
    ))

    if result.fetchone():
        op.drop_index('idx_prescription_lookup', table_name='product_catalog')
        print("[MIGRATION] ✅ Índice idx_prescription_lookup eliminado")
    else:
        print("[MIGRATION] ℹ️  Índice idx_prescription_lookup no existe - skipping")
