"""add_generics_analysis_composite_index

Optimización crítica para queries de análisis de genéricos sustituibles.

Issue: Performance degradation en producción
- Endpoint /api/v1/analysis/substitutable-universe timeout (2.8s)
- Full table scan en 72k productos por filtro compuesto:
  * nomen_tipo_farmaco = 'GENERICO'
  * nomen_estado = 'ALTA'
  * nomen_codigo_homogeneo IS NOT NULL

Impacto esperado: -40% tiempo de query

Revision ID: b8672b3dfbd9
Revises: e3b7179c3f48
Create Date: 2025-11-03 00:31:47.583331+01:00

"""
from typing import Sequence, Union

from alembic import op
import sqlalchemy as sa


# revision identifiers, used by Alembic.
revision: str = 'b8672b3dfbd9'
down_revision: Union[str, None] = 'e3b7179c3f48'
branch_labels: Union[str, Sequence[str], None] = None
depends_on: Union[str, Sequence[str], None] = None


def upgrade() -> None:
    """
    Crea índice compuesto para optimizar queries de análisis de genéricos.

    IMPORTANTE: Usa índice normal (NO CONCURRENTLY) en desarrollo por limitación de Alembic.
    En producción, este índice puede crearse CONCURRENTLY vía SQL directo.

    Partial index: Solo productos con código homogéneo válido (reduce tamaño ~50%)
    Covering index: Incluye las 3 columnas del filtro típico
    """
    # Verificar si el índice ya existe
    conn = op.get_bind()
    result = conn.execute(sa.text(
        "SELECT indexname FROM pg_indexes "
        "WHERE tablename = 'product_catalog' "
        "AND indexname = 'idx_generics_analysis'"
    ))

    if not result.fetchone():
        # Crear índice compuesto (normal en desarrollo, aplicar CONCURRENTLY manual en producción)
        op.create_index(
            'idx_generics_analysis',
            'product_catalog',
            ['nomen_tipo_farmaco', 'nomen_estado', 'nomen_codigo_homogeneo'],
            unique=False,
            postgresql_where=sa.text("nomen_codigo_homogeneo IS NOT NULL AND nomen_codigo_homogeneo != ''")
        )


def downgrade() -> None:
    """
    Elimina el índice de forma segura.
    """
    # Verificar si el índice existe antes de eliminar
    conn = op.get_bind()
    result = conn.execute(sa.text(
        "SELECT indexname FROM pg_indexes "
        "WHERE tablename = 'product_catalog' "
        "AND indexname = 'idx_generics_analysis'"
    ))

    if result.fetchone():
        op.drop_index('idx_generics_analysis', table_name='product_catalog')
