"""add_prescription_category_to_product_catalog

Revision ID: b609b997a76a
Revises: 20251101_01_fix_utf8
Create Date: 2025-11-02 10:47:54.871225+01:00

Issue #16 - Fase 1: Clasificación de Prescripción

Añade columna xfarma_prescription_category a product_catalog con:
- Enum PrescriptionCategory (14 valores posibles)
- Índice para queries optimizadas
- Permite NULL (clasificación manual posterior)
- Migración 100% idempotente (verifica existencia antes de crear)

Estrategia conservadora Fase 1:
- Solo schema changes (sin clasificación automática)
- Valores NULL por defecto
- Event listeners en Fase 3

"""
from typing import Sequence, Union

from alembic import op
import sqlalchemy as sa


# revision identifiers, used by Alembic.
revision: str = 'b609b997a76a'
down_revision: Union[str, None] = '20251101_01_fix_utf8'
branch_labels: Union[str, Sequence[str], None] = None
depends_on: Union[str, Sequence[str], None] = None


def upgrade() -> None:
    """
    Añade columna xfarma_prescription_category de forma idempotente

    Verificaciones:
    1. Tipo ENUM prescription_category_enum existe
    2. Columna xfarma_prescription_category existe
    3. Índice ix_product_catalog_prescription_category existe
    """
    conn = op.get_bind()

    # === PASO 1: Verificar/Crear tipo ENUM ===
    enum_exists = conn.execute(sa.text("""
        SELECT EXISTS (
            SELECT 1 FROM pg_type
            WHERE typname = 'prescription_category_enum'
        )
    """)).scalar()

    if not enum_exists:
        # Crear tipo ENUM con 14 valores
        op.execute("""
            CREATE TYPE prescription_category_enum AS ENUM (
                'formulas_magistrales',
                'vacunas_individualizadas',
                'tiras_reactivas_glucosa',
                'incontinencia_financiada',
                'efectos_financiados',
                'ortopedia_financiada',
                'dietoterapicos',
                'margen_especial_3',
                'margen_especial_2',
                'margen_especial_1',
                'conjunto_homogeneo',
                'medicamentos',
                'veterinaria',
                'uso_humano_no_financiado'
            )
        """)
        print("✅ Tipo ENUM 'prescription_category_enum' creado")
    else:
        print("⏭️  Tipo ENUM 'prescription_category_enum' ya existe (skip)")

    # === PASO 2: Verificar/Añadir columna ===
    column_exists = conn.execute(sa.text("""
        SELECT EXISTS (
            SELECT 1
            FROM information_schema.columns
            WHERE table_name = 'product_catalog'
              AND column_name = 'xfarma_prescription_category'
        )
    """)).scalar()

    if not column_exists:
        op.add_column(
            'product_catalog',
            sa.Column(
                'xfarma_prescription_category',
                sa.Enum(
                    'formulas_magistrales',
                    'vacunas_individualizadas',
                    'tiras_reactivas_glucosa',
                    'incontinencia_financiada',
                    'efectos_financiados',
                    'ortopedia_financiada',
                    'dietoterapicos',
                    'margen_especial_3',
                    'margen_especial_2',
                    'margen_especial_1',
                    'conjunto_homogeneo',
                    'medicamentos',
                    'veterinaria',
                    'uso_humano_no_financiado',
                    name='prescription_category_enum'
                ),
                nullable=True
            )
        )
        print("✅ Columna 'xfarma_prescription_category' añadida")
    else:
        print("⏭️  Columna 'xfarma_prescription_category' ya existe (skip)")

    # === PASO 3: Verificar/Crear índice ===
    index_exists = conn.execute(sa.text("""
        SELECT EXISTS (
            SELECT 1
            FROM pg_indexes
            WHERE tablename = 'product_catalog'
              AND indexname = 'ix_product_catalog_prescription_category'
        )
    """)).scalar()

    if not index_exists:
        op.create_index(
            'ix_product_catalog_prescription_category',
            'product_catalog',
            ['xfarma_prescription_category'],
            unique=False
        )
        print("✅ Índice 'ix_product_catalog_prescription_category' creado")
    else:
        print("⏭️  Índice 'ix_product_catalog_prescription_category' ya existe (skip)")

    # === VERIFICACIÓN FINAL ===
    result = conn.execute(sa.text("""
        SELECT
            c.column_name,
            c.data_type,
            c.is_nullable,
            i.indexname
        FROM information_schema.columns c
        LEFT JOIN pg_indexes i
            ON i.tablename = c.table_name
            AND i.indexname = 'ix_product_catalog_prescription_category'
        WHERE c.table_name = 'product_catalog'
          AND c.column_name = 'xfarma_prescription_category'
    """)).fetchone()

    if result:
        print("✅ Migración exitosa:")
        print(f"   - Columna: {result[0]}")
        print(f"   - Tipo: {result[1]}")
        print(f"   - Nullable: {result[2]}")
        print(f"   - Índice: {result[3] if result[3] else 'No creado'}")
    else:
        raise Exception("❌ ERROR: Columna no creada correctamente")


def downgrade() -> None:
    """
    Elimina columna xfarma_prescription_category de forma idempotente

    Orden de rollback:
    1. Eliminar índice
    2. Eliminar columna
    3. Eliminar tipo ENUM
    """
    conn = op.get_bind()

    # === PASO 1: Verificar/Eliminar índice ===
    index_exists = conn.execute(sa.text("""
        SELECT EXISTS (
            SELECT 1
            FROM pg_indexes
            WHERE tablename = 'product_catalog'
              AND indexname = 'ix_product_catalog_prescription_category'
        )
    """)).scalar()

    if index_exists:
        op.drop_index(
            'ix_product_catalog_prescription_category',
            table_name='product_catalog'
        )
        print("✅ Índice 'ix_product_catalog_prescription_category' eliminado")
    else:
        print("⏭️  Índice 'ix_product_catalog_prescription_category' no existe (skip)")

    # === PASO 2: Verificar/Eliminar columna ===
    column_exists = conn.execute(sa.text("""
        SELECT EXISTS (
            SELECT 1
            FROM information_schema.columns
            WHERE table_name = 'product_catalog'
              AND column_name = 'xfarma_prescription_category'
        )
    """)).scalar()

    if column_exists:
        op.drop_column('product_catalog', 'xfarma_prescription_category')
        print("✅ Columna 'xfarma_prescription_category' eliminada")
    else:
        print("⏭️  Columna 'xfarma_prescription_category' no existe (skip)")

    # === PASO 3: Verificar/Eliminar tipo ENUM ===
    # IMPORTANTE: Solo eliminar ENUM si no hay otras columnas usándolo
    enum_usage = conn.execute(sa.text("""
        SELECT COUNT(*)
        FROM information_schema.columns
        WHERE udt_name = 'prescription_category_enum'
    """)).scalar()

    if enum_usage == 0:
        enum_exists = conn.execute(sa.text("""
            SELECT EXISTS (
                SELECT 1 FROM pg_type
                WHERE typname = 'prescription_category_enum'
            )
        """)).scalar()

        if enum_exists:
            op.execute("DROP TYPE prescription_category_enum")
            print("✅ Tipo ENUM 'prescription_category_enum' eliminado")
        else:
            print("⏭️  Tipo ENUM 'prescription_category_enum' no existe (skip)")
    else:
        print(f"⚠️  Tipo ENUM 'prescription_category_enum' en uso por {enum_usage} columnas (skip)")

    print("✅ Rollback completado exitosamente")
