"""add_nomenclator_to_synctype_enum

Revision ID: 630f26655664
Revises: dfb6fdfab99f
Create Date: 2025-11-15 07:54:42.694269+01:00

"""
from typing import Sequence, Union

from alembic import op
import sqlalchemy as sa


# revision identifiers, used by Alembic.
revision: str = '630f26655664'
down_revision: Union[str, None] = 'dfb6fdfab99f'
branch_labels: Union[str, Sequence[str], None] = None
depends_on: Union[str, Sequence[str], None] = None


def upgrade() -> None:
    """
    Agrega 'NOMENCLATOR' al enum SyncType de forma idempotente.

    PROBLEMA:
    - El modelo Python define SyncType con CIMA, NOMENCLATOR, CIMAVET
    - La base de datos PostgreSQL solo tiene CIMA, CIMAVET
    - Error al intentar insertar registros con sync_type='NOMENCLATOR'

    SOLUCIÓN:
    - Verificar si el valor ya existe en el enum ANTES de agregarlo
    - Usar ALTER TYPE ADD VALUE IF NOT EXISTS (PostgreSQL 9.1+)

    REFERENCIA:
    - REGLA #14: Migraciones idempotentes (verificar existencia ANTES)
    - Issue: Invalid enum value 'NOMENCLATOR' en catalog_sync_history
    """
    # Verificar si el valor 'NOMENCLATOR' ya existe en el enum synctype
    conn = op.get_bind()
    result = conn.execute(sa.text(
        "SELECT EXISTS(SELECT 1 FROM pg_enum "
        "WHERE enumlabel = 'NOMENCLATOR' "
        "AND enumtypid = (SELECT oid FROM pg_type WHERE typname = 'synctype'))"
    ))

    nomenclator_exists = result.scalar()

    if not nomenclator_exists:
        # Agregar valor 'NOMENCLATOR' al enum synctype
        # NOTA: ALTER TYPE ADD VALUE no soporta IF NOT EXISTS en versiones < PostgreSQL 12
        # pero ya verificamos existencia arriba
        op.execute("ALTER TYPE synctype ADD VALUE 'NOMENCLATOR'")
        print("[MIGRATION] ✓ Agregado 'NOMENCLATOR' a enum synctype")
    else:
        print("[MIGRATION] ⊘ 'NOMENCLATOR' ya existe en enum synctype (skip)")


def downgrade() -> None:
    """
    Revertir migración: eliminar 'NOMENCLATOR' del enum SyncType.

    ADVERTENCIA:
    - PostgreSQL NO permite eliminar valores de un enum directamente
    - Solo se puede eliminar recreando completamente el enum
    - Esto requiere eliminar TODAS las columnas que usan el enum primero

    Por seguridad, esta migración NO es reversible automáticamente.
    Si necesitas revertir, debes hacerlo manualmente.
    """
    # No implementado por seguridad
    # Eliminar valores de un enum PostgreSQL requiere recrear el tipo completo
    # lo cual puede romper datos existentes
    pass
