"""add_gin_trigram_index_for_laboratory_search

Revision ID: 50c50579fb27
Revises: b609b997a76a
Create Date: 2025-11-02 11:33:31.715391+01:00

Optimización de performance para búsquedas de laboratorios genéricos.

Problema:
- Query `generic_laboratories_page_2` tarda 1.110s en producción
- LIKE '%pattern%' con múltiples ORs no puede usar índices B-tree estándar
- Full table scan de 67,000+ productos en product_catalog

Solución:
- Habilitar extensión pg_trgm (trigramas PostgreSQL)
- Crear índice GIN sobre UPPER(nomen_laboratorio) con gin_trgm_ops
- Soporta LIKE '%pattern%' eficientemente (10-20x mejora esperada)

Impacto esperado:
- 1.110s → 50-150ms (mejora de ~10-20x)
- Sin cambios de código requeridos
- Índice adicional: ~5-10MB en disco

Migración 100% idempotente:
- Verifica existencia de extensión pg_trgm antes de crear
- Verifica existencia de índice antes de crear
- Safe para rollback
"""
from typing import Sequence, Union

from alembic import op
import sqlalchemy as sa


# revision identifiers, used by Alembic.
revision: str = '50c50579fb27'
down_revision: Union[str, None] = 'b609b997a76a'
branch_labels: Union[str, Sequence[str], None] = None
depends_on: Union[str, Sequence[str], None] = None


def upgrade() -> None:
    """
    Crear índice GIN trigram para optimizar búsquedas de laboratorios.

    Verificaciones idempotentes:
    1. Extensión pg_trgm existe
    2. Índice idx_product_catalog_lab_name_trigram existe
    """
    conn = op.get_bind()

    # === PASO 1: Verificar/Habilitar extensión pg_trgm ===
    extension_exists = conn.execute(sa.text("""
        SELECT EXISTS (
            SELECT 1 FROM pg_extension
            WHERE extname = 'pg_trgm'
        )
    """)).scalar()

    if not extension_exists:
        # Crear extensión pg_trgm (trigramas para búsquedas de texto)
        op.execute("CREATE EXTENSION IF NOT EXISTS pg_trgm")
        print("✅ Extensión pg_trgm habilitada")
    else:
        print("⏭️  Extensión pg_trgm ya existe (skip)")

    # === PASO 2: Verificar/Crear índice GIN trigram ===
    index_exists = conn.execute(sa.text("""
        SELECT EXISTS (
            SELECT 1
            FROM pg_indexes
            WHERE tablename = 'product_catalog'
              AND indexname = 'idx_product_catalog_lab_name_trigram'
        )
    """)).scalar()

    if not index_exists:
        # Crear índice GIN con operador trigram sobre UPPER(nomen_laboratorio)
        # Esto permite búsquedas LIKE '%pattern%' eficientes
        # NOTA: Sin CONCURRENTLY porque Alembic usa transacciones
        # En producción con tráfico bajo (Render deployment), esto es safe
        op.execute("""
            CREATE INDEX idx_product_catalog_lab_name_trigram
            ON product_catalog
            USING GIN (UPPER(nomen_laboratorio) gin_trgm_ops)
        """)
        print("✅ Índice GIN trigram creado (puede tardar 30-60s en producción)")
    else:
        print("⏭️  Índice idx_product_catalog_lab_name_trigram ya existe (skip)")

    # === VERIFICACIÓN FINAL ===
    result = conn.execute(sa.text("""
        SELECT
            i.indexname,
            i.indexdef
        FROM pg_indexes i
        WHERE i.tablename = 'product_catalog'
          AND i.indexname = 'idx_product_catalog_lab_name_trigram'
    """)).fetchone()

    if result:
        print("✅ Migración exitosa:")
        print(f"   - Índice: {result[0]}")
        print(f"   - Definición: {result[1][:80]}...")
    else:
        raise Exception("❌ ERROR: Índice no creado correctamente")


def downgrade() -> None:
    """
    Eliminar índice GIN trigram de forma idempotente.

    NOTA: NO eliminamos la extensión pg_trgm por seguridad
    (podría estar en uso por otros índices/funciones)
    """
    conn = op.get_bind()

    # === PASO 1: Verificar/Eliminar índice ===
    index_exists = conn.execute(sa.text("""
        SELECT EXISTS (
            SELECT 1
            FROM pg_indexes
            WHERE tablename = 'product_catalog'
              AND indexname = 'idx_product_catalog_lab_name_trigram'
        )
    """)).scalar()

    if index_exists:
        op.execute("DROP INDEX IF EXISTS idx_product_catalog_lab_name_trigram")
        print("✅ Índice idx_product_catalog_lab_name_trigram eliminado")
    else:
        print("⏭️  Índice idx_product_catalog_lab_name_trigram no existe (skip)")

    # === PASO 2: NO eliminar extensión pg_trgm ===
    # Razón: Podría estar en uso por otros índices o funciones
    print("⚠️  Extensión pg_trgm NO eliminada (seguridad - podría estar en uso)")

    print("✅ Rollback completado exitosamente")
