"""add_product_catalog_vet_table_issue_354

Revision ID: 3758d081c80a
Revises: 796f8d5c313b
Create Date: 2025-11-08 14:24:10.832129+01:00

Issue #354: Integración CIMAVet - Catálogo de medicamentos veterinarios

Crea tabla product_catalog_vet para almacenar ~3,500 productos veterinarios
desde API CIMAVet de la AEMPS.

Cascada de enriquecimiento actualizada:
1. CIMA (medicamentos uso humano)
2. Nomenclator (catálogo oficial Ministerio)
3. CIMAVet (medicamentos uso veterinario) ← NUEVO
4. manual_review

Características:
- Tabla SEPARADA de product_catalog (minimiza riesgo breaking changes)
- Prefijo "VET-" obligatorio en national_code (evita colisión códigos)
- Campo vet_requiere_receta determina prescription vs free_sale (excepción VETERINARIA)
- Índices GIN para búsquedas eficientes en arrays (especies, principios activos)
- Thread-safe para Render multi-worker (extend_existing=True)

REGLA #14: Migración IDEMPOTENTE (verifica existencia antes de crear)
"""
from typing import Sequence, Union

from alembic import op
import sqlalchemy as sa
from sqlalchemy.dialects import postgresql


# revision identifiers, used by Alembic.
revision: str = '3758d081c80a'
down_revision: Union[str, None] = '796f8d5c313b'
branch_labels: Union[str, Sequence[str], None] = None
depends_on: Union[str, Sequence[str], None] = None


def upgrade() -> None:
    """
    Crear tabla product_catalog_vet para productos veterinarios CIMAVet.

    IDEMPOTENTE: Verifica existencia antes de crear (REGLA #14).
    """
    conn = op.get_bind()

    # ✅ IDEMPOTENT: Verificar si tabla ya existe
    result = conn.execute(sa.text(
        "SELECT EXISTS ("
        "   SELECT FROM information_schema.tables "
        "   WHERE table_schema = 'public' "
        "   AND table_name = 'product_catalog_vet'"
        ")"
    ))
    table_exists = result.scalar()

    if table_exists:
        print("[MIGRATION] Tabla product_catalog_vet ya existe, saltando creación...")
        return

    # Crear tabla product_catalog_vet
    op.create_table(
        'product_catalog_vet',
        sa.Column('national_code', sa.String(length=20), nullable=False, comment='Código nacional con prefijo VET-'),
        sa.Column('vet_nombre_comercial', sa.String(length=500), nullable=False, comment='Nombre comercial producto veterinario'),
        sa.Column('vet_principios_activos', postgresql.ARRAY(sa.String()), nullable=True, comment='Lista de principios activos'),
        sa.Column('vet_laboratorio_titular', sa.String(length=200), nullable=True, comment='Laboratorio titular del registro'),
        sa.Column('vet_forma_farmaceutica', sa.String(length=200), nullable=True, comment='Forma farmacéutica'),
        sa.Column('vet_via_administracion', sa.String(length=100), nullable=True, comment='Vía de administración'),
        sa.Column('vet_especies_destino', postgresql.ARRAY(sa.String()), nullable=True, comment='Especies animales destino'),
        sa.Column('vet_condiciones_prescripcion', sa.String(length=50), nullable=True, comment='Condiciones de prescripción'),
        sa.Column('vet_requiere_receta', sa.Boolean(), nullable=True, comment='CRÍTICO: Determina prescription vs free_sale para VETERINARIA'),
        sa.Column('vet_pvp', sa.Numeric(precision=10, scale=4), nullable=True, comment='Precio venta público con IVA'),
        sa.Column('vet_estado_registro', sa.String(length=50), nullable=True, comment='Estado del registro AEMPS'),
        sa.Column('vet_numero_registro', sa.String(length=100), nullable=True, comment='Número de registro oficial'),
        sa.Column('data_source', sa.String(length=50), nullable=False, server_default='CIMAVet', comment='Siempre CIMAVet'),
        sa.Column('enrichment_confidence', sa.Integer(), nullable=True, server_default='90', comment='Alta confianza (datos oficiales AEMPS)'),
        sa.Column('created_at', sa.DateTime(timezone=True), nullable=False, server_default=sa.text('now()'), comment='Fecha creación'),
        sa.Column('updated_at', sa.DateTime(timezone=True), nullable=False, server_default=sa.text('now()'), comment='Fecha actualización'),
        sa.PrimaryKeyConstraint('national_code', name='pk_product_catalog_vet')
    )

    # ✅ ÍNDICES: IF NOT EXISTS garantiza idempotencia (REGLA #14)

    # Índice principal en national_code (PK ya crea índice, este es adicional para búsquedas)
    conn.execute(sa.text(
        "CREATE INDEX IF NOT EXISTS idx_vet_national_code "
        "ON product_catalog_vet (national_code)"
    ))

    # Índice para búsquedas por nombre comercial
    conn.execute(sa.text(
        "CREATE INDEX IF NOT EXISTS idx_vet_nombre_comercial "
        "ON product_catalog_vet (vet_nombre_comercial)"
    ))

    # Índice para filtrado por laboratorio
    conn.execute(sa.text(
        "CREATE INDEX IF NOT EXISTS idx_vet_laboratorio "
        "ON product_catalog_vet (vet_laboratorio_titular)"
    ))

    # Índice para filtrado por requiere receta (clasificación venta)
    conn.execute(sa.text(
        "CREATE INDEX IF NOT EXISTS idx_vet_requiere_receta "
        "ON product_catalog_vet (vet_requiere_receta)"
    ))

    # Índice GIN para búsquedas en array de especies destino
    # GIN es óptimo para búsquedas en arrays (ej: "Perros" IN vet_especies_destino)
    conn.execute(sa.text(
        "CREATE INDEX IF NOT EXISTS idx_vet_especies "
        "ON product_catalog_vet USING gin (vet_especies_destino)"
    ))

    # Índice GIN para búsquedas en array de principios activos
    conn.execute(sa.text(
        "CREATE INDEX IF NOT EXISTS idx_vet_principios_activos "
        "ON product_catalog_vet USING gin (vet_principios_activos)"
    ))

    print("[MIGRATION] ✅ Tabla product_catalog_vet creada exitosamente con 6 índices")


def downgrade() -> None:
    """
    Eliminar tabla product_catalog_vet.

    IDEMPOTENTE: IF EXISTS garantiza que no falla si tabla no existe.
    """
    conn = op.get_bind()

    # ✅ IDEMPOTENT: DROP TABLE IF EXISTS
    conn.execute(sa.text("DROP TABLE IF EXISTS product_catalog_vet CASCADE"))

    print("[MIGRATION] ✅ Tabla product_catalog_vet eliminada (si existía)")
