"""add_employee_indexes_for_filtering

Optimización para queries de filtrado por empleado (Issue #402).

Feature: Employee filtering en página /generics (PRO plan)
- Endpoint GET /api/v1/employees/{pharmacy_id} necesita agregación única
- Servicios de análisis filtran por employee_code + pharmacy_id
- Queries involucran JOINs con product_catalog y sales_enrichment

Índices creados:
1. idx_sales_employee_pharmacy_basic: Para queries de filtrado simple
2. idx_sales_employee_pharmacy_aggregate: Para agregaciones con GROUP BY

Impacto esperado: -40% tiempo de query en agregaciones de empleados

Revision ID: 33d327f16952
Revises: 1931d756e755
Create Date: 2025-11-13 12:20:16.757135+01:00

"""
from typing import Sequence, Union

from alembic import op
import sqlalchemy as sa


# revision identifiers, used by Alembic.
revision: str = '33d327f16952'
down_revision: Union[str, None] = '1931d756e755'
branch_labels: Union[str, Sequence[str], None] = None
depends_on: Union[str, Sequence[str], None] = None


def upgrade() -> None:
    """
    Crea índices para optimizar filtrado por empleado.

    IMPORTANTE: Idempotente (verifica existencia antes de crear).
    Los índices se crean como normales en desarrollo. En producción,
    considerar crear CONCURRENTLY vía SQL directo.

    Índice 1: Filtrado básico (employee_code, pharmacy_id)
    Índice 2: Agregaciones (pharmacy_id, employee_code, employee_name)
              - Incluye employee_name para evitar lookup adicional en GROUP BY

    Partial index: Solo filas con employee_code NOT NULL (ahorra ~15% espacio)
    """
    conn = op.get_bind()

    # Índice 1: Para filtrado básico por employee_code
    result = conn.execute(sa.text(
        "SELECT indexname FROM pg_indexes "
        "WHERE tablename = 'sales_data' "
        "AND indexname = 'idx_sales_employee_pharmacy_basic'"
    ))

    if not result.fetchone():
        op.create_index(
            'idx_sales_employee_pharmacy_basic',
            'sales_data',
            ['employee_code', 'pharmacy_id'],
            unique=False,
            postgresql_where=sa.text("employee_code IS NOT NULL")
        )

    # Índice 2: Para agregaciones (incluye employee_name)
    result = conn.execute(sa.text(
        "SELECT indexname FROM pg_indexes "
        "WHERE tablename = 'sales_data' "
        "AND indexname = 'idx_sales_employee_pharmacy_aggregate'"
    ))

    if not result.fetchone():
        op.create_index(
            'idx_sales_employee_pharmacy_aggregate',
            'sales_data',
            ['pharmacy_id', 'employee_code', 'employee_name'],
            unique=False,
            postgresql_where=sa.text("employee_code IS NOT NULL")
        )


def downgrade() -> None:
    """
    Elimina los índices de forma segura (verifica existencia antes de eliminar).
    """
    conn = op.get_bind()

    # Eliminar índice 1
    result = conn.execute(sa.text(
        "SELECT indexname FROM pg_indexes "
        "WHERE tablename = 'sales_data' "
        "AND indexname = 'idx_sales_employee_pharmacy_basic'"
    ))

    if result.fetchone():
        op.drop_index('idx_sales_employee_pharmacy_basic', table_name='sales_data')

    # Eliminar índice 2
    result = conn.execute(sa.text(
        "SELECT indexname FROM pg_indexes "
        "WHERE tablename = 'sales_data' "
        "AND indexname = 'idx_sales_employee_pharmacy_aggregate'"
    ))

    if result.fetchone():
        op.drop_index('idx_sales_employee_pharmacy_aggregate', table_name='sales_data')
