"""fix_farmatic_total_amount_calculation

Recalcula total_amount = sale_price * quantity para todos los registros
de Farmatic existentes. Corrige bug donde se usaba "Importe Neto" (después
de copagos/descuentos) en lugar de PVP × Cantidad.

Problema original:
- Subdeclaración del ~66% en ventas (1.151€ vs 3.423€)
- Análisis de ahorro de genéricos incorrectos

Contexto: Commit f1c6b3c fix Farmatic parser total_amount

Revision ID: 20712ca991b2
Revises: dbd73a6ea04d
Create Date: 2025-10-30 23:33:28.507977+01:00

"""
from typing import Sequence, Union

from alembic import op
import sqlalchemy as sa


# revision identifiers, used by Alembic.
revision: str = '20712ca991b2'
down_revision: Union[str, None] = 'dbd73a6ea04d'
branch_labels: Union[str, Sequence[str], None] = None
depends_on: Union[str, Sequence[str], None] = None


def upgrade() -> None:
    """
    Recalcula total_amount para registros de Farmatic existentes.

    IMPORTANTE: Esta migración es idempotente y puede ejecutarse múltiples veces
    sin efectos secundarios. Solo recalcula registros que necesitan corrección.
    """
    conn = op.get_bind()

    # 1. Verificar que las tablas y columnas necesarias existen
    inspector = sa.inspect(conn)

    # Verificar tabla sales_data
    if 'sales_data' not in inspector.get_table_names():
        print("⚠️  Tabla 'sales_data' no existe, saltando migración")
        return

    sales_columns = [col['name'] for col in inspector.get_columns('sales_data')]
    required_columns = ['pharmacy_id', 'total_amount', 'sale_price', 'quantity']
    missing_columns = [col for col in required_columns if col not in sales_columns]

    if missing_columns:
        print(f"⚠️  Columnas faltantes en sales_data: {missing_columns}, saltando migración")
        return

    # Verificar tabla file_uploads
    if 'file_uploads' not in inspector.get_table_names():
        print("⚠️  Tabla 'file_uploads' no existe, saltando migración")
        return

    file_uploads_columns = [col['name'] for col in inspector.get_columns('file_uploads')]
    if 'file_type' not in file_uploads_columns:
        print("⚠️  Columna 'file_type' no existe en file_uploads, saltando migración")
        return

    # 2. Contar registros afectados ANTES de la actualización
    count_query = sa.text("""
        SELECT COUNT(*)
        FROM sales_data sd
        WHERE sd.pharmacy_id IN (
            SELECT DISTINCT fu.pharmacy_id
            FROM file_uploads fu
            WHERE fu.file_type = 'FARMATIC'
        )
        AND sd.sale_price IS NOT NULL
        AND sd.quantity IS NOT NULL
        AND sd.sale_price > 0
        AND sd.quantity > 0
    """)

    result = conn.execute(count_query)
    affected_records = result.scalar()

    if affected_records == 0:
        print("✅ No hay registros de Farmatic para corregir")
        return

    print(f"🔄 Recalculando total_amount para {affected_records} registros de Farmatic...")

    # 3. Actualizar registros (idempotente - puede ejecutarse múltiples veces)
    update_query = sa.text("""
        UPDATE sales_data
        SET total_amount = sale_price * quantity
        WHERE pharmacy_id IN (
            SELECT DISTINCT fu.pharmacy_id
            FROM file_uploads fu
            WHERE fu.file_type = 'FARMATIC'
        )
        AND sale_price IS NOT NULL
        AND quantity IS NOT NULL
        AND sale_price > 0
        AND quantity > 0
    """)

    result = conn.execute(update_query)
    updated_rows = result.rowcount

    print(f"✅ Actualizados {updated_rows} registros de Farmatic")

    # 4. Verificación post-actualización
    verification_query = sa.text("""
        SELECT COUNT(*)
        FROM sales_data sd
        WHERE sd.pharmacy_id IN (
            SELECT DISTINCT fu.pharmacy_id
            FROM file_uploads fu
            WHERE fu.file_type = 'FARMATIC'
        )
        AND sd.sale_price IS NOT NULL
        AND sd.quantity IS NOT NULL
        AND sd.sale_price > 0
        AND sd.quantity > 0
        AND ABS(sd.total_amount - (sd.sale_price * sd.quantity)) > 0.01
    """)

    result = conn.execute(verification_query)
    inconsistent_records = result.scalar()

    if inconsistent_records > 0:
        print(f"⚠️  ADVERTENCIA: {inconsistent_records} registros siguen inconsistentes")
    else:
        print("✅ Verificación completada: Todos los registros son consistentes")


def downgrade() -> None:
    """
    No hay downgrade para esta migración de datos.

    Razón: No podemos restaurar los valores incorrectos originales porque
    no sabemos cuáles eran (no guardamos "Importe Neto" original).

    Esta migración es una corrección de bug crítico que no debe revertirse.
    """
    print("⚠️  Esta migración NO tiene downgrade (corrección de bug crítico)")
    print("   Si necesitas revertir, restaura desde backup de base de datos")
    pass
