"""Create license_cache table for Pivot 2026 License Validation.

Revision ID: 20260110_02_license
Revises: 20260110_01_tracking
Create Date: 2026-01-10

This table caches license validation state for offline operation.
Single-row design (id=1 always) for local single-tenant installations.

Features:
- Grace period tracking (7 days offline before kill-switch)
- Time travel protection (detect clock rollback)
- Machine binding (prevent DB cloning)

See: hub/CLAUDE.md, docs/PIVOT_KAIFARMA_2026.md
"""
import sqlalchemy as sa
from alembic import op

# revision identifiers, used by Alembic
revision = "20260110_02_license"
down_revision = "20260110_01_tracking"
branch_labels = None
depends_on = None


def upgrade() -> None:
    """Create license_cache table."""
    # Check if table already exists (idempotent)
    conn = op.get_bind()
    inspector = sa.inspect(conn)

    if "license_cache" not in inspector.get_table_names():
        op.create_table(
            "license_cache",
            # Primary key - always 1 (single row)
            sa.Column("id", sa.Integer(), primary_key=True, default=1),

            # License identification
            sa.Column("license_key", sa.String(50), nullable=False),

            # Pharmacy info from Hub (cached for offline display)
            sa.Column("pharmacy_id", sa.String(36), nullable=True),
            sa.Column("pharmacy_name", sa.String(200), nullable=True),
            sa.Column("tier", sa.String(20), default="standard"),

            # Validation timestamps
            sa.Column("validated_at", sa.DateTime(timezone=True)),
            sa.Column("expires_at", sa.DateTime(timezone=True)),

            # Grace period tracking
            sa.Column("last_online_validation", sa.DateTime(timezone=True)),
            sa.Column("grace_period_ends", sa.DateTime(timezone=True)),

            # Status
            sa.Column("is_valid", sa.Boolean(), default=False),
            sa.Column("validation_error", sa.String(500), nullable=True),

            # Security: Time travel protection
            sa.Column("last_validated_system_time", sa.DateTime(timezone=True)),

            # Security: Machine binding (anti-clone)
            sa.Column("machine_id", sa.String(100), nullable=True),

            # Metadata
            sa.Column(
                "created_at",
                sa.DateTime(timezone=True),
                server_default=sa.text("NOW()"),
            ),
            sa.Column(
                "updated_at",
                sa.DateTime(timezone=True),
                server_default=sa.text("NOW()"),
            ),
        )

        # Note: Initial row is NOT inserted here.
        # The row is created when LicenseClientService validates for the first time.
        # This allows the license_key to be properly set from environment.


def downgrade() -> None:
    """Drop license_cache table."""
    conn = op.get_bind()
    inspector = sa.inspect(conn)

    if "license_cache" in inspector.get_table_names():
        op.drop_table("license_cache")
