"""Create tracking tables for Control Modules (Pivot 2026).

Revision ID: 20260110_01_tracking
Revises: 20260108_01_erp_sync
Create Date: 2026-01-10

Tables created:
- tracking_encargos: Customer order tracking (Module 1: Cazador de Encargos)
- proposal_analysis_cache: Order proposal analysis (Module 2: Interceptor)
- receipt_audit_records: Receipt audit records (Module 3: Auditor)

These tables are for LOCAL SQLite storage in pharmacy installations.
They track pharmacist actions while ERP remains READ-ONLY.

See: docs/CONTROL_MODULES_ARCHITECTURE.md
"""
import sqlalchemy as sa
from alembic import op

# revision identifiers, used by Alembic
revision = "20260110_01_tracking"
down_revision = "20260110_00_merge"
branch_labels = None
depends_on = None


def upgrade() -> None:
    """Create tracking tables for Control Modules."""
    conn = op.get_bind()
    inspector = sa.inspect(conn)
    existing_tables = inspector.get_table_names()

    # ==========================================================================
    # Table 1: tracking_encargos (Module 1: Cazador de Encargos)
    # ==========================================================================
    if "tracking_encargos" not in existing_tables:
        op.create_table(
            "tracking_encargos",
            # Primary Key - ERP order ID (e.g., "ENC-17399-45123")
            sa.Column("erp_order_id", sa.String(100), primary_key=True),
            # Cached data from ERP (for UI display, avoid round-trips)
            sa.Column(
                "cached_pvl",
                sa.Float(),
                default=0.0,
                nullable=False,
                comment="Unit PVP value (EUR)",
            ),
            sa.Column(
                "cached_units",
                sa.Integer(),
                default=1,
                nullable=False,
                comment="Units ordered",
            ),
            sa.Column(
                "customer_name",
                sa.String(200),
                default="Desconocido",
                nullable=False,
                comment="Customer name snapshot",
            ),
            sa.Column(
                "customer_phone",
                sa.String(50),
                nullable=True,
                comment="Phone for calling",
            ),
            sa.Column(
                "product_description",
                sa.String(500),
                default="",
                nullable=False,
                comment="Product description",
            ),
            sa.Column(
                "product_cn",
                sa.String(20),
                nullable=True,
                comment="Codigo Nacional",
            ),
            # State machine
            sa.Column(
                "status",
                sa.String(20),
                default="NEW",
                nullable=False,
                comment="NEW, ALERTED, CONTACTED, TO_RETURN, RESOLVED",
            ),
            # Timestamps
            sa.Column(
                "first_detected_at",
                sa.DateTime(),
                nullable=False,
                server_default=sa.text("CURRENT_TIMESTAMP"),
            ),
            sa.Column(
                "last_seen_in_erp",
                sa.DateTime(),
                nullable=False,
                server_default=sa.text("CURRENT_TIMESTAMP"),
                comment="Last sync where this order was seen (for reconciliation)",
            ),
            sa.Column("alerted_at", sa.DateTime(), nullable=True),
            sa.Column("contacted_at", sa.DateTime(), nullable=True),
            sa.Column("marked_return_at", sa.DateTime(), nullable=True),
            sa.Column("resolved_at", sa.DateTime(), nullable=True),
            # Notes
            sa.Column("notes", sa.Text(), nullable=True),
        )

        # Index for status filtering (dashboard queries)
        op.create_index(
            "ix_tracking_encargos_status",
            "tracking_encargos",
            ["status"],
        )

    # ==========================================================================
    # Table 2: proposal_analysis_cache (Module 2: Interceptor Pre-Pedido)
    # ==========================================================================
    if "proposal_analysis_cache" not in existing_tables:
        op.create_table(
            "proposal_analysis_cache",
            sa.Column("id", sa.Integer(), primary_key=True, autoincrement=True),
            # Identification
            sa.Column(
                "proposal_id",
                sa.String(100),
                nullable=False,
                index=True,
                comment="ERP proposal ID",
            ),
            sa.Column("supplier_code", sa.String(50), default=""),
            sa.Column("supplier_name", sa.String(200), default=""),
            # Analysis results
            sa.Column(
                "analyzed_at",
                sa.DateTime(),
                nullable=False,
                server_default=sa.text("CURRENT_TIMESTAMP"),
            ),
            sa.Column("total_lines", sa.Integer(), default=0),
            sa.Column("lines_ok", sa.Integer(), default=0),
            sa.Column("lines_warning", sa.Integer(), default=0),
            sa.Column("lines_blocker", sa.Integer(), default=0),
            sa.Column("lines_opportunity", sa.Integer(), default=0),
            sa.Column(
                "potential_savings",
                sa.Float(),
                default=0.0,
                comment="Potential savings in EUR",
            ),
            # Actions taken
            sa.Column(
                "was_reviewed",
                sa.Integer(),
                default=0,
                comment="0=No, 1=Yes (SQLite boolean)",
            ),
            sa.Column("reviewed_at", sa.DateTime(), nullable=True),
        )

    # ==========================================================================
    # Table 3: receipt_audit_records (Module 3: Auditor Post-Recepcion)
    # ==========================================================================
    if "receipt_audit_records" not in existing_tables:
        op.create_table(
            "receipt_audit_records",
            sa.Column("id", sa.Integer(), primary_key=True, autoincrement=True),
            # Receipt identification
            sa.Column(
                "receipt_id",
                sa.String(100),
                nullable=False,
                index=True,
                comment="ERP receipt/delivery note ID",
            ),
            sa.Column("receipt_date", sa.DateTime(), nullable=True),
            sa.Column("supplier_code", sa.String(50), default=""),
            sa.Column("supplier_name", sa.String(200), default=""),
            sa.Column("delivery_note_number", sa.String(100), default=""),
            # Audit results
            sa.Column(
                "audited_at",
                sa.DateTime(),
                nullable=False,
                server_default=sa.text("CURRENT_TIMESTAMP"),
            ),
            sa.Column(
                "status",
                sa.String(20),
                default="pending",
                comment="pending, passed, issues_found, reviewed, actioned",
            ),
            sa.Column("total_lines", sa.Integer(), default=0),
            sa.Column("lines_with_issues", sa.Integer(), default=0),
            sa.Column(
                "total_economic_impact",
                sa.Float(),
                default=0.0,
                comment="Economic impact in EUR",
            ),
            # Issue details (JSON serialized)
            sa.Column(
                "issues_json",
                sa.Text(),
                nullable=True,
                comment="JSON array of detected issues",
            ),
            # Actions taken
            sa.Column(
                "action_taken",
                sa.String(50),
                nullable=True,
                comment="accepted, claimed, returned",
            ),
            sa.Column("actioned_at", sa.DateTime(), nullable=True),
            sa.Column("actioned_by", sa.String(100), nullable=True),
        )

        # Index for status filtering
        op.create_index(
            "ix_receipt_audit_records_status",
            "receipt_audit_records",
            ["status"],
        )


def downgrade() -> None:
    """Drop tracking tables."""
    conn = op.get_bind()
    inspector = sa.inspect(conn)
    existing_tables = inspector.get_table_names()

    # Drop in reverse order
    if "receipt_audit_records" in existing_tables:
        op.drop_index("ix_receipt_audit_records_status", table_name="receipt_audit_records")
        op.drop_table("receipt_audit_records")

    if "proposal_analysis_cache" in existing_tables:
        op.drop_table("proposal_analysis_cache")

    if "tracking_encargos" in existing_tables:
        op.drop_index("ix_tracking_encargos_status", table_name="tracking_encargos")
        op.drop_table("tracking_encargos")
