"""Create erp_sync_state table for Pivot 2026 ERP Direct Access.

Revision ID: 20260108_01
Revises: 20260107_02
Create Date: 2026-01-08

This table tracks sync state for the ERP Direct Access feature.
Single-row design (id=1 always) for local single-tenant installations.

Configuration comes from environment variables, not database.
"""
import sqlalchemy as sa
from alembic import op

# revision identifiers, used by Alembic
revision = "20260108_01_erp_sync"
down_revision = "20260107_02_roles"  # Points to latest migration
branch_labels = None
depends_on = None


def upgrade() -> None:
    """Create erp_sync_state table."""
    # Check if table already exists (idempotent)
    conn = op.get_bind()
    inspector = sa.inspect(conn)

    if "erp_sync_state" not in inspector.get_table_names():
        op.create_table(
            "erp_sync_state",
            # Primary key - always 1 (single row)
            sa.Column("id", sa.Integer(), primary_key=True, default=1),
            # Status
            sa.Column(
                "status",
                sa.String(20),
                default="idle",
                nullable=False,
            ),
            sa.Column("sync_enabled", sa.Boolean(), default=True),
            # Sync timestamps
            sa.Column("last_sync_started_at", sa.DateTime(timezone=True)),
            sa.Column("last_sync_completed_at", sa.DateTime(timezone=True)),
            # High water mark for incremental sync (CRITICAL)
            sa.Column("last_sale_timestamp", sa.DateTime(timezone=True)),
            # Last sync stats
            sa.Column("last_sync_records", sa.Integer(), default=0),
            sa.Column("last_sync_duration_seconds", sa.Integer(), default=0),
            # Cumulative stats
            sa.Column("total_syncs", sa.Integer(), default=0),
            sa.Column("total_records_synced", sa.Integer(), default=0),
            # Error tracking
            sa.Column("consecutive_errors", sa.Integer(), default=0),
            sa.Column("last_error", sa.Text()),
            sa.Column("last_error_at", sa.DateTime(timezone=True)),
            # Metadata
            sa.Column(
                "created_at",
                sa.DateTime(timezone=True),
                server_default=sa.text("NOW()"),
            ),
            sa.Column(
                "updated_at",
                sa.DateTime(timezone=True),
                server_default=sa.text("NOW()"),
            ),
        )

        # Insert initial row (id=1) using parameterized query
        erp_sync_state = sa.table(
            "erp_sync_state",
            sa.column("id", sa.Integer),
            sa.column("status", sa.String),
            sa.column("sync_enabled", sa.Boolean),
        )
        op.execute(
            erp_sync_state.insert().values(id=1, status="idle", sync_enabled=True)
        )


def downgrade() -> None:
    """Drop erp_sync_state table."""
    conn = op.get_bind()
    inspector = sa.inspect(conn)

    if "erp_sync_state" in inspector.get_table_names():
        op.drop_table("erp_sync_state")
