"""Add titular and operativo roles

Issue #541: Roles Titular vs Operativo para kaiFarma Local-First

Revision ID: 20260107_02_roles
Revises: 20260107_01_mv
Create Date: 2026-01-07

Nuevos roles:
- titular: Propietario de farmacia (acceso completo incluyendo datos financieros)
- operativo: Empleado de farmacia (acceso operativo sin datos sensibles)
"""
from alembic import op
import sqlalchemy as sa
from sqlalchemy.dialects import postgresql
import json

# revision identifiers
revision = '20260107_02_roles'
down_revision = '20260107_01_mv'
branch_labels = None
depends_on = None

# Role UUIDs (from subscription_limits.py)
TITULAR_ROLE_UUID = "44444444-4444-4444-4444-444444444444"
OPERATIVO_ROLE_UUID = "55555555-5555-5555-5555-555555555555"

# Permissions for each role
TITULAR_PERMISSIONS = [
    "view_financial_data",
    "view_employee_data",
    "manage_partners",
    "view_operational_data",
    "upload_sales",
    "view_dashboards",
]

OPERATIVO_PERMISSIONS = [
    "view_operational_data",
    "upload_sales",
    "view_dashboards",
]


def upgrade() -> None:
    """Add titular and operativo roles to the roles table."""
    conn = op.get_bind()

    # Check if roles already exist (idempotent)
    titular_exists = conn.execute(sa.text(
        "SELECT 1 FROM roles WHERE id = :id OR name = 'titular'"
    ), {"id": TITULAR_ROLE_UUID}).fetchone()

    operativo_exists = conn.execute(sa.text(
        "SELECT 1 FROM roles WHERE id = :id OR name = 'operativo'"
    ), {"id": OPERATIVO_ROLE_UUID}).fetchone()

    # Insert titular role if not exists
    if not titular_exists:
        conn.execute(sa.text("""
            INSERT INTO roles (id, name, permissions, created_at, updated_at)
            VALUES (
                :id,
                'titular',
                CAST(:permissions AS jsonb),
                CURRENT_TIMESTAMP,
                CURRENT_TIMESTAMP
            )
        """), {
            "id": TITULAR_ROLE_UUID,
            "permissions": json.dumps(TITULAR_PERMISSIONS),
        })
        conn.commit()

    # Insert operativo role if not exists
    if not operativo_exists:
        conn.execute(sa.text("""
            INSERT INTO roles (id, name, permissions, created_at, updated_at)
            VALUES (
                :id,
                'operativo',
                CAST(:permissions AS jsonb),
                CURRENT_TIMESTAMP,
                CURRENT_TIMESTAMP
            )
        """), {
            "id": OPERATIVO_ROLE_UUID,
            "permissions": json.dumps(OPERATIVO_PERMISSIONS),
        })
        conn.commit()


def downgrade() -> None:
    """Remove titular and operativo roles."""
    conn = op.get_bind()

    # Delete roles (will fail if users are assigned - intentional)
    conn.execute(sa.text(
        "DELETE FROM roles WHERE name IN ('titular', 'operativo')"
    ))
    conn.commit()
