"""Add materialized views for ETL nocturno

Issue #543: ETL Nocturno + Vistas Materializadas

Revision ID: 20260107_01_mv
Revises: (auto-detected)
Create Date: 2026-01-07

Tablas creadas:
- mv_sales_monthly: Ventas agregadas por mes
- mv_sales_category: Ventas por categoría NECESIDAD
- mv_sales_brand: Ventas por marca
- mv_stock_kpis: KPIs de stock por producto
- mv_weekly_kpis: KPIs semanales para PDF Semanal
- etl_run_log: Log de ejecuciones ETL
"""
from alembic import op
import sqlalchemy as sa
from sqlalchemy.dialects import postgresql

# revision identifiers
revision = '20260107_01_mv'
down_revision = None  # Will be auto-detected
branch_labels = None
depends_on = None


def upgrade() -> None:
    # Helper to check if table exists
    conn = op.get_bind()

    def table_exists(table_name):
        result = conn.execute(sa.text(
            "SELECT EXISTS (SELECT FROM information_schema.tables WHERE table_name = :name)"
        ), {"name": table_name})
        return result.scalar()

    # 1. mv_sales_monthly
    if not table_exists('mv_sales_monthly'):
        op.create_table(
            'mv_sales_monthly',
            sa.Column('id', sa.Integer(), autoincrement=True, nullable=False),
            sa.Column('pharmacy_id', postgresql.UUID(as_uuid=True), nullable=False),
            sa.Column('year_month', sa.Date(), nullable=False),
            sa.Column('vl_total_sales', sa.Numeric(12, 2), server_default='0'),
            sa.Column('vl_total_units', sa.Integer(), server_default='0'),
            sa.Column('vl_total_margin', sa.Numeric(12, 2), server_default='0'),
            sa.Column('vl_margin_percent', sa.Numeric(5, 2), server_default='0'),
            sa.Column('vl_sku_count', sa.Integer(), server_default='0'),
            sa.Column('rx_total_sales', sa.Numeric(12, 2), server_default='0'),
            sa.Column('rx_total_units', sa.Integer(), server_default='0'),
            sa.Column('rx_total_margin', sa.Numeric(12, 2), server_default='0'),
            sa.Column('rx_sku_count', sa.Integer(), server_default='0'),
            sa.Column('total_sales', sa.Numeric(12, 2), server_default='0'),
            sa.Column('total_units', sa.Integer(), server_default='0'),
            sa.Column('total_margin', sa.Numeric(12, 2), server_default='0'),
            sa.Column('updated_at', sa.DateTime(), server_default=sa.text('CURRENT_TIMESTAMP')),
            sa.ForeignKeyConstraint(['pharmacy_id'], ['pharmacies.id'], ondelete='CASCADE'),
            sa.PrimaryKeyConstraint('id')
        )
        op.create_index('ix_mv_sales_monthly_pharmacy_month', 'mv_sales_monthly', ['pharmacy_id', 'year_month'])
        op.create_index('ix_mv_sales_monthly_month', 'mv_sales_monthly', ['year_month'])

    # 2. mv_sales_category
    if not table_exists('mv_sales_category'):
        op.create_table(
            'mv_sales_category',
            sa.Column('id', sa.Integer(), autoincrement=True, nullable=False),
            sa.Column('pharmacy_id', postgresql.UUID(as_uuid=True), nullable=False),
            sa.Column('year_month', sa.Date(), nullable=False),
            sa.Column('ml_category', sa.String(50), nullable=False),
            sa.Column('total_sales', sa.Numeric(12, 2), server_default='0'),
            sa.Column('total_units', sa.Integer(), server_default='0'),
            sa.Column('total_margin', sa.Numeric(12, 2), server_default='0'),
            sa.Column('margin_percent', sa.Numeric(5, 2), server_default='0'),
            sa.Column('sku_count', sa.Integer(), server_default='0'),
            sa.Column('transaction_count', sa.Integer(), server_default='0'),
            sa.Column('top_brand', sa.String(100), nullable=True),
            sa.Column('top_brand_sales', sa.Numeric(12, 2), server_default='0'),
            sa.Column('updated_at', sa.DateTime(), server_default=sa.text('CURRENT_TIMESTAMP')),
            sa.ForeignKeyConstraint(['pharmacy_id'], ['pharmacies.id'], ondelete='CASCADE'),
            sa.PrimaryKeyConstraint('id')
        )
        op.create_index('ix_mv_sales_category_pharmacy_month_cat', 'mv_sales_category', ['pharmacy_id', 'year_month', 'ml_category'])

    # 3. mv_sales_brand
    if not table_exists('mv_sales_brand'):
        op.create_table(
            'mv_sales_brand',
            sa.Column('id', sa.Integer(), autoincrement=True, nullable=False),
            sa.Column('pharmacy_id', postgresql.UUID(as_uuid=True), nullable=False),
            sa.Column('year_month', sa.Date(), nullable=False),
            sa.Column('detected_brand', sa.String(100), nullable=False),
            sa.Column('ml_category', sa.String(50), nullable=True),
            sa.Column('total_sales', sa.Numeric(12, 2), server_default='0'),
            sa.Column('total_units', sa.Integer(), server_default='0'),
            sa.Column('total_margin', sa.Numeric(12, 2), server_default='0'),
            sa.Column('margin_percent', sa.Numeric(5, 2), server_default='0'),
            sa.Column('sku_count', sa.Integer(), server_default='0'),
            sa.Column('updated_at', sa.DateTime(), server_default=sa.text('CURRENT_TIMESTAMP')),
            sa.ForeignKeyConstraint(['pharmacy_id'], ['pharmacies.id'], ondelete='CASCADE'),
            sa.PrimaryKeyConstraint('id')
        )
        op.create_index('ix_mv_sales_brand_pharmacy_month_brand', 'mv_sales_brand', ['pharmacy_id', 'year_month', 'detected_brand'])

    # 4. mv_stock_kpis
    if not table_exists('mv_stock_kpis'):
        op.create_table(
            'mv_stock_kpis',
            sa.Column('id', sa.Integer(), autoincrement=True, nullable=False),
            sa.Column('pharmacy_id', postgresql.UUID(as_uuid=True), nullable=False),
            sa.Column('product_code', sa.String(20), nullable=False),
            sa.Column('product_name', sa.String(255), nullable=True),
            sa.Column('stock_qty', sa.Integer(), server_default='0'),
            sa.Column('stock_value', sa.Numeric(12, 2), server_default='0'),
            sa.Column('daily_velocity', sa.Numeric(8, 3), server_default='0'),
            sa.Column('coverage_days', sa.Integer(), server_default='0'),
            sa.Column('last_sale_date', sa.Date(), nullable=True),
            sa.Column('days_without_sale', sa.Integer(), server_default='0'),
            sa.Column('is_dead_stock', sa.Boolean(), server_default='false'),
            sa.Column('is_slow_mover', sa.Boolean(), server_default='false'),
            sa.Column('is_critical', sa.Boolean(), server_default='false'),
            sa.Column('abc_class', sa.String(1), nullable=True),
            sa.Column('updated_at', sa.DateTime(), server_default=sa.text('CURRENT_TIMESTAMP')),
            sa.ForeignKeyConstraint(['pharmacy_id'], ['pharmacies.id'], ondelete='CASCADE'),
            sa.PrimaryKeyConstraint('id')
        )
        op.create_index('ix_mv_stock_kpis_pharmacy_product', 'mv_stock_kpis', ['pharmacy_id', 'product_code'])
        op.create_index('ix_mv_stock_kpis_dead_stock', 'mv_stock_kpis', ['pharmacy_id', 'is_dead_stock'])
        op.create_index('ix_mv_stock_kpis_critical', 'mv_stock_kpis', ['pharmacy_id', 'is_critical'])

    # 5. mv_weekly_kpis
    if not table_exists('mv_weekly_kpis'):
        op.create_table(
            'mv_weekly_kpis',
            sa.Column('id', sa.Integer(), autoincrement=True, nullable=False),
            sa.Column('pharmacy_id', postgresql.UUID(as_uuid=True), nullable=False),
            sa.Column('week_start', sa.Date(), nullable=False),
            sa.Column('total_sales', sa.Numeric(12, 2), server_default='0'),
            sa.Column('vl_sales', sa.Numeric(12, 2), server_default='0'),
            sa.Column('rx_sales', sa.Numeric(12, 2), server_default='0'),
            sa.Column('total_margin', sa.Numeric(12, 2), server_default='0'),
            sa.Column('margin_percent', sa.Numeric(5, 2), server_default='0'),
            sa.Column('transaction_count', sa.Integer(), server_default='0'),
            sa.Column('unique_products', sa.Integer(), server_default='0'),
            sa.Column('yoy_sales_change', sa.Numeric(6, 2), nullable=True),
            sa.Column('yoy_margin_change', sa.Numeric(6, 2), nullable=True),
            sa.Column('top_category', sa.String(50), nullable=True),
            sa.Column('top_category_sales', sa.Numeric(12, 2), server_default='0'),
            sa.Column('alert_count', sa.Integer(), server_default='0'),
            sa.Column('critical_alert_count', sa.Integer(), server_default='0'),
            sa.Column('updated_at', sa.DateTime(), server_default=sa.text('CURRENT_TIMESTAMP')),
            sa.ForeignKeyConstraint(['pharmacy_id'], ['pharmacies.id'], ondelete='CASCADE'),
            sa.PrimaryKeyConstraint('id')
        )
        op.create_index('ix_mv_weekly_kpis_pharmacy_week', 'mv_weekly_kpis', ['pharmacy_id', 'week_start'])

    # 6. etl_run_log
    if not table_exists('etl_run_log'):
        op.create_table(
            'etl_run_log',
            sa.Column('id', sa.Integer(), autoincrement=True, nullable=False),
            sa.Column('pharmacy_id', postgresql.UUID(as_uuid=True), nullable=True),
            sa.Column('run_type', sa.String(50), nullable=False),
            sa.Column('started_at', sa.DateTime(), nullable=False),
            sa.Column('completed_at', sa.DateTime(), nullable=True),
            sa.Column('duration_seconds', sa.Integer(), nullable=True),
            sa.Column('status', sa.String(20), nullable=False),
            sa.Column('error_message', sa.String(500), nullable=True),
            sa.Column('rows_processed', sa.Integer(), server_default='0'),
            sa.Column('tables_updated', sa.Integer(), server_default='0'),
            sa.Column('details', sa.String(2000), nullable=True),
            sa.ForeignKeyConstraint(['pharmacy_id'], ['pharmacies.id'], ondelete='SET NULL'),
            sa.PrimaryKeyConstraint('id')
        )
        op.create_index('ix_etl_run_log_pharmacy_started', 'etl_run_log', ['pharmacy_id', 'started_at'])
        op.create_index('ix_etl_run_log_status', 'etl_run_log', ['status'])


def downgrade() -> None:
    # Drop tables in reverse order
    op.drop_table('etl_run_log')
    op.drop_table('mv_weekly_kpis')
    op.drop_table('mv_stock_kpis')
    op.drop_table('mv_sales_brand')
    op.drop_table('mv_sales_category')
    op.drop_table('mv_sales_monthly')
