"""Add seasonality_indices table (Issue #507)

Revision ID: 20260106_03
Revises: 20260106_02
Create Date: 2026-01-06

Issue #507: Indice de Estacionalidad por Categoria.
Permite ajustar metricas de momentum por expectativa estacional,
evitando falsos positivos en Decision Box (ej: protector solar -45% en octubre = normal).

REGLA #14: Verificacion de existencia antes de crear (idempotente).
"""

import uuid

import sqlalchemy as sa
from alembic import op
from sqlalchemy.dialects import postgresql

# revision identifiers, used by Alembic.
revision = "20260106_03"
down_revision = "20260106_02"
branch_labels = None
depends_on = None


# ==============================================================================
# SEED DATA: Sector default seasonality indices (7 categories x 12 months)
# ==============================================================================
SECTOR_SEASONALITY_DEFAULTS = {
    "proteccion_solar": [0.3, 0.4, 0.6, 0.8, 1.2, 1.5, 1.8, 1.6, 1.0, 0.5, 0.3, 0.3],
    "antigripales": [1.5, 1.3, 1.0, 0.7, 0.5, 0.4, 0.3, 0.4, 0.6, 1.0, 1.4, 1.6],
    "alergias": [0.5, 0.6, 1.2, 1.6, 1.5, 1.2, 0.8, 0.6, 0.7, 0.6, 0.5, 0.5],
    "adelgazantes": [1.4, 1.3, 1.2, 1.1, 1.0, 0.9, 0.8, 0.8, 1.1, 1.0, 0.9, 0.7],
    "dermocosmetica": [0.9, 0.9, 1.0, 1.0, 1.1, 1.2, 1.3, 1.2, 1.0, 0.9, 0.8, 0.9],
    "repelentes_picaduras": [0.2, 0.3, 0.5, 0.8, 1.2, 1.5, 1.8, 1.5, 0.8, 0.4, 0.2, 0.2],
    "hidratacion": [1.2, 1.1, 1.0, 0.9, 0.8, 0.9, 1.0, 1.0, 0.9, 1.0, 1.1, 1.2],
}


def upgrade() -> None:
    """Create seasonality_indices table and seed sector defaults (idempotent)."""
    conn = op.get_bind()

    # Verificar si la tabla ya existe (REGLA #14: migracion idempotente)
    result = conn.execute(
        sa.text(
            "SELECT EXISTS (SELECT 1 FROM information_schema.tables WHERE table_name = 'seasonality_indices')"
        )
    )
    table_exists = result.scalar()

    if not table_exists:
        op.create_table(
            "seasonality_indices",
            sa.Column("id", postgresql.UUID(as_uuid=True), nullable=False),
            sa.Column(
                "category",
                sa.String(50),
                nullable=False,
                comment="Categoria NECESIDAD L1 (proteccion_solar, antigripales, etc.)",
            ),
            sa.Column(
                "month",
                sa.Integer(),
                nullable=False,
                comment="Mes del anio (1=enero, 12=diciembre)",
            ),
            sa.Column(
                "index_value",
                sa.Numeric(4, 2),  # Headroom futuro (soporta hasta 99.99)
                nullable=False,
                server_default="1.00",
                comment="Indice estacional: 1.0=promedio, 0.5=baja, 1.5=alta",
            ),
            sa.Column(
                "pharmacy_id",
                postgresql.UUID(as_uuid=True),
                nullable=True,
                comment="ID farmacia o NULL para defaults de sector",
            ),
            sa.Column(
                "source",
                sa.String(20),
                nullable=False,
                server_default="sector_default",
                comment="Origen: sector_default, calculated, manual",
            ),
            sa.Column(
                "created_at",
                sa.DateTime(timezone=True),
                server_default=sa.text("now()"),
                nullable=True,
            ),
            sa.Column(
                "updated_at",
                sa.DateTime(timezone=True),
                server_default=sa.text("now()"),
                nullable=True,
            ),
            sa.PrimaryKeyConstraint("id"),
            sa.ForeignKeyConstraint(
                ["pharmacy_id"],
                ["pharmacies.id"],
                name="fk_seasonality_indices_pharmacy_id",
                ondelete="CASCADE",
            ),
        )

        # Crear indices
        op.create_index(
            "ix_seasonality_indices_category",
            "seasonality_indices",
            ["category"],
            unique=False,
        )
        op.create_index(
            "ix_seasonality_indices_month",
            "seasonality_indices",
            ["month"],
            unique=False,
        )
        op.create_index(
            "ix_seasonality_indices_pharmacy_id",
            "seasonality_indices",
            ["pharmacy_id"],
            unique=False,
        )

        # Constraint unico: category + month + pharmacy_id
        op.create_unique_constraint(
            "uq_seasonality_category_month_pharmacy",
            "seasonality_indices",
            ["category", "month", "pharmacy_id"],
        )

        # Check constraints
        op.execute(
            sa.text(
                "ALTER TABLE seasonality_indices ADD CONSTRAINT ck_seasonality_month_range "
                "CHECK (month >= 1 AND month <= 12)"
            )
        )
        op.execute(
            sa.text(
                "ALTER TABLE seasonality_indices ADD CONSTRAINT ck_seasonality_index_range "
                "CHECK (index_value >= 0.1 AND index_value <= 3.0)"
            )
        )

        print("Created seasonality_indices table with indexes and constraints")

        # Seed sector defaults
        _seed_sector_defaults(conn)
    else:
        print("Table seasonality_indices already exists, skipping creation")

        # Verificar si hay datos sector, si no insertarlos
        result = conn.execute(
            sa.text(
                "SELECT COUNT(*) FROM seasonality_indices WHERE pharmacy_id IS NULL"
            )
        )
        sector_count = result.scalar()

        if sector_count == 0:
            _seed_sector_defaults(conn)
        else:
            print(f"Found {sector_count} sector defaults, skipping seed")


def _seed_sector_defaults(conn) -> None:
    """Insert sector default seasonality indices."""
    insert_count = 0

    for category, monthly_indices in SECTOR_SEASONALITY_DEFAULTS.items():
        for month, index_value in enumerate(monthly_indices, start=1):
            conn.execute(
                sa.text(
                    """
                    INSERT INTO seasonality_indices
                    (id, category, month, index_value, pharmacy_id, source, created_at, updated_at)
                    VALUES (:id, :category, :month, :index_value, NULL, 'sector_default', now(), now())
                    ON CONFLICT (category, month, pharmacy_id) DO NOTHING
                    """
                ),
                {
                    "id": str(uuid.uuid4()),
                    "category": category,
                    "month": month,
                    "index_value": index_value,
                },
            )
            insert_count += 1

    print(f"Seeded {insert_count} sector default seasonality indices (7 categories x 12 months)")


def downgrade() -> None:
    """Drop seasonality_indices table."""
    conn = op.get_bind()

    # Verificar si la tabla existe antes de intentar eliminarla
    result = conn.execute(
        sa.text(
            "SELECT EXISTS (SELECT 1 FROM information_schema.tables WHERE table_name = 'seasonality_indices')"
        )
    )
    table_exists = result.scalar()

    if table_exists:
        op.drop_table("seasonality_indices")
        print("Dropped seasonality_indices table")
    else:
        print("Table seasonality_indices does not exist, skipping drop")
