"""Add brand_aliases table

Revision ID: 20260106_02
Revises: 20260106_01
Create Date: 2026-01-06

Sistema de corrección de marcas para BrandDetectionService.
Allows admins to:
- Merge brands: "oralkin" → "kin"
- Exclude false positives: "clorhexidina" → (not a brand)

REGLA #14: Verificación de existencia antes de crear (idempotente).
"""

import logging

import sqlalchemy as sa
from alembic import op

# revision identifiers, used by Alembic.
revision = "20260106_02"
down_revision = "20260106_01"
branch_labels = None
depends_on = None

logger = logging.getLogger(__name__)

# ==============================================================================
# SEED DATA: Initial brand aliases
# ==============================================================================
INITIAL_ALIASES = [
    # (source_brand, target_brand, action, reason)
    # Exclude false positives (ingredients detected as brands)
    ("clorhexidina", None, "EXCLUDE", "Ingrediente activo, no es marca"),

    # Merge brands (same brand, different names)
    ("oralkin", "kin", "ALIAS", "Misma marca, nombre diferente"),

    # Migrate legacy hardcoded normalizations from brand_detection_service.py
    ("la roche", "la roche posay", "ALIAS", "Normalización legacy (línea 2041)"),
    ("gema herrerias", "gh", "ALIAS", "Abreviación (línea 2044)"),
    ("ivb wellness", "ivb", "ALIAS", "Abreviación (línea 2046)"),
]


def upgrade() -> None:
    """Create brand_aliases table with enum and seed initial data."""
    conn = op.get_bind()

    # ====================================================================
    # STEP 1: Check if table already exists (idempotent)
    # ====================================================================
    table_exists = conn.execute(
        sa.text(
            "SELECT table_name FROM information_schema.tables "
            "WHERE table_schema='public' AND table_name='brand_aliases'"
        )
    ).fetchone()

    if table_exists:
        logger.info("brand_aliases table already exists, skipping creation")
        # Still try to seed missing aliases
        _seed_missing_aliases(conn)
        return

    # ====================================================================
    # STEP 2: Create enum type
    # ====================================================================
    enum_exists = conn.execute(
        sa.text(
            "SELECT 1 FROM pg_type WHERE typname = 'brand_alias_action_enum'"
        )
    ).fetchone()

    if not enum_exists:
        op.execute(
            sa.text(
                "CREATE TYPE brand_alias_action_enum AS ENUM ('alias', 'exclude')"
            )
        )
        logger.info("Created brand_alias_action_enum type")

    # ====================================================================
    # STEP 3: Create table (using raw SQL to avoid SQLAlchemy Enum issues)
    # ====================================================================
    op.execute(
        sa.text(
            """
            CREATE TABLE brand_aliases (
                id SERIAL PRIMARY KEY,
                source_brand VARCHAR(100) NOT NULL UNIQUE,
                target_brand VARCHAR(100),
                action brand_alias_action_enum NOT NULL DEFAULT 'alias',
                is_active BOOLEAN NOT NULL DEFAULT true,
                reason VARCHAR(500),
                usage_count INTEGER NOT NULL DEFAULT 0,
                last_used_at TIMESTAMP WITH TIME ZONE,
                created_at TIMESTAMP WITH TIME ZONE NOT NULL DEFAULT NOW()
            )
            """
        )
    )
    logger.info("Created brand_aliases table")

    # ====================================================================
    # STEP 4: Create indexes
    # ====================================================================
    # Note: source_brand unique constraint already defined in table creation

    # Index for active aliases (most common query)
    op.create_index(
        "ix_brand_alias_active",
        "brand_aliases",
        ["is_active"],
    )

    # Index for target lookup (reverse lookups)
    op.create_index(
        "ix_brand_alias_target",
        "brand_aliases",
        ["target_brand"],
    )

    # Index for action type filtering
    op.create_index(
        "ix_brand_alias_action",
        "brand_aliases",
        ["action"],
    )

    # Case-insensitive index for source search
    op.execute(
        sa.text(
            """
            CREATE INDEX ix_brand_alias_source_lower
            ON brand_aliases (lower(source_brand))
            """
        )
    )

    logger.info("Created indexes for brand_aliases table")

    # ====================================================================
    # STEP 5: Seed initial aliases
    # ====================================================================
    _seed_all_aliases(conn)


def _seed_all_aliases(conn) -> None:
    """Seed all initial aliases."""
    for source, target, action, reason in INITIAL_ALIASES:
        conn.execute(
            sa.text(
                """
                INSERT INTO brand_aliases (source_brand, target_brand, action, is_active, reason, usage_count, created_at)
                VALUES (:source, :target, CAST(:action AS brand_alias_action_enum), true, :reason, 0, NOW())
                """
            ),
            {"source": source, "target": target, "action": action, "reason": reason},
        )
    logger.info(f"Seeded {len(INITIAL_ALIASES)} brand aliases")


def _seed_missing_aliases(conn) -> None:
    """Seed only missing aliases (for idempotency)."""
    seeded = 0
    for source, target, action, reason in INITIAL_ALIASES:
        exists = conn.execute(
            sa.text(
                "SELECT 1 FROM brand_aliases WHERE source_brand = :source"
            ),
            {"source": source},
        ).fetchone()

        if not exists:
            conn.execute(
                sa.text(
                    """
                    INSERT INTO brand_aliases (source_brand, target_brand, action, is_active, reason, usage_count, created_at)
                    VALUES (:source, :target, CAST(:action AS brand_alias_action_enum), true, :reason, 0, NOW())
                    """
                ),
                {"source": source, "target": target, "action": action, "reason": reason},
            )
            seeded += 1

    if seeded > 0:
        logger.info(f"Seeded {seeded} missing brand aliases")


def downgrade() -> None:
    """Drop brand_aliases table and enum."""
    conn = op.get_bind()

    # Check if table exists before dropping
    table_exists = conn.execute(
        sa.text(
            "SELECT table_name FROM information_schema.tables "
            "WHERE table_schema='public' AND table_name='brand_aliases'"
        )
    ).fetchone()

    if not table_exists:
        logger.info("brand_aliases table does not exist, skipping drop")
    else:
        # Drop indexes first (in reverse order of creation)
        op.execute(sa.text("DROP INDEX IF EXISTS ix_brand_alias_source_lower"))
        op.execute(sa.text("DROP INDEX IF EXISTS ix_brand_alias_action"))
        op.execute(sa.text("DROP INDEX IF EXISTS ix_brand_alias_target"))
        op.execute(sa.text("DROP INDEX IF EXISTS ix_brand_alias_active"))

        # Drop table
        op.drop_table("brand_aliases")
        logger.info("Dropped brand_aliases table")

    # Drop enum type
    enum_exists = conn.execute(
        sa.text(
            "SELECT 1 FROM pg_type WHERE typname = 'brand_alias_action_enum'"
        )
    ).fetchone()

    if enum_exists:
        op.execute(sa.text("DROP TYPE brand_alias_action_enum"))
        logger.info("Dropped brand_alias_action_enum type")
