"""Add pharmacy_targets table for benchmarks (Issue #510)

Revision ID: 20260106_01
Revises: 20260103_01
Create Date: 2026-01-06

Issue #510: Benchmarks y Líneas de Referencia en Gráficos
Permite definir líneas de referencia configurables (mínimo, objetivo, excelente)
para contextualizar rendimiento en gráficos del dashboard VentaLibre.
"""

from alembic import op
import sqlalchemy as sa
from sqlalchemy.dialects import postgresql

# revision identifiers, used by Alembic.
revision = "20260106_01"
down_revision = "20260103_01"
branch_labels = None
depends_on = None


def upgrade() -> None:
    """Create pharmacy_targets table (idempotent)."""
    conn = op.get_bind()

    # Verificar si la tabla ya existe (REGLA #14: migración idempotente)
    result = conn.execute(
        sa.text(
            "SELECT EXISTS (SELECT 1 FROM information_schema.tables WHERE table_name = 'pharmacy_targets')"
        )
    )
    table_exists = result.scalar()

    if not table_exists:
        op.create_table(
            "pharmacy_targets",
            sa.Column("id", postgresql.UUID(as_uuid=True), nullable=False),
            sa.Column("pharmacy_id", postgresql.UUID(as_uuid=True), nullable=False),
            sa.Column(
                "target_type",
                sa.String(50),
                nullable=False,
                comment="Tipo de target: margin_min, margin_target, margin_excellent, hhi_max, sales_target, price_avg",
            ),
            sa.Column(
                "category",
                sa.String(100),
                nullable=True,
                comment="Categoría NECESIDAD específica o NULL para target global",
            ),
            sa.Column(
                "value",
                sa.Numeric(12, 2),
                nullable=False,
                comment="Valor numérico del target (porcentaje, cantidad, etc.)",
            ),
            sa.Column(
                "color",
                sa.String(20),
                nullable=False,
                server_default="#28a745",
                comment="Color de la línea en formato hex (#RRGGBB)",
            ),
            sa.Column(
                "line_style",
                sa.String(20),
                nullable=False,
                server_default="dash",
                comment="Estilo de línea: solid, dash, dot",
            ),
            sa.Column(
                "label",
                sa.String(100),
                nullable=True,
                comment="Etiqueta mostrada junto a la línea de referencia",
            ),
            sa.Column(
                "created_at",
                sa.DateTime(timezone=True),
                server_default=sa.text("now()"),
                nullable=True,
            ),
            sa.Column(
                "updated_at",
                sa.DateTime(timezone=True),
                server_default=sa.text("now()"),
                nullable=True,
            ),
            sa.Column(
                "user_configured",
                sa.Boolean(),
                nullable=False,
                server_default="false",
                comment="TRUE si fue configurado manualmente por el usuario (no default)",
            ),
            sa.PrimaryKeyConstraint("id"),
            sa.ForeignKeyConstraint(
                ["pharmacy_id"],
                ["pharmacies.id"],
                name="fk_pharmacy_targets_pharmacy_id",
                ondelete="CASCADE",
            ),
        )

        # Crear índices
        op.create_index(
            "ix_pharmacy_targets_pharmacy_id",
            "pharmacy_targets",
            ["pharmacy_id"],
            unique=False,
        )
        op.create_index(
            "ix_pharmacy_targets_target_type",
            "pharmacy_targets",
            ["target_type"],
            unique=False,
        )
        op.create_index(
            "ix_pharmacy_targets_category",
            "pharmacy_targets",
            ["category"],
            unique=False,
        )

        # Constraint único: pharmacy_id + target_type + category
        op.create_unique_constraint(
            "uq_pharmacy_target_type_category",
            "pharmacy_targets",
            ["pharmacy_id", "target_type", "category"],
        )

        print("Created pharmacy_targets table with indexes and constraints")
    else:
        print("Table pharmacy_targets already exists, skipping creation")


def downgrade() -> None:
    """Drop pharmacy_targets table."""
    conn = op.get_bind()

    # Verificar si la tabla existe antes de intentar eliminarla
    result = conn.execute(
        sa.text(
            "SELECT EXISTS (SELECT 1 FROM information_schema.tables WHERE table_name = 'pharmacy_targets')"
        )
    )
    table_exists = result.scalar()

    if table_exists:
        op.drop_table("pharmacy_targets")
        print("Dropped pharmacy_targets table")
    else:
        print("Table pharmacy_targets does not exist, skipping drop")
