"""Add indexes for prescription analytics performance.

Revision ID: 20260103_01
Revises: 20260102_02
Create Date: 2026-01-03

Issue: Prescription endpoints are slow (2-4s) due to missing indexes on:
- sales_enrichment JOIN columns
- product_catalog grouping/filtering columns
- sales_data date range queries

Expected improvement: 2-4s -> <500ms for prescription queries.

REGLA #14: Idempotent - verify index existence before creating.

NOTE on CONCURRENTLY: For very large tables (>1M rows), consider using
CREATE INDEX CONCURRENTLY to avoid table locks during index creation.
This requires running outside a transaction and using raw SQL.
For xFarma's current scale (~100k rows), standard CREATE INDEX is fine.
"""

import logging
from typing import Sequence, Union

import sqlalchemy as sa
from alembic import op

# revision identifiers, used by Alembic.
revision: str = "20260103_01"
down_revision: Union[str, None] = "20260102_02"
branch_labels: Union[str, Sequence[str], None] = None
depends_on: Union[str, Sequence[str], None] = None

logger = logging.getLogger(__name__)


def index_exists(conn, index_name: str) -> bool:
    """Check if an index exists."""
    result = conn.execute(
        sa.text(
            """
            SELECT 1 FROM pg_indexes
            WHERE indexname = :index_name
            """
        ),
        {"index_name": index_name},
    )
    return result.fetchone() is not None


def upgrade() -> None:
    """Create performance indexes for prescription analytics."""
    conn = op.get_bind()

    # =========================================================================
    # INDEX 1: sales_enrichment JOIN optimization
    # =========================================================================
    # Used in every prescription query for the 3-way JOIN:
    # sales_data -> sales_enrichment -> product_catalog
    idx_name = "ix_se_prescription_join"
    if not index_exists(conn, idx_name):
        op.create_index(
            idx_name,
            "sales_enrichment",
            ["sales_data_id", "product_catalog_id"],
            postgresql_where=sa.text("enrichment_status = 'enriched'"),
        )
        logger.info(f"Created index {idx_name}")
    else:
        logger.info(f"Index {idx_name} already exists, skipping")

    # =========================================================================
    # INDEX 2: product_catalog grouping fields (prescription analytics)
    # =========================================================================
    # Used for GROUP BY in waterfall, top-contributors queries
    # Covers: nomen_nombre_homogeneo, nomen_principio_activo, xfarma_prescription_category
    idx_name = "ix_pc_prescription_grouping"
    if not index_exists(conn, idx_name):
        op.create_index(
            idx_name,
            "product_catalog",
            ["xfarma_prescription_category", "nomen_nombre_homogeneo", "nomen_principio_activo"],
            postgresql_where=sa.text("xfarma_prescription_category IS NOT NULL"),
        )
        logger.info(f"Created index {idx_name}")
    else:
        logger.info(f"Index {idx_name} already exists, skipping")

    # =========================================================================
    # INDEX 3: product_catalog ATC code filtering
    # =========================================================================
    # Used for ATC distribution queries (LEFT(cima_atc_code, N))
    idx_name = "ix_pc_atc_code_partial"
    if not index_exists(conn, idx_name):
        op.create_index(
            idx_name,
            "product_catalog",
            ["cima_atc_code"],
            postgresql_where=sa.text("cima_atc_code IS NOT NULL AND xfarma_prescription_category IS NOT NULL"),
        )
        logger.info(f"Created index {idx_name}")
    else:
        logger.info(f"Index {idx_name} already exists, skipping")

    # =========================================================================
    # INDEX 4: sales_data composite for prescription date ranges
    # =========================================================================
    # Covers: pharmacy_id + sale_date range scans (most common filter pattern)
    idx_name = "ix_sd_pharmacy_date_prescription"
    if not index_exists(conn, idx_name):
        op.create_index(
            idx_name,
            "sales_data",
            ["pharmacy_id", "sale_date"],
        )
        logger.info(f"Created index {idx_name}")
    else:
        logger.info(f"Index {idx_name} already exists, skipping")

    logger.info("Prescription performance indexes migration completed")


def downgrade() -> None:
    """Drop performance indexes."""
    conn = op.get_bind()

    indexes_to_drop = [
        ("ix_sd_pharmacy_date_prescription", "sales_data"),
        ("ix_pc_atc_code_partial", "product_catalog"),
        ("ix_pc_prescription_grouping", "product_catalog"),
        ("ix_se_prescription_join", "sales_enrichment"),
    ]

    for idx_name, table_name in indexes_to_drop:
        if index_exists(conn, idx_name):
            op.drop_index(idx_name, table_name=table_name)
            logger.info(f"Dropped index {idx_name}")
        else:
            logger.info(f"Index {idx_name} does not exist, skipping drop")
