"""Create action_trackings and monthly_roi_summaries tables (Issue #514)

Issue #514: ROI Tracker (Feedback Loop de Acciones)
Cerrar el círculo diagnóstico → acción → resultado.

Tablas:
- action_trackings: Registro de acciones sugeridas y su seguimiento
- monthly_roi_summaries: Agregación mensual de ROI por farmacia

REGLA #14: Verificación de existencia antes de crear (idempotente).

Revision ID: 20260102_02
Revises: 20260102_01
Create Date: 2026-01-02
"""

import logging
from typing import Sequence, Union

import sqlalchemy as sa
from alembic import op
from sqlalchemy.dialects.postgresql import JSONB, UUID

# revision identifiers, used by Alembic.
revision: str = "20260102_02"
down_revision: Union[str, None] = "20260102_01"
branch_labels: Union[str, Sequence[str], None] = None
depends_on: Union[str, Sequence[str], None] = None

logger = logging.getLogger(__name__)


def upgrade() -> None:
    """Create action_trackings and monthly_roi_summaries tables."""
    conn = op.get_bind()

    # =========================================================================
    # STEP 1: Create action_trackings table
    # =========================================================================
    table_exists = conn.execute(
        sa.text(
            "SELECT table_name FROM information_schema.tables "
            "WHERE table_schema='public' AND table_name='action_trackings'"
        )
    ).fetchone()

    if not table_exists:
        op.create_table(
            "action_trackings",
            # Primary key
            sa.Column("id", UUID(as_uuid=True), primary_key=True),
            # Foreign key to pharmacy
            sa.Column(
                "pharmacy_id",
                UUID(as_uuid=True),
                sa.ForeignKey("pharmacies.id", ondelete="CASCADE"),
                nullable=False,
            ),
            # Action details
            sa.Column(
                "action_type",
                sa.String(20),
                nullable=False,
                comment="Tipo: liquidation, restock, pricing, diversify",
            ),
            sa.Column(
                "action_description",
                sa.Text(),
                nullable=False,
                comment="Descripción legible de la acción sugerida",
            ),
            sa.Column(
                "affected_products",
                JSONB(),
                nullable=False,
                server_default="[]",
                comment="Lista de productos afectados",
            ),
            # Expected impact
            sa.Column(
                "expected_impact_eur",
                sa.Float(),
                nullable=False,
                server_default="0.0",
                comment="Impacto económico esperado en EUR",
            ),
            sa.Column(
                "expected_impact_description",
                sa.Text(),
                nullable=True,
                comment="Explicación del cálculo del impacto esperado",
            ),
            # Status
            sa.Column(
                "status",
                sa.String(20),
                nullable=False,
                server_default="pending",
                comment="Estado: pending, executed, discarded, postponed",
            ),
            sa.Column(
                "executed_at",
                sa.DateTime(timezone=True),
                nullable=True,
                comment="Fecha/hora en que el usuario marcó como ejecutada",
            ),
            sa.Column(
                "discarded_at",
                sa.DateTime(timezone=True),
                nullable=True,
                comment="Fecha/hora en que el usuario descartó la acción",
            ),
            sa.Column(
                "discard_reason",
                sa.Text(),
                nullable=True,
                comment="Razón opcional por la que se descartó",
            ),
            sa.Column(
                "postponed_until",
                sa.DateTime(timezone=True),
                nullable=True,
                comment="Fecha hasta la cual está pospuesta",
            ),
            # Actual impact
            sa.Column(
                "actual_impact_eur",
                sa.Float(),
                nullable=True,
                comment="Impacto real medido en EUR",
            ),
            sa.Column(
                "actual_impact_description",
                sa.Text(),
                nullable=True,
                comment="Explicación del impacto real calculado",
            ),
            sa.Column(
                "impact_calculated_at",
                sa.DateTime(timezone=True),
                nullable=True,
                comment="Fecha/hora del último cálculo de impacto real",
            ),
            # Source references
            sa.Column(
                "insight_hash",
                sa.String(64),
                nullable=True,
                comment="Hash del insight que generó esta acción",
            ),
            # Timestamps
            sa.Column(
                "created_at",
                sa.DateTime(timezone=True),
                server_default=sa.text("NOW()"),
                nullable=False,
            ),
            sa.Column(
                "updated_at",
                sa.DateTime(timezone=True),
                nullable=True,
            ),
        )
        logger.info("Created action_trackings table")

        # Create indexes for action_trackings
        op.create_index(
            "ix_action_trackings_pharmacy_id",
            "action_trackings",
            ["pharmacy_id"],
        )
        op.create_index(
            "ix_action_tracking_pharmacy_status",
            "action_trackings",
            ["pharmacy_id", "status"],
        )
        op.create_index(
            "ix_action_tracking_pharmacy_type",
            "action_trackings",
            ["pharmacy_id", "action_type"],
        )
        op.create_index(
            "ix_action_tracking_created_at",
            "action_trackings",
            ["created_at"],
        )
        op.create_index(
            "ix_action_tracking_insight_hash",
            "action_trackings",
            ["insight_hash"],
        )
        logger.info("Created indexes for action_trackings table")
    else:
        logger.info("action_trackings table already exists, skipping creation")

    # =========================================================================
    # STEP 2: Create monthly_roi_summaries table
    # =========================================================================
    table_exists = conn.execute(
        sa.text(
            "SELECT table_name FROM information_schema.tables "
            "WHERE table_schema='public' AND table_name='monthly_roi_summaries'"
        )
    ).fetchone()

    if not table_exists:
        op.create_table(
            "monthly_roi_summaries",
            # Primary key
            sa.Column("id", UUID(as_uuid=True), primary_key=True),
            # Foreign key to pharmacy
            sa.Column(
                "pharmacy_id",
                UUID(as_uuid=True),
                sa.ForeignKey("pharmacies.id", ondelete="CASCADE"),
                nullable=False,
            ),
            # Month identifier
            sa.Column(
                "month",
                sa.Date(),
                nullable=False,
                comment="Primer día del mes (ej: 2026-01-01)",
            ),
            # Action counts
            sa.Column(
                "actions_suggested",
                sa.Integer(),
                nullable=False,
                server_default="0",
                comment="Total de acciones sugeridas en el mes",
            ),
            sa.Column(
                "actions_executed",
                sa.Integer(),
                nullable=False,
                server_default="0",
                comment="Acciones marcadas como ejecutadas",
            ),
            sa.Column(
                "actions_discarded",
                sa.Integer(),
                nullable=False,
                server_default="0",
                comment="Acciones descartadas por el usuario",
            ),
            sa.Column(
                "actions_postponed",
                sa.Integer(),
                nullable=False,
                server_default="0",
                comment="Acciones pospuestas (pendientes de re-evaluación)",
            ),
            # Impact totals
            sa.Column(
                "total_expected_impact",
                sa.Float(),
                nullable=False,
                server_default="0.0",
                comment="Suma del impacto esperado de acciones ejecutadas (EUR)",
            ),
            sa.Column(
                "total_actual_impact",
                sa.Float(),
                nullable=False,
                server_default="0.0",
                comment="Suma del impacto real medido (EUR)",
            ),
            # Breakdown by type
            sa.Column(
                "impact_by_type",
                JSONB(),
                nullable=False,
                server_default="{}",
                comment="Impacto por tipo de acción",
            ),
            # ROI calculation
            sa.Column(
                "subscription_cost",
                sa.Float(),
                nullable=False,
                server_default="99.0",
                comment="Coste de suscripción mensual (EUR)",
            ),
            sa.Column(
                "roi_percentage",
                sa.Float(),
                nullable=True,
                comment="ROI = (total_actual_impact / subscription_cost) * 100",
            ),
            # Timestamps
            sa.Column(
                "calculated_at",
                sa.DateTime(timezone=True),
                server_default=sa.text("NOW()"),
                nullable=False,
                comment="Fecha/hora del último cálculo",
            ),
        )
        logger.info("Created monthly_roi_summaries table")

        # Create indexes and constraints for monthly_roi_summaries
        op.create_index(
            "ix_roi_summaries_pharmacy_id",
            "monthly_roi_summaries",
            ["pharmacy_id"],
        )
        op.create_index(
            "ix_roi_summary_month",
            "monthly_roi_summaries",
            ["month"],
        )

        # Unique constraint: one summary per pharmacy per month
        op.create_unique_constraint(
            "uq_roi_summary_pharmacy_month",
            "monthly_roi_summaries",
            ["pharmacy_id", "month"],
        )
        logger.info("Created indexes and constraints for monthly_roi_summaries table")
    else:
        logger.info("monthly_roi_summaries table already exists, skipping creation")


def downgrade() -> None:
    """Drop action_trackings and monthly_roi_summaries tables."""
    conn = op.get_bind()

    # =========================================================================
    # Drop monthly_roi_summaries table
    # =========================================================================
    table_exists = conn.execute(
        sa.text(
            "SELECT table_name FROM information_schema.tables "
            "WHERE table_schema='public' AND table_name='monthly_roi_summaries'"
        )
    ).fetchone()

    if table_exists:
        # Drop constraints and indexes
        op.drop_constraint(
            "uq_roi_summary_pharmacy_month",
            "monthly_roi_summaries",
            type_="unique",
        )
        op.drop_index("ix_roi_summary_month", table_name="monthly_roi_summaries")
        op.drop_index("ix_roi_summaries_pharmacy_id", table_name="monthly_roi_summaries")
        # Drop table
        op.drop_table("monthly_roi_summaries")
        logger.info("Dropped monthly_roi_summaries table")
    else:
        logger.info("monthly_roi_summaries table does not exist, skipping drop")

    # =========================================================================
    # Drop action_trackings table
    # =========================================================================
    table_exists = conn.execute(
        sa.text(
            "SELECT table_name FROM information_schema.tables "
            "WHERE table_schema='public' AND table_name='action_trackings'"
        )
    ).fetchone()

    if table_exists:
        # Drop indexes
        op.drop_index("ix_action_tracking_insight_hash", table_name="action_trackings")
        op.drop_index("ix_action_tracking_created_at", table_name="action_trackings")
        op.drop_index("ix_action_tracking_pharmacy_type", table_name="action_trackings")
        op.drop_index("ix_action_tracking_pharmacy_status", table_name="action_trackings")
        op.drop_index("ix_action_trackings_pharmacy_id", table_name="action_trackings")
        # Drop table
        op.drop_table("action_trackings")
        logger.info("Dropped action_trackings table")
    else:
        logger.info("action_trackings table does not exist, skipping drop")
