"""Create narrative_feedbacks table (Issue #509)

Issue #509: Executive Summary NLG - Resumen Ejecutivo con Natural Language Generation.

Tabla para almacenar feedback de usuarios sobre narrativas generadas.
Incluye snapshots de KPIs e insights para debugging de alucinaciones.

REGLA #14: Verificacion de existencia antes de crear (idempotente).

Revision ID: 20260101_01
Revises: 20251231_01
Create Date: 2026-01-01
"""

import logging

import sqlalchemy as sa
from alembic import op
from sqlalchemy.dialects.postgresql import JSONB, UUID

# revision identifiers, used by Alembic.
revision = "20260101_01"
down_revision = "20251231_01"
branch_labels = None
depends_on = None

logger = logging.getLogger(__name__)


def upgrade() -> None:
    """Create narrative_feedbacks table."""
    conn = op.get_bind()

    # ====================================================================
    # STEP 1: Check if table already exists (idempotent)
    # ====================================================================
    table_exists = conn.execute(
        sa.text(
            "SELECT table_name FROM information_schema.tables "
            "WHERE table_schema='public' AND table_name='narrative_feedbacks'"
        )
    ).fetchone()

    if table_exists:
        logger.info("narrative_feedbacks table already exists, skipping creation")
        return

    # ====================================================================
    # STEP 2: Create table
    # ====================================================================
    logger.info("Creating narrative_feedbacks table...")

    op.create_table(
        "narrative_feedbacks",
        sa.Column("id", UUID(as_uuid=True), primary_key=True),
        sa.Column(
            "pharmacy_id",
            UUID(as_uuid=True),
            sa.ForeignKey("pharmacies.id", ondelete="CASCADE"),
            nullable=False,
        ),
        sa.Column("narrative_hash", sa.String(64), nullable=False),
        sa.Column("is_helpful", sa.Boolean, nullable=False),
        sa.Column("feedback_text", sa.Text, nullable=True),
        sa.Column("kpis_snapshot", JSONB, nullable=True),
        sa.Column("insights_snapshot", JSONB, nullable=True),
        sa.Column("narrative_content", JSONB, nullable=True),
        sa.Column("grounding_valid", sa.Boolean, nullable=True),
        sa.Column(
            "created_at",
            sa.DateTime(timezone=True),
            server_default=sa.func.now(),
            nullable=False,
        ),
    )

    # ====================================================================
    # STEP 3: Create indexes
    # ====================================================================
    logger.info("Creating indexes for narrative_feedbacks...")

    # Index for pharmacy lookup
    op.create_index(
        "ix_narrative_feedback_pharmacy",
        "narrative_feedbacks",
        ["pharmacy_id"],
    )

    # Index for narrative hash lookup
    op.create_index(
        "ix_narrative_feedback_hash",
        "narrative_feedbacks",
        ["narrative_hash"],
    )

    # Composite index for analytics: pharmacy + is_helpful
    op.create_index(
        "ix_narrative_feedback_pharmacy_helpful",
        "narrative_feedbacks",
        ["pharmacy_id", "is_helpful"],
    )

    # Index for finding recent feedbacks
    op.create_index(
        "ix_narrative_feedback_created_at",
        "narrative_feedbacks",
        ["created_at"],
    )

    logger.info("narrative_feedbacks table and indexes created successfully")


def downgrade() -> None:
    """Drop narrative_feedbacks table."""
    conn = op.get_bind()

    # Check if table exists before dropping
    table_exists = conn.execute(
        sa.text(
            "SELECT table_name FROM information_schema.tables "
            "WHERE table_schema='public' AND table_name='narrative_feedbacks'"
        )
    ).fetchone()

    if not table_exists:
        logger.info("narrative_feedbacks table does not exist, skipping drop")
        return

    # Drop indexes first
    op.drop_index("ix_narrative_feedback_created_at", table_name="narrative_feedbacks")
    op.drop_index(
        "ix_narrative_feedback_pharmacy_helpful", table_name="narrative_feedbacks"
    )
    op.drop_index("ix_narrative_feedback_hash", table_name="narrative_feedbacks")
    op.drop_index("ix_narrative_feedback_pharmacy", table_name="narrative_feedbacks")

    # Drop table
    op.drop_table("narrative_feedbacks")

    logger.info("narrative_feedbacks table dropped successfully")
