"""Create interchangeable_curated_groups table (ADR-004)

Revision ID: 20251231_01
Revises: 20251230_04
Create Date: 2025-12-31

ADR-004: Simplificación del Sistema de Clasificación
- Elimina clustering semántico dinámico
- Reemplaza por grupos curados manualmente desde Admin
- Los 8 grupos de Fase 3 se migran como seed inicial
"""
from typing import Sequence, Union

from alembic import op
import sqlalchemy as sa
from sqlalchemy.dialects import postgresql

# revision identifiers, used by Alembic.
revision: str = '20251231_01'
down_revision: Union[str, None] = '20251230_04'
branch_labels: Union[str, Sequence[str], None] = None
depends_on: Union[str, Sequence[str], None] = None


def upgrade() -> None:
    # Check if table already exists (idempotent)
    conn = op.get_bind()
    result = conn.execute(sa.text("""
        SELECT EXISTS (
            SELECT FROM information_schema.tables
            WHERE table_name = 'interchangeable_curated_groups'
        )
    """))
    if result.scalar():
        return  # Table already exists

    # Create interchangeable_curated_groups table
    op.create_table(
        'interchangeable_curated_groups',
        sa.Column('id', postgresql.UUID(as_uuid=True), primary_key=True,
                  server_default=sa.text('gen_random_uuid()')),
        sa.Column('group_name', sa.String(100), nullable=False),
        sa.Column('group_slug', sa.String(100), nullable=False),
        sa.Column('description', sa.Text(), nullable=True),
        sa.Column('necesidad_l1', sa.String(50), nullable=True,
                  comment='L1 category (necesidad) for this group'),
        sa.Column('subcategory_l2', sa.String(50), nullable=True,
                  comment='L2 subcategory if applicable'),
        sa.Column('source', sa.String(30), nullable=False,
                  server_default='manual_curated',
                  comment='legacy_clustering or manual_curated'),
        sa.Column('is_active', sa.Boolean(), nullable=False, server_default='true'),
        sa.Column('product_count', sa.Integer(), nullable=False, server_default='0'),
        sa.Column('brand_count', sa.Integer(), nullable=False, server_default='0'),
        sa.Column('total_sales_amount', sa.Numeric(12, 2), nullable=True,
                  comment='Cached total sales for products in group'),
        sa.Column('created_at', sa.DateTime(timezone=True),
                  server_default=sa.func.now(), nullable=False),
        sa.Column('updated_at', sa.DateTime(timezone=True),
                  server_default=sa.func.now(), onupdate=sa.func.now(), nullable=False),
        sa.Column('created_by', sa.String(100), nullable=True,
                  comment='User who created/curated this group'),
    )

    # Create unique constraint on slug
    op.create_unique_constraint(
        'uq_interchangeable_curated_groups_slug',
        'interchangeable_curated_groups',
        ['group_slug']
    )

    # Create index for L1 lookups
    op.create_index(
        'ix_interchangeable_curated_groups_necesidad',
        'interchangeable_curated_groups',
        ['necesidad_l1']
    )

    # Create junction table for group membership
    op.create_table(
        'interchangeable_group_members',
        sa.Column('id', postgresql.UUID(as_uuid=True), primary_key=True,
                  server_default=sa.text('gen_random_uuid()')),
        sa.Column('group_id', postgresql.UUID(as_uuid=True), nullable=False),
        sa.Column('ean13', sa.String(13), nullable=True,
                  comment='EAN13 code of product'),
        sa.Column('codigo_nacional', sa.String(20), nullable=True,
                  comment='Spanish national code'),
        sa.Column('product_name', sa.String(255), nullable=True,
                  comment='Cached product name for display'),
        sa.Column('detected_brand', sa.String(100), nullable=True),
        sa.Column('added_at', sa.DateTime(timezone=True),
                  server_default=sa.func.now(), nullable=False),
        sa.Column('added_by', sa.String(100), nullable=True),
        sa.ForeignKeyConstraint(['group_id'], ['interchangeable_curated_groups.id'],
                                ondelete='CASCADE'),
    )

    # Create indexes for member lookups
    op.create_index(
        'ix_interchangeable_group_members_group_id',
        'interchangeable_group_members',
        ['group_id']
    )
    op.create_index(
        'ix_interchangeable_group_members_ean13',
        'interchangeable_group_members',
        ['ean13']
    )
    op.create_index(
        'ix_interchangeable_group_members_codigo_nacional',
        'interchangeable_group_members',
        ['codigo_nacional']
    )

    # Seed with Phase 3 groups (8 groups from clustering analysis)
    op.execute("""
        INSERT INTO interchangeable_curated_groups
            (group_name, group_slug, description, necesidad_l1, source, created_by)
        VALUES
            ('Alimentación Bebé', 'alimentacion-bebe',
             'Leches infantiles y fórmulas de continuación intercambiables (Blemil, Pedialac, etc.)',
             'alimentacion_bebe', 'legacy_clustering', 'system_seed'),
            ('Prótesis Dental', 'protesis-dental',
             'Adhesivos y limpiadores de prótesis dentales (Corega, KIN, Lacer)',
             'protesis_dental', 'legacy_clustering', 'system_seed'),
            ('Higiene Dental', 'higiene-dental',
             'Colutorios, pastas y complementos de higiene bucal (GUM, Lacer)',
             'higiene_bucal', 'legacy_clustering', 'system_seed'),
            ('Arrugas y Antiedad', 'arrugas-antiedad',
             'Tratamientos antiedad y antiarrugas (Endocare, Esthederm)',
             'dermocosmetica', 'legacy_clustering', 'system_seed'),
            ('Encías', 'encias',
             'Productos para el cuidado de encías sensibles (Be+, Bexident)',
             'higiene_bucal', 'legacy_clustering', 'system_seed'),
            ('Probióticos', 'probioticos',
             'Probióticos y flora intestinal (Lactibiane, Bivos)',
             'digestivo', 'legacy_clustering', 'system_seed'),
            ('Protección Solar Facial', 'proteccion-solar-facial',
             'Protectores solares faciales premium (ISDIN, Heliocare)',
             'proteccion_solar', 'legacy_clustering', 'system_seed'),
            ('Vitaminas y Multivitamínicos', 'vitaminas-multivitaminicos',
             'Complejos vitamínicos y suplementos (Supradyn, Multicentrum)',
             'suplementos', 'legacy_clustering', 'system_seed')
    """)


def downgrade() -> None:
    op.drop_table('interchangeable_group_members')
    op.drop_table('interchangeable_curated_groups')
