"""Create insight_feedbacks table (Issue #506)

Issue #506: Insight Engine v2.0 - Motor de detección automática de anomalías.

Tabla para almacenar feedback de usuarios (snooze/dismiss/resolve) sobre insights.
Permite filtrar insights que ya fueron vistos o accionados.

REGLA #14: Verificación de existencia antes de crear (idempotente).

Revision ID: 20251230_04
Revises: 20251230_03
Create Date: 2025-12-30
"""

import logging

import sqlalchemy as sa
from alembic import op
from sqlalchemy.dialects.postgresql import UUID

# revision identifiers, used by Alembic.
revision = "20251230_04"
down_revision = "20251230_03"
branch_labels = None
depends_on = None

logger = logging.getLogger(__name__)


def upgrade() -> None:
    """Create insight_feedbacks table."""
    conn = op.get_bind()

    # ====================================================================
    # STEP 1: Check if table already exists (idempotent)
    # ====================================================================
    table_exists = conn.execute(
        sa.text(
            "SELECT table_name FROM information_schema.tables "
            "WHERE table_schema='public' AND table_name='insight_feedbacks'"
        )
    ).fetchone()

    if table_exists:
        logger.info("insight_feedbacks table already exists, skipping creation")
        return

    # ====================================================================
    # STEP 2: Create table
    # ====================================================================
    op.create_table(
        "insight_feedbacks",
        # Primary key
        sa.Column("id", UUID(as_uuid=True), primary_key=True),
        # Foreign key to pharmacy
        sa.Column(
            "pharmacy_id",
            UUID(as_uuid=True),
            sa.ForeignKey("pharmacies.id", ondelete="CASCADE"),
            nullable=False,
        ),
        # Insight identification
        sa.Column(
            "insight_rule_code",
            sa.String(20),
            nullable=False,
            comment="Código de la regla (ej: STOCK_001, MARGIN_002)",
        ),
        sa.Column(
            "insight_hash",
            sa.String(64),
            nullable=False,
            comment="SHA256[:16] de affected_items para deduplicación",
        ),
        # User action
        sa.Column(
            "action",
            sa.String(20),
            nullable=False,
            comment="snooze, dismiss, resolve",
        ),
        # Snooze details
        sa.Column(
            "snoozed_until",
            sa.DateTime(timezone=True),
            nullable=True,
            comment="Fecha hasta la que el insight está pospuesto",
        ),
        # User notes
        sa.Column(
            "notes",
            sa.Text(),
            nullable=True,
            comment="Notas opcionales del usuario",
        ),
        # Timestamps
        sa.Column(
            "created_at",
            sa.DateTime(timezone=True),
            server_default=sa.text("NOW()"),
            nullable=False,
        ),
    )
    logger.info("Created insight_feedbacks table")

    # ====================================================================
    # STEP 3: Create indexes
    # ====================================================================

    # Index on pharmacy_id (most common filter)
    op.create_index(
        "ix_insight_feedbacks_pharmacy_id",
        "insight_feedbacks",
        ["pharmacy_id"],
    )

    # Composite index for looking up active feedbacks by pharmacy and rule
    op.create_index(
        "ix_insight_feedback_pharmacy_rule",
        "insight_feedbacks",
        ["pharmacy_id", "insight_rule_code"],
    )

    # Index for finding specific insight by hash
    op.create_index(
        "ix_insight_feedback_hash",
        "insight_feedbacks",
        ["insight_hash"],
    )

    # Partial index for cleanup of expired snoozes (only non-null snoozed_until)
    op.execute(
        sa.text(
            """
            CREATE INDEX ix_insight_feedback_snoozed_until
            ON insight_feedbacks (snoozed_until)
            WHERE snoozed_until IS NOT NULL
            """
        )
    )

    logger.info("Created indexes for insight_feedbacks table")


def downgrade() -> None:
    """Drop insight_feedbacks table."""
    conn = op.get_bind()

    # Check if table exists before dropping
    table_exists = conn.execute(
        sa.text(
            "SELECT table_name FROM information_schema.tables "
            "WHERE table_schema='public' AND table_name='insight_feedbacks'"
        )
    ).fetchone()

    if not table_exists:
        logger.info("insight_feedbacks table does not exist, skipping drop")
        return

    # Drop indexes first (in reverse order of creation)
    op.execute(sa.text("DROP INDEX IF EXISTS ix_insight_feedback_snoozed_until"))
    op.drop_index("ix_insight_feedback_hash", table_name="insight_feedbacks")
    op.drop_index("ix_insight_feedback_pharmacy_rule", table_name="insight_feedbacks")
    op.drop_index("ix_insight_feedbacks_pharmacy_id", table_name="insight_feedbacks")

    # Drop table
    op.drop_table("insight_feedbacks")
    logger.info("Dropped insight_feedbacks table")
