"""Add L2 subcategory fields to ProductCatalogVentaLibre

Issue #505: Sub-categorías Level 2 para análisis granular

Añade campos para clasificación L2 de 18 subcategorías en 3 L1:
- DERMOCOSMETICA (7): higiene_limpieza, tratamiento_avanzado, etc.
- SUPLEMENTOS (6): descanso_estres, energia_vitalidad, etc.
- HIGIENE_BUCAL (5): higiene_diaria_basica, sensibilidad_encias, etc.

Campos añadidos:
- ml_subcategory_l2: Código de subcategoría L2
- ml_subcategory_l2_confidence: Confianza de predicción (0.0-1.0)
- subcategory_l2_source: Fuente (tier1_keywords, groq, human)

Revision ID: 20251230_03
Revises: 20251230_02
Create Date: 2025-12-30

"""
from alembic import op
import sqlalchemy as sa

# revision identifiers, used by Alembic.
revision = '20251230_03'
down_revision = '20251230_02'
branch_labels = None
depends_on = None


def upgrade() -> None:
    """Add L2 subcategory fields to product_catalog_venta_libre."""
    conn = op.get_bind()
    inspector = sa.inspect(conn)
    existing_columns = [col['name'] for col in inspector.get_columns('product_catalog_venta_libre')]

    # === 1. ml_subcategory_l2 ===
    if 'ml_subcategory_l2' not in existing_columns:
        op.add_column(
            'product_catalog_venta_libre',
            sa.Column(
                'ml_subcategory_l2',
                sa.String(100),
                nullable=True,
                comment='Subcategoría L2 (higiene_limpieza, solar_facial, etc.)'
            )
        )
        op.create_index(
            'ix_venta_libre_ml_subcategory_l2',
            'product_catalog_venta_libre',
            ['ml_subcategory_l2'],
            unique=False,
            if_not_exists=True
        )

    # === 2. ml_subcategory_l2_confidence ===
    if 'ml_subcategory_l2_confidence' not in existing_columns:
        op.add_column(
            'product_catalog_venta_libre',
            sa.Column(
                'ml_subcategory_l2_confidence',
                sa.Float(),
                nullable=True,
                server_default='0.0',
                comment='Confianza de predicción L2 (0.0-1.0)'
            )
        )

    # === 3. subcategory_l2_source ===
    if 'subcategory_l2_source' not in existing_columns:
        op.add_column(
            'product_catalog_venta_libre',
            sa.Column(
                'subcategory_l2_source',
                sa.String(50),
                nullable=True,
                comment='Fuente L2: tier1_keywords, groq, human'
            )
        )

    # === 4. Índice compuesto L1+L2 para queries jerárquicas ===
    conn.execute(sa.text("""
        CREATE INDEX IF NOT EXISTS ix_venta_libre_l1_l2_combined
        ON product_catalog_venta_libre (ml_category, ml_subcategory_l2)
        WHERE ml_subcategory_l2 IS NOT NULL
    """))

    # === 5. Índice para feedback queue L2 ===
    conn.execute(sa.text("""
        CREATE INDEX IF NOT EXISTS ix_venta_libre_l2_feedback_queue
        ON product_catalog_venta_libre (ml_subcategory_l2, subcategory_l2_source)
        WHERE ml_subcategory_l2 IS NOT NULL AND is_active = true
    """))


def downgrade() -> None:
    """Remove L2 subcategory fields from product_catalog_venta_libre."""
    conn = op.get_bind()
    inspector = sa.inspect(conn)
    existing_columns = [col['name'] for col in inspector.get_columns('product_catalog_venta_libre')]

    # Eliminar índices parciales
    conn.execute(sa.text("DROP INDEX IF EXISTS ix_venta_libre_l2_feedback_queue"))
    conn.execute(sa.text("DROP INDEX IF EXISTS ix_venta_libre_l1_l2_combined"))

    # Eliminar columnas en orden inverso
    columns_to_drop = ['subcategory_l2_source', 'ml_subcategory_l2_confidence', 'ml_subcategory_l2']

    for col in columns_to_drop:
        if col in existing_columns:
            if col == 'ml_subcategory_l2':
                op.drop_index('ix_venta_libre_ml_subcategory_l2', table_name='product_catalog_venta_libre', if_exists=True)
            op.drop_column('product_catalog_venta_libre', col)
