"""Add municipality_code and region_code to pharmacies table

Issue #504: Añadir códigos INE/ISO para ubicación de farmacias.

Campos añadidos:
- municipality_code: Código INE municipio (ej: 41091 para Sevilla)
- region_code: Código ISO 3166-2:ES (ej: ES-AN para Andalucía)

Ambos campos son nullable inicialmente para no bloquear datos existentes.
Se añaden índices para optimizar futuras consultas de festivos multinivel (#503).

Revision ID: 20251230_02
Revises: 20251230_01
Create Date: 2025-12-30

REGLA #14: Verificación de existencia antes de modificar (idempotente).
"""

import logging

import sqlalchemy as sa
from alembic import op

# revision identifiers, used by Alembic.
revision = "20251230_02"
down_revision = "20251230_01"
branch_labels = None
depends_on = None

logger = logging.getLogger(__name__)


def upgrade() -> None:
    """Add municipality_code and region_code columns to pharmacies table."""
    conn = op.get_bind()

    # ====================================================================
    # STEP 1: Check if columns already exist (idempotent - REGLA #14)
    # ====================================================================
    existing_columns = conn.execute(
        sa.text(
            """
            SELECT column_name
            FROM information_schema.columns
            WHERE table_schema = 'public'
              AND table_name = 'pharmacies'
              AND column_name IN ('municipality_code', 'region_code')
            """
        )
    ).fetchall()
    existing_column_names = {row[0] for row in existing_columns}

    # ====================================================================
    # STEP 2: Add municipality_code column if not exists
    # ====================================================================
    if "municipality_code" not in existing_column_names:
        op.add_column(
            "pharmacies",
            sa.Column("municipality_code", sa.String(10), nullable=True),
        )
        op.create_index(
            "ix_pharmacies_municipality_code",
            "pharmacies",
            ["municipality_code"],
        )
        logger.info("Added municipality_code column with index")
    else:
        logger.info("municipality_code column already exists, skipping")

    # ====================================================================
    # STEP 3: Add region_code column if not exists
    # ====================================================================
    if "region_code" not in existing_column_names:
        op.add_column(
            "pharmacies",
            sa.Column("region_code", sa.String(5), nullable=True),
        )
        op.create_index(
            "ix_pharmacies_region_code",
            "pharmacies",
            ["region_code"],
        )
        logger.info("Added region_code column with index")
    else:
        logger.info("region_code column already exists, skipping")


def downgrade() -> None:
    """Remove municipality_code and region_code columns from pharmacies table."""
    conn = op.get_bind()

    # Check which columns exist before dropping
    existing_columns = conn.execute(
        sa.text(
            """
            SELECT column_name
            FROM information_schema.columns
            WHERE table_schema = 'public'
              AND table_name = 'pharmacies'
              AND column_name IN ('municipality_code', 'region_code')
            """
        )
    ).fetchall()
    existing_column_names = {row[0] for row in existing_columns}

    # Drop region_code if exists
    if "region_code" in existing_column_names:
        op.drop_index("ix_pharmacies_region_code", table_name="pharmacies")
        op.drop_column("pharmacies", "region_code")
        logger.info("Dropped region_code column and index")

    # Drop municipality_code if exists
    if "municipality_code" in existing_column_names:
        op.drop_index("ix_pharmacies_municipality_code", table_name="pharmacies")
        op.drop_column("pharmacies", "municipality_code")
        logger.info("Dropped municipality_code column and index")
