"""Create dim_date table (Date Dimension)

Issue #503: Implementar tabla dim_date para Time Intelligence.

Centraliza toda la lógica temporal:
- Elimina EXTRACT() en runtime (pre-calculado)
- Nombres españoles (lunes, enero, etc.)
- Temporadas farmacéuticas (gripe, alergia, verano)
- Festivos nacionales españoles
- ISO-8601 para semanas (lunes como inicio)

Revision ID: 20251230_01
Revises: 152ce9c4efb1
Create Date: 2025-12-30

"""

import logging
from typing import Sequence, Union

import sqlalchemy as sa
from alembic import op

# revision identifiers, used by Alembic.
revision: str = "20251230_01"
down_revision: Union[str, None] = "152ce9c4efb1"
branch_labels: Union[str, Sequence[str], None] = None
depends_on: Union[str, Sequence[str], None] = None

logger = logging.getLogger(__name__)


def upgrade() -> None:
    """Create dim_date table with indexes."""
    conn = op.get_bind()
    inspector = sa.inspect(conn)

    # REGLA #14: Verificar existencia (idempotente)
    tables = inspector.get_table_names()
    if "dim_date" in tables:
        logger.info("SKIP: Table dim_date already exists")
        print("SKIP: Table dim_date already exists")
        return

    # Crear tabla dim_date
    op.create_table(
        "dim_date",
        # Primary key = fecha (DATE, no UUID)
        sa.Column(
            "date_key",
            sa.Date,
            primary_key=True,
            comment="Fecha (natural key para JOINs con sale_date)",
        ),
        # ===== Jerarquía temporal =====
        sa.Column(
            "year",
            sa.Integer,
            nullable=False,
            index=True,
            comment="Año (2020-2030)",
        ),
        sa.Column(
            "quarter",
            sa.Integer,
            nullable=False,
            comment="Trimestre (1-4)",
        ),
        sa.Column(
            "month",
            sa.Integer,
            nullable=False,
            index=True,
            comment="Mes (1-12)",
        ),
        sa.Column(
            "week_of_year",
            sa.Integer,
            nullable=False,
            comment="Semana ISO-8601 (1-53)",
        ),
        sa.Column(
            "day_of_month",
            sa.Integer,
            nullable=False,
            comment="Día del mes (1-31)",
        ),
        sa.Column(
            "day_of_year",
            sa.Integer,
            nullable=False,
            comment="Día del año (1-366)",
        ),
        sa.Column(
            "weekday",
            sa.Integer,
            nullable=False,
            index=True,
            comment="Día semana ISO-8601: 1=Lun, 7=Dom",
        ),
        # ===== Nombres en español =====
        sa.Column(
            "month_name",
            sa.String(20),
            nullable=False,
            comment="Nombre mes español: Enero, Febrero...",
        ),
        sa.Column(
            "month_name_short",
            sa.String(3),
            nullable=False,
            comment="Abreviatura mes: Ene, Feb...",
        ),
        sa.Column(
            "weekday_name",
            sa.String(20),
            nullable=False,
            comment="Nombre día español: Lunes, Martes...",
        ),
        sa.Column(
            "weekday_name_short",
            sa.String(3),
            nullable=False,
            comment="Abreviatura día: Lun, Mar...",
        ),
        # ===== Flags de calendario =====
        sa.Column(
            "is_weekend",
            sa.Boolean,
            nullable=False,
            default=False,
            comment="True si sábado o domingo",
        ),
        sa.Column(
            "is_holiday",
            sa.Boolean,
            nullable=False,
            default=False,
            comment="True si festivo nacional",
        ),
        sa.Column(
            "holiday_name",
            sa.String(100),
            nullable=True,
            comment="Nombre del festivo: Navidad, Año Nuevo...",
        ),
        # ===== Temporadas farmacéuticas =====
        sa.Column(
            "is_flu_season",
            sa.Boolean,
            nullable=False,
            default=False,
            comment="Temporada gripe: Oct-Mar",
        ),
        sa.Column(
            "is_allergy_season",
            sa.Boolean,
            nullable=False,
            default=False,
            comment="Temporada alergia: Mar-Jun",
        ),
        sa.Column(
            "is_summer_season",
            sa.Boolean,
            nullable=False,
            default=False,
            comment="Temporada verano: Jul-Ago",
        ),
        # ===== Claves de agrupación =====
        sa.Column(
            "year_month",
            sa.String(7),
            nullable=False,
            comment="Clave año-mes: 2025-01",
        ),
        sa.Column(
            "year_quarter",
            sa.String(7),
            nullable=False,
            comment="Clave año-trimestre: 2025-Q1",
        ),
    )
    print("  + Created table dim_date")

    # Índice compuesto para reportes mensuales (feedback usuario)
    op.execute(
        """
        CREATE INDEX IF NOT EXISTS idx_dim_date_year_month_composite
        ON dim_date (year, month)
    """
    )
    print("  + Created composite index idx_dim_date_year_month_composite")

    print("Migration 20251230_01 completed: dim_date table created")


def downgrade() -> None:
    """Drop dim_date table."""
    conn = op.get_bind()
    inspector = sa.inspect(conn)

    # Verificar si la tabla existe
    tables = inspector.get_table_names()
    if "dim_date" not in tables:
        logger.info("SKIP: Table dim_date does not exist")
        print("SKIP: Table dim_date does not exist")
        return

    # Drop indexes first
    try:
        op.execute("DROP INDEX IF EXISTS idx_dim_date_year_month_composite")
    except Exception:
        pass

    # Drop table
    op.drop_table("dim_date")
    print("Migration 20251230_01 downgrade: dim_date table dropped")
