"""Consolidate category aliases in sales_enrichment.

Migrates ml_category values from source categories to their
normalized target categories as defined in category_normalization.py.

Issue #488: Consolidate service categories
Additional: 23 other category aliases

Revision ID: 20251228_03
Revises: 20251228_01
Create Date: 2025-12-28

"""

from typing import Sequence, Union

from alembic import op
import sqlalchemy as sa

# revision identifiers, used by Alembic.
revision: str = "20251228_03"
down_revision: Union[str, None] = "20251228_01"
branch_labels: Union[str, Sequence[str], None] = None
depends_on: Union[str, Sequence[str], None] = None

# Category aliases from category_normalization.py
CATEGORY_ALIASES = {
    "aftas_llagas": "aftas",
    "heridas_cicatrizacion": "cicatrizacion",
    "mareo": "mareo_viaje",
    "vitaminas_minerales": "vitaminas_general",
    "calcio_vitamina_d": "calcio_huesos",
    "desinfeccion": "heridas_apositos",
    "dolor_fiebre": "dolor",
    "gases": "gases_flatulencia",
    "gases_digestion": "gases_flatulencia",
    "memoria": "memoria_concentracion",
    "tos_seca": "mucosidad_respiratoria",
    "tos_garganta": "mucosidad_respiratoria",
    "varices": "circulacion_piernas",
    "xerostomia": "boca_seca",
    "tinnitus": "acufenos",
    "candidiasis": "hongos_vaginales",
    "hongos_piel": "hongos_pies",
    "colageno": "articulaciones",
    "compresion_vendajes": "vendaje_compresion",
    "infeccion_urinaria": "cistitis",
    "irritacion_panal": "dermatitis_panal",
    "quemaduras": "quemaduras_aftersun",
    "ortopedia": "material_ortopedico",
    "servicio": "servicios_farmacia",
}


def upgrade() -> None:
    """Migrate source categories to target categories."""
    conn = op.get_bind()

    for source, target in CATEGORY_ALIASES.items():
        result = conn.execute(
            sa.text(
                """
                UPDATE sales_enrichment
                SET ml_category = :target
                WHERE ml_category = :source
                """
            ),
            {"source": source, "target": target},
        )
        if result.rowcount > 0:
            print(f"  Migrated {result.rowcount} records: {source} -> {target}")


def downgrade() -> None:
    """Revert target categories back to source categories.

    Note: This is a best-effort rollback. Records that were already
    in target category before migration cannot be distinguished.
    """
    conn = op.get_bind()

    # Reverse the mapping
    for source, target in CATEGORY_ALIASES.items():
        conn.execute(
            sa.text(
                """
                UPDATE sales_enrichment
                SET ml_category = :source
                WHERE ml_category = :target
                """
            ),
            {"source": source, "target": target},
        )
